#!/bin/bash
# NOTE: Empty arrays are guarded with length checks before iteration to avoid bash 3.2 'unbound variable' errors with set -u
set -euo pipefail

# Helper to run `uv sync` with project defaults.
# Respects optional environment variables:
#   UV_BIN          : path to uv executable (default: uv)
#   KW_UV_GROUPS    : comma/space separated dependency groups to include (e.g. "test,docs")
#   KW_UV_EXTRAS    : comma/space separated optional extras to install (e.g. "notebook,mlx")
#   KW_UV_NO_DEFAULT: when set to 1, skips adding the default `dev` group

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$SCRIPT_DIR"
UV_BIN="${UV_BIN:-uv}"

# Ensure uv has a writable cache location inside the workspace.
if [[ -z "${UV_CACHE_DIR:-}" ]]; then
    export UV_CACHE_DIR="${PROJECT_ROOT}/.uv-cache"
fi
mkdir -p "${UV_CACHE_DIR}"

run_post_uv() {
    local post_sync="${PROJECT_ROOT}/post-uv-sync.sh"
    if [[ -x "$post_sync" ]]; then
        "$post_sync"
    fi
}

if ! command -v "$UV_BIN" >/dev/null 2>&1; then
    cat <<EOF >&2
Error: uv not found on PATH.
Install uv via: curl -LsSf https://astral.sh/uv/install.sh | sh
EOF
    exit 1
fi

if [[ -n "${KW_OFFLINE_WHEELS:-}" ]]; then
    echo "Offline wheel bundle detected at ${KW_OFFLINE_WHEELS}; skipping uv sync and installing requirements.txt via uv pip."
    if [[ ! -d "${KW_OFFLINE_WHEELS}" ]]; then
        echo "Offline wheels directory ${KW_OFFLINE_WHEELS} not found." >&2
        exit 1
    fi
    UV_NO_INDEX=1 UV_PIP_FIND_LINKS="${KW_OFFLINE_WHEELS}" \
        "$UV_BIN" pip install --project "${PROJECT_ROOT}" --no-deps --find-links "${KW_OFFLINE_WHEELS}" \
        -r "${PROJECT_ROOT}/requirements.txt"
    pip_exit_code=$?
    
    # Run post-sync hook even in offline mode to reinstall kamiwaza wheel
    run_post_uv
    
    exit $pip_exit_code
fi

PYPROJECT_FILE="${PROJECT_ROOT}/pyproject.toml"
PYPROJECT_BACKUP=""

cleanup_and_exit() {
    local exit_code=$1
    trap - EXIT
    if [[ -n "$PYPROJECT_BACKUP" && -f "$PYPROJECT_BACKUP" ]]; then
        mv "$PYPROJECT_BACKUP" "$PYPROJECT_FILE"
    fi
    exit "$exit_code"
}

trap 'cleanup_and_exit "$?"' EXIT

if grep -qE '^\s*managed\s*=\s*false' "$PYPROJECT_FILE" 2>/dev/null; then
    # Temporarily treat the project as uv-managed for the duration of this sync.
    PYPROJECT_BACKUP="$(mktemp "${PROJECT_ROOT}/.pyproject.backup.XXXXXX")"
    cp "$PYPROJECT_FILE" "$PYPROJECT_BACKUP"
    tmp_override="$(mktemp "${PROJECT_ROOT}/.pyproject.override.XXXXXX")"
    sed 's/^\([[:space:]]*managed[[:space:]]*=[[:space:]]*\)false/\1true/' "$PYPROJECT_FILE" > "$tmp_override"
    mv "$tmp_override" "$PYPROJECT_FILE"
fi

ARGS=("$UV_BIN" --project "$PROJECT_ROOT")
ARGS+=(sync "$@")

# Dependency groups to sync (use --group flag)
groups_to_sync=()

# Extras to install via pip (use ".[extra]" syntax)
extras_to_install=()

# By default, include the 'dev' dependency group
if [[ "${KW_UV_NO_DEFAULT:-0}" != "1" ]]; then
    groups_to_sync+=("dev")
fi

# Parse KW_UV_GROUPS for additional dependency groups
env_groups="${KW_UV_GROUPS:-}"
if [[ -n "$env_groups" ]]; then
    IFS=', ' read -ra GROUP_LIST <<< "$env_groups"
    for group in "${GROUP_LIST[@]}"; do
        group_trim="${group//[[:space:]]/}"
        if [[ -n "$group_trim" ]]; then
            groups_to_sync+=("$group_trim")
        fi
    done
fi

# Parse KW_UV_EXTRAS for optional extras (notebook, mlx, etc.)
env_extras="${KW_UV_EXTRAS:-}"
if [[ -n "$env_extras" ]]; then
    IFS=', ' read -ra EXTRA_LIST <<< "$env_extras"
    for extra in "${EXTRA_LIST[@]}"; do
        extra_trim="${extra//[[:space:]]/}"
        if [[ -n "$extra_trim" ]]; then
            extras_to_install+=("$extra_trim")
        fi
    done
fi

# Deduplicate groups
deduped_groups=()
group_flags=" "
if ((${#groups_to_sync[@]})); then
    for group in "${groups_to_sync[@]}"; do
        if [[ "$group_flags" != *" $group "* ]]; then
            deduped_groups+=("$group")
            group_flags+=" $group "
        fi
    done
fi

# Deduplicate extras
deduped_extras=()
extra_flags=" "
if ((${#extras_to_install[@]})); then
    for extra in "${extras_to_install[@]}"; do
        if [[ "$extra_flags" != *" $extra "* ]]; then
            deduped_extras+=("$extra")
            extra_flags+=" $extra "
        fi
    done
fi

# Add dependency groups to sync command
if ((${#deduped_groups[@]})); then
    for group in "${deduped_groups[@]}"; do
        ARGS+=(--group "$group")
    done
fi

echo ">> ${ARGS[*]}"
if ! "${ARGS[@]}"; then
    exit $?
fi

# Install extras via pip (for things like notebook, mlx that are in [project.optional-dependencies])
if [[ ${#deduped_extras[@]} -gt 0 ]]; then
    extras_spec=$(printf '%s,' "${deduped_extras[@]}")
    extras_spec=${extras_spec%,}
    echo "Installing optional extras: ${extras_spec}"
    if ! "$UV_BIN" pip install --project "$PROJECT_ROOT" ".[${extras_spec}]"; then
        echo "Failed to install extras: ${extras_spec}" >&2
        exit 1
    fi
fi

EXPORT_REQ_SCRIPT="${PROJECT_ROOT}/scripts/export_requirements.sh"
if [[ -x "$EXPORT_REQ_SCRIPT" ]]; then
    echo "Exporting requirements via ${EXPORT_REQ_SCRIPT}"
    UV_BIN="${UV_BIN}" "$EXPORT_REQ_SCRIPT"
fi

run_post_uv
