#!/usr/bin/env python3
"""
List and remove app templates and tools from the database.

This script allows listing and removing app templates and tool deployments
from the Kamiwaza database. It can remove all items or specific ones by name.
"""

import argparse
import logging
import sys
from pathlib import Path
from typing import List, Optional

# Add the kamiwaza root to the path so we can import modules
sys.path.insert(0, str(Path(__file__).parent.parent))
sys.path.insert(0, str(Path(__file__).parent.parent / "kamiwaza"))

from kamiwaza.serving.garden.apps.templates import TemplateService
from kamiwaza.serving.garden.tool.tool_service import ToolService
from kamiwaza.db.engine import DatabaseManager
from kamiwaza.serving.config import settings

# Set up logging
log_level = logging.DEBUG if "--debug" in sys.argv else logging.INFO
logging.basicConfig(
    level=log_level, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)


class GardenItemManager:
    """Handles listing and removing app templates and tools from the database."""

    def __init__(self):
        """Initialize the manager with required services."""
        # Initialize database connection with lite mode support
        from kamiwaza.lib.util import get_kamiwaza_lite_setting, get_kamiwaza_root
        import os

        is_lite = get_kamiwaza_lite_setting()

        # Determine the correct database URL based on lite mode
        if is_lite:
            try:
                kamiwaza_root = get_kamiwaza_root()
                lite_db_dir = os.path.join(kamiwaza_root, "db-lite")
                sqlite_path = os.path.join(lite_db_dir, "kamiwaza.db")
                database_url = f"sqlite:///{sqlite_path}"
                logger.debug(
                    f"Running in KAMIWAZA_LITE mode with SQLite: {database_url}"
                )
            except Exception as e:
                logger.error(f"Failed to determine SQLite path: {e}")
                database_url = settings.database_url
        else:
            database_url = settings.database_url
            logger.debug(f"Running in full mode with CockroachDB: {database_url}")

        # Initialize database with the correct URL
        DatabaseManager.get_instance("main", database_url)

        self.template_service = TemplateService()
        self.tool_service = ToolService()

    def list_templates(
        self, format: str = "table", include_tools: bool = False
    ) -> List[dict]:
        """List all app templates."""
        try:
            templates = self.template_service.list_templates()
            logger.debug(f"Retrieved {len(templates)} templates")

            # Filter out tool templates unless explicitly requested
            if not include_tools:
                templates = [t for t in templates if not t.name.startswith("tool-")]
                logger.debug(
                    f"Filtered to {len(templates)} app templates (excluding tools)"
                )
        except Exception as e:
            logger.error(f"Error retrieving templates: {e}")
            import traceback

            traceback.print_exc()
            return []

        if format == "json":
            return [t.model_dump() for t in templates]

        # Table format
        if not templates:
            print("No app templates found")
            return []

        print(f"Found {len(templates)} app templates:")
        print("-" * 100)
        print(f"{'ID':^36} | {'Name':^30} | {'Version':^10} | {'Source':^10}")
        print("-" * 100)

        for template in templates:
            print(
                f"{str(template.id):^36} | {template.name[:30]:^30} | {template.version[:10]:^10} | {template.source_type:^10}"
            )

        return [t.model_dump() for t in templates]

    def list_tool_templates(self, format: str = "table") -> List[dict]:
        """List all tool templates from app_templates table."""
        try:
            templates = self.template_service.list_templates()
            logger.debug(f"Retrieved {len(templates)} templates")

            # Filter to only tool templates
            tool_templates = [t for t in templates if t.name.startswith("tool-")]
            logger.debug(f"Found {len(tool_templates)} tool templates")
        except Exception as e:
            logger.error(f"Error retrieving tool templates: {e}")
            import traceback

            traceback.print_exc()
            return []

        if format == "json":
            return [t.model_dump() for t in tool_templates]

        # Table format
        if not tool_templates:
            print("No tool templates found")
            return []

        print(f"Found {len(tool_templates)} tool templates:")
        print("-" * 100)
        print(f"{'ID':^36} | {'Name':^30} | {'Version':^10} | {'Source':^10}")
        print("-" * 100)

        for template in tool_templates:
            print(
                f"{str(template.id):^36} | {template.name[:30]:^30} | {template.version[:10]:^10} | {template.source_type:^10}"
            )

        return [t.model_dump() for t in tool_templates]

    def list_tools(self, format: str = "table") -> List[dict]:
        """List all tool deployments."""
        try:
            deployments = self.tool_service.list_tool_deployments()

            if format == "json":
                return [d.model_dump() for d in deployments]

            # Table format
            if not deployments:
                print("No tool deployments found")
                return []

            print(f"Found {len(deployments)} tool deployments:")
            print("-" * 120)
            print(f"{'ID':^36} | {'Name':^30} | {'Status':^15} | {'Template':^30}")
            print("-" * 120)

            for deployment in deployments:
                template_name = (
                    deployment.template_name
                    if hasattr(deployment, "template_name")
                    else "N/A"
                )
                print(
                    f"{str(deployment.id):^36} | {deployment.name[:30]:^30} | {deployment.status:^15} | {template_name[:30]:^30}"
                )

            return [d.model_dump() for d in deployments]
        except Exception as e:
            logger.error(f"Error listing tools: {e}")
            return []

    def remove_templates(
        self,
        template_names: Optional[List[str]] = None,
        force: bool = False,
        include_tools: bool = False,
    ) -> bool:
        """Remove app templates by name or all if no names specified."""
        templates = self.template_service.list_templates()

        # Filter out tool templates unless explicitly requested
        if not include_tools:
            templates = [t for t in templates if not t.name.startswith("tool-")]

        # Filter templates if names specified
        if template_names:
            templates_to_remove = [t for t in templates if t.name in template_names]
            if len(templates_to_remove) != len(template_names):
                found_names = [t.name for t in templates_to_remove]
                missing = set(template_names) - set(found_names)
                logger.warning(f"Templates not found: {', '.join(missing)}")
        else:
            templates_to_remove = templates

        if not templates_to_remove:
            logger.info("No templates to remove")
            return True

        # Confirm removal
        if not force:
            logger.info(f"About to remove {len(templates_to_remove)} templates:")
            for t in templates_to_remove:
                logger.info(f"  - {t.name} (v{t.version})")

            response = input("\nAre you sure? (yes/N): ")
            if response.lower() != "yes":
                logger.info("Removal cancelled")
                return False

        # Remove templates
        success_count = 0
        for template in templates_to_remove:
            try:
                logger.info(f"Removing template: {template.name}")
                result = self.template_service.delete_template(template.id)
                if result.get("success", False):
                    success_count += 1
                    logger.info(f"  ✓ Successfully removed {template.name}")
                else:
                    logger.error(
                        f"  ✗ Failed to remove {template.name}: {result.get('message', 'Unknown error')}"
                    )
            except Exception as e:
                logger.error(f"  ✗ Error removing {template.name}: {e}")

        logger.info(f"Removed {success_count}/{len(templates_to_remove)} templates")
        return success_count == len(templates_to_remove)

    def remove_tool_templates(
        self, template_names: Optional[List[str]] = None, force: bool = False
    ) -> bool:
        """Remove tool templates by name or all if no names specified."""
        templates = self.template_service.list_templates()

        # Filter to only tool templates
        tool_templates = [t for t in templates if t.name.startswith("tool-")]

        # Filter templates if names specified
        if template_names:
            templates_to_remove = [
                t for t in tool_templates if t.name in template_names
            ]
            if len(templates_to_remove) != len(template_names):
                found_names = [t.name for t in templates_to_remove]
                missing = set(template_names) - set(found_names)
                logger.warning(f"Tool templates not found: {', '.join(missing)}")
        else:
            templates_to_remove = tool_templates

        if not templates_to_remove:
            logger.info("No tool templates to remove")
            return True

        # Confirm removal
        if not force:
            logger.info(f"About to remove {len(templates_to_remove)} tool templates:")
            for t in templates_to_remove:
                logger.info(f"  - {t.name} (v{t.version})")

            response = input("\nAre you sure? (yes/N): ")
            if response.lower() != "yes":
                logger.info("Removal cancelled")
                return False

        # Remove templates
        success_count = 0
        for template in templates_to_remove:
            try:
                logger.info(f"Removing tool template: {template.name}")
                result = self.template_service.delete_template(template.id)
                if result.get("success", False):
                    success_count += 1
                    logger.info(f"  ✓ Successfully removed {template.name}")
                else:
                    logger.error(
                        f"  ✗ Failed to remove {template.name}: {result.get('message', 'Unknown error')}"
                    )
            except Exception as e:
                logger.error(f"  ✗ Error removing {template.name}: {e}")

        logger.info(
            f"Removed {success_count}/{len(templates_to_remove)} tool templates"
        )
        return success_count == len(templates_to_remove)

    async def remove_tools(
        self, tool_names: Optional[List[str]] = None, force: bool = False
    ) -> bool:
        """Remove tool deployments by name or all if no names specified."""
        try:
            deployments = self.tool_service.list_tool_deployments()

            # Filter tools if names specified
            if tool_names:
                tools_to_remove = [d for d in deployments if d.name in tool_names]
                if len(tools_to_remove) != len(tool_names):
                    found_names = [d.name for d in tools_to_remove]
                    missing = set(tool_names) - set(found_names)
                    logger.warning(f"Tools not found: {', '.join(missing)}")
            else:
                tools_to_remove = deployments

            if not tools_to_remove:
                logger.info("No tools to remove")
                return True

            # Confirm removal
            if not force:
                logger.info(f"About to remove {len(tools_to_remove)} tool deployments:")
                for t in tools_to_remove:
                    logger.info(f"  - {t.name} (status: {t.status})")

                response = input("\nAre you sure? (yes/N): ")
                if response.lower() != "yes":
                    logger.info("Removal cancelled")
                    return False

            # Remove tools
            success_count = 0
            for tool in tools_to_remove:
                try:
                    logger.info(f"Removing tool: {tool.name}")
                    # Stop the deployment first if it's running
                    if tool.status not in ["STOPPED", "FAILED"]:
                        logger.info(f"  Stopping deployment {tool.name}...")
                        await self.tool_service.stop_deployment(tool.id)

                    # Then delete it
                    await self.tool_service.delete_deployment(tool.id)
                    success_count += 1
                    logger.info(f"  ✓ Successfully removed {tool.name}")
                except Exception as e:
                    logger.error(f"  ✗ Error removing {tool.name}: {e}")

            logger.info(f"Removed {success_count}/{len(tools_to_remove)} tools")
            return success_count == len(tools_to_remove)
        except Exception as e:
            logger.error(f"Error removing tools: {e}")
            return False


async def main():
    """Main entry point for the script."""
    parser = argparse.ArgumentParser(
        description="List and remove app templates and tools from the database"
    )

    subparsers = parser.add_subparsers(dest="command", help="Commands")

    # List commands
    list_parser = subparsers.add_parser("list", help="List items")
    list_parser.add_argument(
        "type",
        choices=["apps", "tools", "tool-templates", "tool-deployments", "all"],
        help="Type of items to list (apps=app templates, tools=tool templates+deployments, tool-templates=just templates, tool-deployments=just deployments)",
    )
    list_parser.add_argument(
        "--format", choices=["table", "json"], default="table", help="Output format"
    )

    # Remove commands
    remove_parser = subparsers.add_parser("remove", help="Remove items")
    remove_parser.add_argument(
        "type", choices=["apps", "tools"], help="Type of items to remove"
    )
    remove_parser.add_argument(
        "names",
        nargs="*",
        help="Names of items to remove (removes all if not specified)",
    )
    remove_parser.add_argument(
        "--force", "-f", action="store_true", help="Skip confirmation prompt"
    )

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    manager = GardenItemManager()

    try:
        if args.command == "list":
            if args.type == "apps":
                manager.list_templates(args.format, include_tools=False)
            elif args.type == "tool-templates":
                manager.list_tool_templates(args.format)
            elif args.type == "tool-deployments":
                manager.list_tools(args.format)
            elif args.type == "tools":
                # Show both tool templates and deployments
                manager.list_tool_templates(args.format)
                print("\n")
                manager.list_tools(args.format)
            elif args.type == "all":
                manager.list_templates(args.format, include_tools=False)
                print("\n")
                manager.list_tool_templates(args.format)
                print("\n")
                manager.list_tools(args.format)

        elif args.command == "remove":
            if args.type == "apps":
                success = manager.remove_templates(args.names, args.force)
                sys.exit(0 if success else 1)
            elif args.type == "tools":
                success = manager.remove_tool_templates(args.names, args.force)
                sys.exit(0 if success else 1)

    except Exception as e:
        logger.error(f"Script failed with error: {e}")
        import traceback

        traceback.print_exc()
        sys.exit(1)


if __name__ == "__main__":
    import asyncio

    asyncio.run(main())
