"""
Utility library for generating a local RSA keypair for JWT (RS256).

This is used for components that still require a local public key
(e.g., Traefik JWT plugin) during the transition to full JWKS-based
validation via Keycloak.
"""
from cryptography.hazmat.primitives import serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.hazmat.backends import default_backend
import os
import stat


def generate_jwt_keys(runtime_dir: str) -> None:
    """
    Generate an RSA keypair and save to runtime_dir if missing.

    Files created:
    - jwt_private_key.pem
    - jwt_public_key.pem
    """
    os.makedirs(runtime_dir, exist_ok=True)

    private_key_path = os.path.join(runtime_dir, 'jwt_private_key.pem')
    public_key_path = os.path.join(runtime_dir, 'jwt_public_key.pem')

    if os.path.exists(private_key_path) and os.path.exists(public_key_path):
        return

    private_key = rsa.generate_private_key(
        public_exponent=65537,
        key_size=2048,
        backend=default_backend()
    )

    public_key = private_key.public_key()

    pem_private = private_key.private_bytes(
        encoding=serialization.Encoding.PEM,
        format=serialization.PrivateFormat.PKCS8,
        encryption_algorithm=serialization.NoEncryption(),
    )

    pem_public = public_key.public_bytes(
        encoding=serialization.Encoding.PEM,
        format=serialization.PublicFormat.SubjectPublicKeyInfo,
    )

    old_umask = os.umask(0o077)
    try:
        with open(private_key_path, 'wb') as f:
            f.write(pem_private)
        os.chmod(private_key_path, stat.S_IRUSR | stat.S_IWUSR)

        with open(public_key_path, 'wb') as f:
            f.write(pem_public)
        os.chmod(
            public_key_path,
            stat.S_IRUSR | stat.S_IWUSR | stat.S_IRGRP | stat.S_IROTH,
        )
    finally:
        os.umask(old_umask)
