#!/usr/bin/env python3
"""
Container validation script for Kamiwaza
"""
import sys
import json
import os

def main():
    try:
        # Read container data from stdin
        containers = json.load(sys.stdin)
        env = os.getenv('KAMIWAZA_ENV', 'default')
        lite = os.getenv('KAMIWAZA_LITE', 'false').lower() == 'true'
        use_auth = os.getenv('KAMIWAZA_USE_AUTH', 'false').lower() == 'true'

        # Filter for kamiwaza containers
        kamiwaza_containers = [c for c in containers if c['Name'].startswith(f'{env}-kamiwaza-')]
        kamiwaza_containers = [c for c in kamiwaza_containers if 'otel-collector' not in c['Name']]

        # Determine expected containers based on mode and feature flags
        expected = {
            f'{env}-kamiwaza-etcd': {'total': 2, 'running': 1},
            f'{env}-kamiwaza-traefik': {'total': 1, 'running': 1},
        }

        # In non-lite mode, include database and DataHub
        if not lite:
            expected[f'{env}-kamiwaza-cockroachdb'] = {'total': 1, 'running': 1}
            expected[f'{env}-kamiwaza-datahub'] = {'total': 10, 'running': 6}

        # Keycloak is only required when authentication is enabled to align
        # with containers-up.sh, which skips deploying Keycloak when
        # KAMIWAZA_USE_AUTH is not true.
        if use_auth:
            expected[f'{env}-kamiwaza-keycloak'] = {'total': 2, 'running': 2}

        # Optional components controlled by feature flags
        if os.getenv('KAMIWAZA_MILVUS_ENABLED', 'true').lower() == 'true':
            expected[f'{env}-kamiwaza-milvus'] = {'total': 3, 'running': 3}
        loki_enabled = os.getenv('KAMIWAZA_LOKI_ENABLED', 'false').lower() == 'true'
        if loki_enabled:
            expected[f'{env}-kamiwaza-loki'] = {'total': 2, 'running': 2}

        # Validate each container
        ready_count = 0
        starting_count = 0
        error_count = 0
        
        for container in kamiwaza_containers:
            name = container['Name']
            status = container['Status']
            
            if name not in expected:
                continue
                
            # Parse container states
            states = {'running': 0, 'exited': 0, 'starting': 0}
            for state in ['running', 'exited', 'starting']:
                if f'{state}(' in status:
                    try:
                        count = int(status.split(f'{state}(')[1].split(')')[0])
                        states[state] = count
                    except (ValueError, IndexError):
                        states[state] = 0
            
            # Check if container meets expectations
            expected_running = expected[name]['running']
            actual_running = states['running']
            actual_starting = states['starting']
            
            if actual_running >= expected_running:
                ready_count += 1
            elif actual_running + actual_starting >= expected_running:
                starting_count += 1
            else:
                error_count += 1
                print(f"ERROR: {name} has {actual_running} running (expected {expected_running})", file=sys.stderr)
        
        total_expected = len(expected)
        
        # Determine result
        if ready_count == total_expected:
            print("All containers running", file=sys.stderr)
            sys.exit(0)
        elif starting_count > 0 and error_count == 0:
            print(f"Containers starting: {ready_count}/{total_expected} ready", file=sys.stderr)
            sys.exit(2)
        else:
            print(f"Container errors: {error_count} failed, {starting_count} starting, {ready_count} ready", file=sys.stderr)
            sys.exit(1)
            
    except Exception as e:
        print(f"Validation script error: {e}", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
