import sys
import os
import uuid
from kamiwaza.logger.ring_buffer import RingBuffer
from kamiwaza.logger.logger import KamiwazaLogger
from kamiwaza.licenses.manager.manager import LicenseManager
import logging
from urllib.parse import urlparse
from kamiwaza.lib.util import get_kamiwaza_root

"""
This module initializes and configures the global KamiwazaLogger instance,
which is used for telemetry and event tracking throughout the application.
It handles versioning, license information, session management, and user
preferences for telemetry.
"""

logging.basicConfig(level=logging.WARNING)
logger = logging.getLogger(__name__)

def get_version_from_env(env_file: str = "kamiwaza.version.json") -> str:
    """
    Attempts to read the version from kamiwaza root first, then from the local directory.
    Exits the program if both attempts fail or the version is invalid.
    """
    import json

    def try_read_version(path: str):
        try:
            with open(path, "r", encoding="utf-8") as f:
                version_data = json.load(f)
                return (
                    version_data.get("KAMIWAZA_VERSION_MAJOR"),
                    version_data.get("KAMIWAZA_VERSION_MINOR"),
                    version_data.get("KAMIWAZA_VERSION_PATCH"),
                )
        except FileNotFoundError as e:
            logger.warning(f"{path} not found: {e}")
            return None, None, None
        except json.JSONDecodeError as e:
            logger.error(f"Invalid JSON format in {path}: {e}")
            return None, None, None
        except Exception as e:
            logger.error(f"Unexpected error reading {path}: {e}", exc_info=True)
            return None, None, None

    # First try in the kamiwaza root directory
    kamiwaza_root = get_kamiwaza_root()
    root_env_file = os.path.join(kamiwaza_root, env_file)
    major, minor, patch = try_read_version(root_env_file)

    # If not found, try the local directory
    if not all([major, minor, patch]):
        major, minor, patch = try_read_version(env_file)

    version = f"{major}.{minor}.{patch}" if all([major, minor, patch]) else ""
    if not version or version == "..":
        logger.error("Could not construct version from kamiwaza.version.json in kamiwaza root or local directory")
        sys.exit(1)
    return version

def get_license_info():
    license_dir = 'licenses'
    if not os.path.exists(license_dir):
        os.makedirs(license_dir)

    try:
        lm = LicenseManager(license_dir)
        entitlements = list(lm.get_active_features().keys())
        # Get all license texts from .lic files
        license_texts = []
        for filename in os.listdir(license_dir):
            if filename.endswith('.lic'):
                try:
                    with open(os.path.join(license_dir, filename), 'r') as f:
                        license_texts.append(f.read().strip())
                except Exception as e:
                    print(f"ERROR: Failed to read license file {filename}: {e}")
                    continue
        
        license_info = {
            'entitlements': entitlements,
            'license_texts': license_texts
        }
    except Exception as e:
        logger.warning(f"Failed to get license info: {e}")

        license_info = {
            'entitlements': [],
            'license_texts': []
        }

    return license_info

def _parse_debconf_config(path: str = "/var/cache/debconf/config.dat") -> dict:
    """Parse the debconf database and return a mapping of Name ➜ Value."""

    data: dict[str, str] = {}
    stanza: dict[str, str] = {}

    try:
        with open(path, "r", encoding="utf-8") as f:
            for raw_line in f:
                line = raw_line.rstrip("\n")

                # Blank line marks the end of a stanza
                if line == "":
                    if "Name" in stanza and "Value" in stanza:
                        data[stanza["Name"]] = stanza["Value"]
                    stanza.clear()
                    continue

                if ": " in line:
                    key, value = line.split(": ", 1)
                    stanza[key] = value

        # Flush the last stanza (file may not end with a blank line)
        if "Name" in stanza and "Value" in stanza:
            data[stanza["Name"]] = stanza["Value"]

    except Exception as exc:
        logger.warning("Could not read debconf config: %s", exc)

    return data


def get_telemetry_opt_out() -> bool:
    """Return True if the user has opted *out* of telemetry reporting.

    The debconf value kamiwaza/usage_reporting is **true** when the user has
    agreed to send usage telemetry. Therefore the user has *not* opted out in
    that case.  Any other value means telemetry should be disabled.
    """

    if not os.path.exists("/var/cache/debconf/config.dat"):
        return False
    
    config = _parse_debconf_config()
    usage_reporting = config.get("kamiwaza/usage_reporting", "false").lower()

    # Opt-out is the inverse of "usage_reporting is true"
    return str(usage_reporting).lower() != "true"

def get_contact_info() -> dict:
    """Return a dict containing the user's contact information."""

    if not os.path.exists("/var/cache/debconf/config.dat"):
        return {"name": "", "email": ""}

    config = _parse_debconf_config()
    email = config.get("kamiwaza/user_email", "")

    return {"name": "", "email": email or ""}

def validate_analytics_url(url: str) -> bool:
    """Validate if the provided analytics URL is properly formatted."""
    try:
        parsed = urlparse(url)
        return bool(parsed.scheme and parsed.netloc and parsed.scheme in ['http', 'https'])
    except Exception as e:
        logger.warning(f"Failed to parse analytics URL '{url}': {e}")
        return False

version = get_version_from_env()

# -------------------------------------------------------------
# Global Session ID Handling
# -------------------------------------------------------------
# We want every process in a running Kamiwaza installation to use the
# *same* session_id when emitting telemetry events.  To achieve this we
# do the following:
#   1.  Check if the special environment variable ``KAMIWAZA_SESSION_ID``
#       is already set.
#   2.  If it is set, reuse the value verbatim.
#   3.  If it is *not* set, generate a brand-new UUID4 and place it in
#       the environment so that all child processes inherit it.
#
# The value is then passed into ``KamiwazaLogger`` via its ``session_id``
# parameter.  Subsequent imports of this module within the *same* Python
# process will, of course, see the already-initialised logger instance
# and therefore share the same session id.  Imports performed in *other*
# processes will see the environment variable and also adopt the exact
# same id, ensuring uniformity across the entire installation.
# -------------------------------------------------------------

ENV_SESSION_KEY = "KAMIWAZA_SESSION_ID"

session_id = os.environ.get(ENV_SESSION_KEY)
if not session_id:
    session_id = str(uuid.uuid4())
    # Make it available to this process and all children.
    os.environ[ENV_SESSION_KEY] = session_id

license_info = get_license_info()

# 1. Initialize the Event Queue
# Customize max_size and db_path as per your application's needs.
# Using a db_path enables disk persistence for the queue.
event_queue = RingBuffer(max_size=1000, db_path="telemetry_queue.db")

# 2. Initialize the KamiwazaLogger instance
# Replace placeholder values with your actual product and license details.
# Ensure the upload_url points to your telemetry API endpoint.
license_type = ', '.join(license_info['entitlements']) or "Unknown"
# Pass the raw license key to KamiwazaLogger (it will handle hashing internally)
license_texts = license_info['license_texts']
if license_texts:
    # Filter out empty strings and join
    non_empty_texts = [text for text in license_texts if text.strip()]
    license_key = ','.join(non_empty_texts) if non_empty_texts else "Unknown"
else:
    license_key = "Unknown"

# Set telemetry opt out
telemetry_opt_out_override = os.getenv("TELEMETRY_OPT_OUT_OVERRIDE")
if telemetry_opt_out_override == "true":    # For developer installations
    telemetry_opt_out = True
elif telemetry_opt_out_override == "false":
    telemetry_opt_out = False
else:
    telemetry_opt_out = get_telemetry_opt_out()

logger.info(f"telemetry_opt_out: {telemetry_opt_out}")

# Set the upload_url with validation
upload_url = os.getenv('KAMIWAZA_ANALYTICS_URL', 'https://analytics.kamiwaza.ai/v1/events')

# Validate the analytics URL
if not validate_analytics_url(upload_url):
    logger.warning(f"Invalid analytics URL provided: '{upload_url}'. Using default URL.")
    upload_url = 'https://analytics.kamiwaza.ai/v1/events'
    # If the default URL is also invalid, disable telemetry
    if not validate_analytics_url(upload_url):
        logger.error("Default analytics URL is also invalid. Disabling telemetry.")
        telemetry_opt_out = True
        upload_url = 'https://analytics.kamiwaza.ai/v1/events'  # Use as placeholder

logger.info(f"Analytics upload_url: {upload_url}")

# Instantiate the global logger with error handling
kamiwaza_logger_instance = None
try:
    kamiwaza_logger_instance = KamiwazaLogger(
        queue=event_queue,
        product_name="Kamiwaza",
        product_version=version,
        license_type=license_type,
        license_key=license_key,
        session_id=session_id,
        contact_info=get_contact_info(),
        telemetry_opt_out=telemetry_opt_out,
        upload_url=upload_url,  # Telemetry API
        # enable_local_logging=True, # Default is True, set to False to disable
    )
    logger.info("KamiwazaLogger initialized successfully")
except Exception as e:
    logger.error(f"Failed to initialize KamiwazaLogger: {e}")
    # Create a minimal fallback logger that disables telemetry
    try:
        kamiwaza_logger_instance = KamiwazaLogger(
            queue=event_queue,
            product_name="Kamiwaza",
            product_version=version,
            license_type=license_type,
            license_key=license_key,
            session_id=session_id,
            contact_info=get_contact_info(),
            telemetry_opt_out=True,  # Force disable telemetry
            # enable_local_logging=True, # Default is True, set to False to disable
        )
        logger.warning("Created fallback KamiwazaLogger with telemetry disabled")
    except Exception as fallback_error:
        logger.critical(f"Failed to create fallback logger: {fallback_error}")
        kamiwaza_logger_instance = None

# 3. Dependency function to get the logger instance
# FastAPI will call this function to provide the logger to your endpoints.
def get_logger() -> KamiwazaLogger | None:
    return kamiwaza_logger_instance