#!/bin/bash

set -e

# Capture the script path as invoked by the user
SCRIPT_PATH="$0"

STARTUP_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
# shellcheck disable=SC1091
source "${STARTUP_DIR}/../set-kamiwaza-root.sh"
# shellcheck disable=SC1091
source "${STARTUP_DIR}/../common-python.sh"

# Check if script is being run as root/sudo
if [ "$EUID" -eq 0 ]; then
    echo "This script should be run as the installed user, not with sudo or by root."
    exit 1
fi

# Detect operating system
IS_MACOS=false
if [[ "$OSTYPE" == "darwin"* ]]; then
    IS_MACOS=true
fi

# Place this at the very top (after the shebang) of any launcher script.
ensure_nofile_min() {
  local min=${1:-65536}
  local cur hard target

  cur=$(ulimit -Sn 2>/dev/null) || { echo "warn: ulimit -Sn failed" >&2; return 0; }
  hard=$(ulimit -Hn 2>/dev/null) || { echo "warn: ulimit -Hn failed" >&2; return 0; }

  # Already unlimited? Nothing to do.
  if [[ "$cur" == "unlimited" ]]; then
    echo "[nofile] soft=unlimited hard=$hard"
    return 0
  fi

  # Normalize non-integers
  case $cur  in (''|*[!0-9]*)  cur=0 ;; esac            # // non-integer soft -> treat as 0
  if [[ "$hard" == "unlimited" ]]; then
    target=$min                                          # // hard is effectively infinite
  else
    case $hard in (''|*[!0-9]*) hard=$min ;; esac        # // non-integer hard -> assume min threshold
    target=$(( hard < min ? hard : min ))                # // never exceed hard
  fi

  if (( cur < target )); then
    ulimit -Sn "$target" || echo "warn: could not set soft nofile to $target" >&2
  fi
}

ensure_nofile_min 65536 || echo "warn: could not set soft nofile to 65536" >&2

# Shared color helpers
cecho() {
    local color="$1"; shift
    if [ -t 1 ]; then
        tput setaf "$color" 2>/dev/null || true
        echo "$*"
        tput sgr0 2>/dev/null || true
    else
        echo "$*"
    fi
}

# Quiet/json-aware variant (used by status)
cecho_quiet() {
    local color="$1"; shift
    if [ "${quiet:-false}" = false ] && [ "${json:-false}" = false ]; then
        cecho "$color" "$@"
    elif [ "${quiet:-false}" = false ]; then
        echo "$*"
    fi
}


# Store the original command for dev-friendly output
ORIGINAL_COMMAND="$0"
if [ "$#" -gt 0 ]; then
    ORIGINAL_COMMAND="$0 $*"
fi

if [ -f ${KAMIWAZA_ROOT:-..}/.kamiwaza_install_community ]; then
    export KAMIWAZA_COMMUNITY=true
fi


# Source configurations
if [ -f /etc/kamiwaza/env.sh ]; then
    source /etc/kamiwaza/env.sh
    export KAMIWAZA_COMMUNITY=false
elif [ -f "${KAMIWAZA_ROOT:-$STARTUP_DIR/..}/env.sh" ]; then
    source "${KAMIWAZA_ROOT:-$STARTUP_DIR/..}/env.sh"
elif [ ! -f "${KAMIWAZA_ROOT:-$STARTUP_DIR/..}/.kamiwaza_install_community" ]; then
    echo " ###### WARNING ######: /etc/kamiwaza/env.sh does not exist. Contact support@kamiwaza.ai"
fi

if [[ -z "${KW_PY_NO_SYNC:-}" ]]; then
    export KW_PY_NO_SYNC=1
fi

# Locations
LOGROTATE_CONF="/etc/logrotate.d/kamiwaza"

# Determine log directory using Python utility for consistency with application logic
# Priority: KAMIWAZA_LOG_DIR → $KAMIWAZA_ROOT/logs
# Note: stderr flows to terminal (for uv sync messages, warnings, etc.) while stdout is captured
# Create a separate file descriptor for stderr that points to terminal
exec 3>&2
if ! LOG_DIR=$(kw_py -c "from kamiwaza.lib.logging.utils import get_log_directory; print(get_log_directory())" 2>&3); then
    echo "ERROR: Failed to determine log directory" >&2
    exit 1
fi
# Strip any trailing whitespace/newlines from the path
LOG_DIR=$(echo "$LOG_DIR" | tr -d '\n\r')

# Don't export KAMIWAZA_LOG_DIR - let Python determine it independently
# Exporting it would cause Python to log it as an "override" even though we're just using the default

# Set container startup defaults if not already set
export CONTAINER_STARTUP_TIMEOUT="${CONTAINER_STARTUP_TIMEOUT:-300}"
export CONTAINER_GRACE_PERIOD="${CONTAINER_GRACE_PERIOD:-60}"

# Immediately rotate logs for a clean start
rotate_logs() {
    # Rotate logs in each configured directory
    for log_dir in "/opt/kamiwaza/logs" "/var/log" "/tmp"; do
        # Skip if directory doesn't exist
        [ ! -d "$log_dir" ] && continue
        
        # Rotate each log file up to 3 times before startup
        for logfile in kamiwazad-azure_disk_mount.log kamiwazad-containers.log kamiwazad-core.log kamiwazad-frontend.log kamiwazad-jupyter.log kamiwazad-ray.log kamiwazad.log; do
            if [ -f "${log_dir}/${logfile}" ]; then
                for i in $(seq 3 -1 1); do
                    if [ -f "${log_dir}/${logfile}.${i}" ]; then
                        mv "${log_dir}/${logfile}.${i}" "${log_dir}/${logfile}.$((i+1))"
                    fi
                done
                mv "${log_dir}/${logfile}" "${log_dir}/${logfile}.1"
            fi
        done
    done
}

# Worker detection logic
if [ -f "/etc/kamiwaza/config/is_worker" ]; then
    export KAMIWAZAD_IS_WORKER=1
    echo "INFO: Running as worker node (config file present)"
elif [ -n "${KAMIWAZA_HEAD_IP:-}" ]; then
    # Check if KAMIWAZA_HEAD_IP matches any local IP
    if ifconfig | grep inet | awk '{print $2}' | grep -q "^${KAMIWAZA_HEAD_IP}$" > /dev/null 2>&1; then
        export KAMIWAZAD_IS_WORKER=0
    else
        echo "ERROR: KAMIWAZA_HEAD_IP (${KAMIWAZA_HEAD_IP}) does not match any local interface" >&2
        echo "Available IPs on this machine:" >&2
        ifconfig | grep inet | awk '{print "  " $2}' | grep -v "127.0.0.1" >&2
        echo "" >&2
        echo "Please update KAMIWAZA_HEAD_IP in your configuration to match one of the above IPs," >&2
        echo "or unset it to run in head node mode." >&2
        exit 1
    fi
else
    export KAMIWAZAD_IS_WORKER=0
fi

if [ "${KAMIWAZAD_IS_WORKER:-999}" -eq 0 ]; then
    touch /tmp/kamiwazad.starting        
fi

# Ensure Python dependencies (best-effort, do not fail startup)
if command -v uv >/dev/null 2>&1; then
    uv pip install --project "${KAMIWAZA_ROOT:-$STARTUP_DIR/..}" -q python-dotenv psutil pyyaml || \
        echo "Warning: uv failed to install runtime dependencies (continuing)" >&2
else
    if command -v python >/dev/null 2>&1; then
        python -m pip install -q --user python-dotenv psutil pyyaml || \
            echo "Warning: pip failed to install runtime dependencies (continuing)" >&2
    else
        echo "Warning: Python not found; cannot ensure runtime dependencies (continuing)" >&2
    fi
fi


# Define paths
PYTHON_INTERPRETER_COMMAND=()
PYTHON_INTERPRETER_STRING=""

if command -v uv >/dev/null 2>&1; then
    PYTHON_INTERPRETER_COMMAND=(uv --project "${KAMIWAZA_ROOT:-$STARTUP_DIR/..}" run --frozen python)
    PYTHON_INTERPRETER_STRING="uv --project \"${KAMIWAZA_ROOT:-$STARTUP_DIR/..}\" run --frozen python"
else
    PYTHON_INTERPRETER_COMMAND=(python)
    PYTHON_INTERPRETER_STRING="python"
fi

run_kamiwazad() {
    if [ ${#PYTHON_INTERPRETER_COMMAND[@]} -eq 0 ]; then
        echo "Python interpreter command is not configured." >&2
        return 127
    fi
    "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" "$@"
}

# Helper function to run Python CLI tools (like ray) through uv when available
run_py_cli() {
    local tool="$1"
    shift

    if command -v uv >/dev/null 2>&1; then
        # Use uv to run the tool from the project environment
        uv --project "${KAMIWAZA_ROOT:-$STARTUP_DIR/..}" run --frozen -- "$tool" "$@"
    elif command -v "$tool" >/dev/null 2>&1; then
        # Fallback to direct command if uv not available
        "$tool" "$@"
    else
        echo "Error: $tool command not found. Ensure uv dependencies are installed (run 'uv sync')." >&2
        return 127
    fi
}

KAMIWAZA_SCRIPT="${KAMIWAZA_ROOT:-}/startup/kamiwazad.py"
PID_FILE="/tmp/kamiwazad.pid"
LOG_FILE="$LOG_DIR/kamiwazad.log"

if [ -z "${KAMIWAZA_ROOT:-}" ]; then
    echo "KAMIWAZA_ROOT is not set. Repair your install or contact support@kamiwaza.ai"
    exit 1
fi

cd "${KAMIWAZA_ROOT}"

###############################################################################
# Developer sentinel helpers
# Use the presence of `${KAMIWAZA_ROOT}/kamiwaza-shibboleth` to gate
# developer-only commands surfaced to engineers but hidden from end users.
###############################################################################

is_dev_env() {
    [ -f "${KAMIWAZA_ROOT}/kamiwaza-shibboleth" ]
}

# Helper function to show dev-friendly command
show_command() {
    local color="$1"
    local message="$2"
    if [ -t 1 ]; then
        tput setaf "$color" 2>/dev/null || true
        echo "$message"
        tput sgr0 2>/dev/null || true
    else
        echo "$message"
    fi
}

setup_logging() {
    # Create log directory if it doesn't exist
    if [ ! -d "$LOG_DIR" ]; then
        if ! mkdir -p "$LOG_DIR" 2>/dev/null; then
            if [ "$IS_MACOS" = true ]; then
                sudo mkdir -p "$LOG_DIR" 2>/dev/null || LOG_DIR="/tmp"
            else
                sudo -n mkdir -p "$LOG_DIR" 2>/dev/null || LOG_DIR="/tmp"
            fi
        fi

        # If sudo mkdir worked, set permissions
        if [ -d "$LOG_DIR" ] && [ "$LOG_DIR" != "/tmp" ]; then
            if [ "$IS_MACOS" = true ]; then
                sudo chown "$USER:$(id -g)" "$LOG_DIR"
                sudo chmod 755 "$LOG_DIR"
            else
                sudo -n chown "$USER:$(id -g)" "$LOG_DIR"
                sudo -n chmod 755 "$LOG_DIR"
            fi
        fi
    fi

    # Touch log file and set permissions if it doesn't exist
    if [ ! -f "$LOG_FILE" ]; then
        if ! touch "$LOG_FILE" 2>/dev/null; then
            if [ "$IS_MACOS" = true ]; then
                sudo touch "$LOG_FILE"
            else
                sudo -n touch "$LOG_FILE"
            fi
        fi
        if [ -f "$LOG_FILE" ]; then
            if [ "$IS_MACOS" = true ]; then
                sudo chown "$USER:$(id -g)" "$LOG_FILE"
                sudo chmod 644 "$LOG_FILE"
            else
                sudo -n chown "$USER:$(id -g)" "$LOG_FILE"
                sudo -n chmod 644 "$LOG_FILE"
            fi
        fi
    fi
}

start() {

    local missing_only=false
    if [ "${1:-}" = "--missing" ]; then
        missing_only=true
    fi

    # If Docker is NOT available, kill docker sudo dockerd & run dockerd as kamiwaza user
    if ! docker version >/dev/null 2>&1; then
        echo "Restarting docker daemon..."
        # killall dockerd || true
        sudo dockerd > /dev/null 2>&1 &
    fi

    # WSL detection
    IS_WSL=false
    if grep -qi microsoft /proc/version 2>/dev/null; then
        IS_WSL=true
    fi

    
    if [ -f $PID_FILE ]; then
        PID=$(cat $PID_FILE)
        if kill -0 $PID 2>/dev/null; then
            if [ "$missing_only" = true ]; then
                echo "Checking for missing services..."
                run_kamiwazad start --missing
                return
            else
                echo "kamiwazad is already running (PID: $PID)"
                return
            fi
        else
            echo "Stale PID file found. Removing and starting kamiwazad..."
            rm $PID_FILE
        fi
    fi

    # Setup logging infrastructure
    setup_logging
    
    # Rotate logs before starting
    rotate_logs


    # Process cloud-init files if present
    if [ -f "/tmp/kamiwaza_public_key" ] && [ -f "/tmp/kamiwaza_known_host" ]; then
        echo "Ingesting Kamiwaza keys..."
        
        if ! grep -qf "/tmp/kamiwaza_known_host" "${CUSTOMER_KNOWN_HOSTS:-}"; then
            cat "/tmp/kamiwaza_known_host" >> "${CUSTOMER_KNOWN_HOSTS:-}"
            echo "Added known hosts entry from cloud-init"
        fi

        if [ ! -f "${CUSTOMER_PRIVATE_KEY:-}" ]; then
            cp "/tmp/kamiwaza_public_key" "${CUSTOMER_PRIVATE_KEY:-}"
            chmod 600 "${CUSTOMER_PRIVATE_KEY:-}"
            echo "Installed customer private key from cloud-init"
        fi

        rm -f "/tmp/kamiwaza_public_key" "/tmp/kamiwaza_known_host"
        echo "Removed temporary cloud-init files"
    fi
    
    echo
    cecho 6 "═══════════════════════════════════════════════════════════════"
    cecho 6 "  STARTING KAMIWAZA SERVICES"
    cecho 6 "═══════════════════════════════════════════════════════════════"
    echo
    
    if [ "$missing_only" = true ]; then
        cecho 3 "  ⚡ Mode: Starting missing services only"
    else  
        cecho 2 "  🚀 Mode: Full startup sequence"
    fi
    echo
    
    cecho 7 "  📝 Logs: $LOG_FILE"
    local env_mode="Full"
    if [ "${KAMIWAZA_LITE:-}" = "true" ]; then
        env_mode="Lite"
    fi
    cecho 7 "  🔧 Environment: $env_mode mode"
    echo

    cecho 3 "  ⏳ Initializing daemon..."
    nohup "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" start ${1:-} 1>"$LOG_FILE" 2>&1 &
    PID=$!
    echo $PID > $PID_FILE
    
    # Enhanced startup verification with progress
    cecho 7 "     Checking daemon startup..."
    sleep 2
    
    if kill -0 $PID 2>/dev/null; then
        cecho 2 "  ✓ Daemon started successfully (PID: $PID)"
        echo
        
        # Show startup progress monitoring
        cecho 7 "  📊 Monitor startup progress with:"
        cecho 7 "     $0 status -w"
        if [ "$SCRIPT_PATH" != "$0" ]; then
            cecho 7 "     $SCRIPT_PATH status -w"
        fi
        echo
        cecho 7 "  📋 Check current status with:"
        cecho 7 "     $0 status"
        if [ "$SCRIPT_PATH" != "$0" ]; then
            cecho 7 "     $SCRIPT_PATH status"
        fi
        echo
        
        cecho 6 "═══════════════════════════════════════════════════════════════"
        cecho 2 "  STARTUP INITIATED SUCCESSFULLY"
        cecho 6 "═══════════════════════════════════════════════════════════════"
        echo
    else
        cecho 1 "  ✗ Failed to start kamiwazad daemon"
        echo
        cecho 3 "  💡 Troubleshooting tips:"
        cecho 7 "     • Check logs: tail -f $LOG_FILE"
        cecho 7 "     • Run diagnostics: $0 doctor"
        cecho 7 "     • Run diagnostics: $SCRIPT_PATH doctor"
        cecho 7 "     • Check permissions and dependencies"
        echo
        rm -f $PID_FILE
        exit 1
    fi
}

stop() {
    local force_stop=false
    local daemon_was_running=false
    
    if [ -f $PID_FILE ]; then
        PID=$(cat $PID_FILE)
        if kill -0 $PID 2>/dev/null; then
            daemon_was_running=true
            cecho 3 "⏹  Stopping kamiwazad daemon (PID: $PID)..."
            echo
        else
            cecho 3 "⚠  Daemon not running (stale PID file found)"
            rm $PID_FILE
            force_stop=true
        fi
    else
        cecho 3 "⚠  Daemon not running"
        force_stop=true
    fi

    # Stop services (the Python script will show detailed progress)
    if [ "$daemon_was_running" = true ]; then
        # Daemon is running, stop it and let it handle service shutdown
        "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" stop
        kill $PID 2>/dev/null || true
        rm -f $PID_FILE
        echo
        cecho 2 "✓ Shutdown completed"
    elif [ "$force_stop" = true ]; then
        # No daemon running, try direct service shutdown
        cecho 7 "○ Checking for running services to stop..."
        echo
        "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" stop
        echo
        cecho 2 "✓ Direct shutdown completed"
    fi

    echo
}

status() {
    local check_all=false json=false quiet=false watch=false logs=false timeout=180
    local auto_fallback=true
    
    # Parse flags with getopts
    OPTIND=1  # Reset getopts index
    while getopts ":aqjwlt:h" opt; do
        case $opt in
            a) check_all=true ;;
            q) quiet=true ;;
            j) json=true ;;
            w) watch=true ;;
            l) logs=true ;;
            t) timeout="$OPTARG" ;;
            h) echo "Usage: $0 status [-a] [-q] [-j] [-w] [-l] [-t timeout]"
               echo "  -a    Check all services even if daemon is down"
               echo "  -q    Quiet mode (minimal output)"
               echo "  -j    JSON output format" 
               echo "  -w    Watch mode for startup monitoring"
               echo "  -l    Show recent logs after status"
               echo "  -t N  Timeout for watch mode (default: 180s)"
               return 0 ;;
            \?) echo "Invalid option: -$OPTARG. Use -h for help." >&2; return 3 ;;
            :) echo "Option -$OPTARG requires an argument." >&2; return 3 ;;
        esac
    done
    
    # Validate Python components first
    if [ ! -f "$KAMIWAZA_SCRIPT" ] || ! command -v "${PYTHON_INTERPRETER_COMMAND[0]}" >/dev/null 2>&1; then
        echo "kamiwazad internal error: Python components missing." >&2
        return 3
    fi
    
    local daemon_running=false
    local daemon_pid=""
    local exit_code=0
    
    # Check daemon status
    if [ -f "$PID_FILE" ]; then
        daemon_pid=$(cat "$PID_FILE")
        if kill -0 "$daemon_pid" 2>/dev/null; then
            daemon_running=true
            [ "$quiet" = false ] && cecho_quiet 2 "kamiwazad is running (PID: $daemon_pid)"
        else
            [ "$quiet" = false ] && cecho_quiet 3 "kamiwazad is NOT running (stale PID file found)"
            rm -f "$PID_FILE"
            exit_code=1
        fi
    else
        [ "$quiet" = false ] && cecho_quiet 3 "kamiwazad is NOT running"
        exit_code=1
    fi
    
    # Watch mode for container startup monitoring
    if [ "$watch" = true ]; then
        [ "$quiet" = false ] && echo "Watching service status (timeout: ${timeout}s)..."
        local start_time=$(date +%s)
        local end_time=$((start_time + timeout))
        local active_services_count=0
        local first_iteration=true
        
        while [ $(date +%s) -lt $end_time ]; do
            local current_time=$(date +%s)
            local elapsed=$((current_time - start_time))
            
            # Get status from Python script
            local status_output
            if [ "$daemon_running" = true ] || [ "$check_all" = true ]; then
                export KAMIWAZA_ROOT
                local raw_status_output
                set +e
                raw_status_output=$(timeout 10s "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" status ${check_all:+-a} --json 2>&1)
                local status_exit=$?
                set -e
                # Attempt to extract the JSON payload (discard debug logging noise)
                status_output=$(printf '%s' "$raw_status_output" | python -c '
import sys, json

text = sys.stdin.read()

marker = "\"daemon\""
marker_index = text.rfind(marker)
if marker_index == -1:
    sys.exit(0)

start = text.rfind("{", 0, marker_index)
if start == -1:
    sys.exit(0)

chunk = text[start:]
depth = 0
end = None
for index, char in enumerate(chunk):
    if char == "{":
        depth += 1
    elif char == "}":
        depth -= 1
        if depth == 0:
            end = index + 1
            break

if end is None:
    sys.exit(0)

candidate = chunk[:end]

try:
    json.loads(candidate)
except json.JSONDecodeError:
    sys.exit(0)
else:
    sys.stdout.write(candidate)
')
                if [ -z "$status_output" ]; then
                    status_output="$raw_status_output"
                fi
                if [ $status_exit -eq 124 ]; then
                    # Timeout occurred, use fallback
                    status_output='{"daemon": {"running": true}, "services": {"containers": "running", "ray": "running", "core": "running", "jupyter": "running", "frontend": "running"}, "timestamp": 0}'
                    status_exit=0
                fi
            else
                status_output='{"daemon": {"running": false}, "services": {}, "timestamp": 0}'
                local status_exit=1
            fi
            
            # Parse JSON status output to count active services and their states
            if echo "$status_output" | grep -q '"running": true' >/dev/null 2>&1; then
                # Daemon is running, analyze service statuses with better parsing
                local count_summary
                count_summary=$(printf '%s' "$status_output" | python -c '
import json, sys

try:
    data = json.load(sys.stdin)
except json.JSONDecodeError:
    sys.exit(1)

services = data.get("services", {})
active_statuses = [status for status in services.values() if status not in {"disabled", "not-applicable"}]

total = len(active_statuses)
running = sum(1 for status in active_statuses if status == "running")
starting = sum(1 for status in active_statuses if status == "starting")
error = sum(1 for status in active_statuses if status in {"error", "failed"})
stopped = sum(1 for status in active_statuses if status == "stopped")

print(total, running, starting, error, stopped)
')

                local total_active_services=0
                local running_count=0
                local starting_count=0
                local error_count=0
                local stopped_count=0

                if [ -n "$count_summary" ]; then
                    read -r total_active_services running_count starting_count error_count stopped_count <<EOF
$count_summary
EOF
                fi

                # On first iteration, establish baseline of active services
                if [ "$first_iteration" = true ]; then
                    active_services_count=$total_active_services
                    first_iteration=false
                    [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] Monitoring $active_services_count active service(s)..."
                fi
                
                # Calculate services in final states (running, error, failed, stopped when daemon down)
                local final_state_count=0
                final_state_count=$((${running_count:-0} + ${error_count:-0}))
                if [ "$daemon_running" = false ]; then
                    final_state_count=$((${final_state_count:-0} + ${stopped_count:-0}))
                fi
                
                # Provide detailed status update
                if [ "${starting_count:-0}" -gt 0 ]; then
                    [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] $starting_count starting, $running_count running, $error_count with errors ($final_state_count/$active_services_count ready)"
                elif [ "${error_count:-0}" -gt 0 ] && [ "${starting_count:-0}" -eq 0 ]; then
                    [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] $error_count service(s) with errors, $running_count running ($final_state_count/$active_services_count ready)"
                elif [ "${running_count:-0}" -gt 0 ]; then
                    [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] $running_count service(s) running ($final_state_count/$active_services_count ready)"
                else
                    [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] Checking services... ($final_state_count/$active_services_count ready)"
                fi
                
                # Exit when all active services are in final states
                if [ "${active_services_count:-0}" -gt 0 ] && [ "${final_state_count:-0}" -ge "${active_services_count:-0}" ]; then
                    if [ "${error_count:-0}" -gt 0 ]; then
                        [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] All services reached final state ($error_count with errors)"
                        exit_code=2
                    else
                        [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] All services ready! ($running_count running)"
                    fi
                    break
                fi
            else
                [ "$quiet" = false ] && echo "[$elapsed/${timeout}s] Daemon not running"
                break
            fi
            
            sleep 2
        done
        
        if [ $(date +%s) -ge $end_time ]; then
            [ "$quiet" = false ] && echo "Watch timeout reached"
            exit_code=2
        fi
    fi
    
    # Get and display service status
    if [ "$daemon_running" = true ]; then
        local cmd_args="status"
        [ "$check_all" = true ] && cmd_args="$cmd_args -a"
        [ "$json" = true ] && cmd_args="$cmd_args --json"
        
        # Run status and optionally filter out services with N/A or DISABLED statuses for human-readable output
        local python_output=""
        local python_exit=0
        if [ "$json" = true ]; then
            python_output=$("${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" $cmd_args)
            python_exit=$?
            echo "$python_output"
        else
            # First run without PTY to capture accurate exit code
            local plain_output
            plain_output=$("${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" $cmd_args)
            python_exit=$?
            # Then run inside a PTY to preserve colors if possible
            if command -v script >/dev/null 2>&1; then
                if script -q -c "printf test" /dev/null >/dev/null 2>&1; then
                    python_output=$(script -q -c "${PYTHON_INTERPRETER_STRING} $KAMIWAZA_SCRIPT $cmd_args" /dev/null)
                else
                    python_output=$(script -q /dev/null "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" $cmd_args)
                fi
            else
                # Try to force color output in minimal environments
                python_output=$(TERM=${TERM:-xterm-256color} CLICOLOR=1 CLICOLOR_FORCE=1 FORCE_COLOR=1 PY_COLORS=1 "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" $cmd_args)
            fi
            # Filter lines that indicate N/A or DISABLED service statuses while preserving other content
            echo "$python_output" | awk '
                {
                    original_line = $0;
                    clean_line = original_line;
                    # Strip ANSI escape sequences for matching only
                    gsub(/\033\[[0-9;]*[A-Za-z]/, "", clean_line);
                    if (clean_line ~ /(^|[^A-Za-z])(N\/A|DISABLED)([^A-Za-z]|$)/) next;
                    print original_line;
                }
            '
        fi
        
        # Map Python exit codes to shell exit codes
        case $python_exit in
            0) ;; # All good, keep current exit_code
            *) exit_code=2 ;; # Service issues
        esac
    elif [ "$check_all" = true ] || [ "$auto_fallback" = true ]; then
        [ "$quiet" = false ] && [ "$json" = false ] && echo "Checking service statuses directly..."
        
        local cmd_args=("status" "-a")
        if [ "$json" = true ]; then
            cmd_args+=("--json")
        fi
        local cmd_args_str
        cmd_args_str=$(printf '%q ' "$KAMIWAZA_SCRIPT" "${cmd_args[@]}")
        cmd_args_str=${cmd_args_str% }
        
        # Run status and optionally filter out services with N/A or DISABLED statuses for human-readable output
        local python_output=""
        local python_exit=0
        if [ "$json" = true ]; then
            python_output=$("${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" "${cmd_args[@]}")
            python_exit=$?
            echo "$python_output"
        else
            # First run without PTY to capture accurate exit code
            local plain_output
            plain_output=$("${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" "${cmd_args[@]}")
            python_exit=$?
            # Then run inside a PTY to preserve colors if possible
            if command -v script >/dev/null 2>&1; then
                if script -q -c "printf test" /dev/null >/dev/null 2>&1; then
                    local interpreter_str
                    interpreter_str=$(printf '%q ' "${PYTHON_INTERPRETER_COMMAND[@]}")
                    interpreter_str=${interpreter_str% }
                    python_output=$(script -q -c "${interpreter_str} ${cmd_args_str}" /dev/null)
                else
                    python_output=$(script -q /dev/null "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" "${cmd_args[@]}")
                fi
            else
                # Try to force color output in minimal environments
                python_output=$(TERM=${TERM:-xterm-256color} CLICOLOR=1 CLICOLOR_FORCE=1 FORCE_COLOR=1 PY_COLORS=1 "${PYTHON_INTERPRETER_COMMAND[@]}" "$KAMIWAZA_SCRIPT" "${cmd_args[@]}")
            fi
            # Filter lines that indicate N/A or DISABLED service statuses while preserving other content
            echo "$python_output" | awk '
                {
                    original_line = $0;
                    clean_line = original_line;
                    # Strip ANSI escape sequences for matching only
                    gsub(/\033\[[0-9;]*[A-Za-z]/, "", clean_line);
                    if (clean_line ~ /(^|[^A-Za-z])(N\/A|DISABLED)([^A-Za-z]|$)/) next;
                    print original_line;
                }
            '
        fi
        
        # If daemon is down but services are accessible, that's still an issue
        case $python_exit in
            0) exit_code=1 ;; # Services OK but daemon down
            *) exit_code=2 ;; # Services also have issues
        esac
    fi
    
    # Show logs if requested
    if [ "$logs" = true ] && [ -f "$LOG_FILE" ]; then
        [ "$quiet" = false ] && [ "$json" = false ] && echo "--- Recent logs ---"
        tail -n 20 "$LOG_FILE"
    fi
    
    return $exit_code
}
doctor() {
    print_section() {
        echo
        cecho 6 "═══ $1 ═══"
        echo
    }
    
    print_check() {
        local status="$1"
        local message="$2"
        case "$status" in
            "ok") cecho 2 "  ✓ $message" ;;
            "warn") cecho 3 "  ⚠ $message" ;;
            "error") cecho 1 "  ✗ $message" ;; 
            "info") cecho 7 "  ℹ $message" ;;
        esac
    }
    
    # Header
    echo
    cecho 6 "╔══════════════════════════════════════════════════════════════╗"
    cecho 6 "║                    KAMIWAZA DIAGNOSTIC REPORT                ║"
    cecho 6 "╚══════════════════════════════════════════════════════════════╝"
    
    print_section "Network Ports Analysis"
    local critical_ports=(3000 7777 8265 443)
    local port_issues=0
    
    for port in "${critical_ports[@]}"; do
        local port_info=$(lsof -i :$port 2>/dev/null)
        if [ -n "$port_info" ]; then
            print_check "ok" "Port $port: In use"
            echo "$port_info" | grep -v "COMMAND" | while read line; do
                if [ -n "$line" ]; then
                    local process=$(echo "$line" | awk '{print $1 " (PID: " $2 ")"}')
                    cecho 7 "      → $process"
                fi
            done
        else
            case $port in
                3000) print_check "warn" "Port $port: Frontend not detected" ;;
                7777) print_check "warn" "Port $port: API server not detected" ;;
                8265) print_check "warn" "Port $port: Ray dashboard not detected" ;;
                443) print_check "info" "Port $port: No HTTPS traffic" ;;
            esac
            port_issues=$((port_issues + 1))
        fi
    done
    
    print_section "Container Environment"
    if command -v docker >/dev/null 2>&1; then
        if docker info >/dev/null 2>&1; then
            print_check "ok" "Docker daemon is accessible"
            
            # Get container statistics
            local total_containers=$(docker ps -a --format "table {{.Names}}" | tail -n +2 | wc -l)
            local running_containers=$(docker ps --format "table {{.Names}}" | tail -n +2 | wc -l)
            local kamiwaza_containers=$(docker ps -a --format "table {{.Names}}" | grep -c "kamiwaza" || echo "0")
            
            if [ $total_containers -gt 0 ]; then
                print_check "info" "Total containers: $total_containers (Running: $running_containers)"
                if [ $kamiwaza_containers -gt 0 ]; then
                    print_check "ok" "Kamiwaza containers detected: $kamiwaza_containers"
                else
                    print_check "warn" "No Kamiwaza containers found"
                fi
                
                # Show key container status
                cecho 7 "    Key containers:"
                docker ps -a --format "table {{.Names}}\t{{.Status}}" | grep -E "(kamiwaza|default)" | while read line; do
                    if [ -n "$line" ] && [[ ! "$line" =~ ^NAME.*STATUS ]]; then
                        local name=$(echo "$line" | awk '{print $1}')
                        local status=$(echo "$line" | cut -d$'\t' -f2)
                        if [[ "$status" =~ ^Up ]]; then
                            cecho 2 "      ✓ $name: $status"
                        elif [[ "$status" =~ ^Exited ]]; then
                            cecho 3 "      ○ $name: $status"
                        else
                            cecho 1 "      ✗ $name: $status"
                        fi
                    fi
                done
            else
                print_check "warn" "No containers found"
            fi
        else
            print_check "error" "Docker daemon is not reachable"
        fi
    else
        print_check "error" "Docker is not installed"
    fi
    
    print_section "System Environment"
    
    # Check essential environment variables
    local env_vars=("KAMIWAZA_ROOT" "KAMIWAZA_HEAD_IP" "USER")
    for var in "${env_vars[@]}"; do
        local value="${!var}"
        if [ -n "$value" ]; then
            print_check "ok" "$var: $value"
        else
            print_check "warn" "$var: Not set"
        fi
    done
    
    # Check optional but useful variables
    if [ -n "${KAMIWAZA_COMMUNITY:-}" ]; then
        print_check "info" "KAMIWAZA_COMMUNITY: ${KAMIWAZA_COMMUNITY}"
    fi
    
    if [ -n "${KAMIWAZA_LITE:-}" ]; then
        print_check "info" "KAMIWAZA_LITE: ${KAMIWAZA_LITE}"
    fi
    
    # Check disk space in KAMIWAZA_ROOT
    if [ -n "${KAMIWAZA_ROOT:-}" ] && [ -d "$KAMIWAZA_ROOT" ]; then
        local disk_usage=$(df -h "$KAMIWAZA_ROOT" | awk 'NR==2 {print $5}' | sed 's/%//')
        if [ "$disk_usage" -lt 80 ]; then
            print_check "ok" "Disk usage: ${disk_usage}% (Available)"
        elif [ "$disk_usage" -lt 90 ]; then
            print_check "warn" "Disk usage: ${disk_usage}% (Getting full)"
        else
            print_check "error" "Disk usage: ${disk_usage}% (Critical)"
        fi
    fi
    
    print_section "Service Status"
    echo "  Checking current service states..."
    echo
    
    # Call our enhanced status command
    status -q
    
    # Summary
    print_section "Diagnostic Summary"
    
    local total_issues=$port_issues
    if [ $total_issues -eq 0 ]; then
        print_check "ok" "No critical issues detected"
        cecho 7 "    System appears to be functioning normally"
    elif [ $total_issues -le 2 ]; then
        print_check "warn" "$total_issues minor issue(s) detected"
        cecho 7 "    Review warnings above and consider restarting services"
    else
        print_check "error" "$total_issues issue(s) detected"
        cecho 7 "    Multiple problems found - may need troubleshooting"
    fi
    
    echo
    cecho 7 "  Helpful commands:"
    cecho 7 "    • $0 restart    - Restart all services"
    cecho 7 "    • $0 status -l  - View recent logs"
    cecho 7 "    • $SCRIPT_PATH restart    - Restart all services"
    cecho 7 "    • $SCRIPT_PATH status -l  - View recent logs"
    cecho 7 "    • docker logs <container> - Check specific container logs"
    
    echo
    cecho 6 "╚═══════════════════════════════════════════════════════════════════════════════╝"
    echo
}

version() {
    echo
    cecho 6 "╔═══════════════════════════════════════════════════════════════╗"
    cecho 6 "║                     KAMIWAZA VERSION INFO                     ║"
    cecho 6 "╚═══════════════════════════════════════════════════════════════╝"
    echo
    
    local version_file="${KAMIWAZA_ROOT:-.}/kamiwaza.version.json"
    
    if [ -f "$version_file" ]; then
        # Extract version components
        local major=$(grep -o '"KAMIWAZA_VERSION_MAJOR":"[^"]*"' "$version_file" | sed 's/.*"KAMIWAZA_VERSION_MAJOR":"\([^"]*\)".*/\1/')
        local minor=$(grep -o '"KAMIWAZA_VERSION_MINOR":"[^"]*"' "$version_file" | sed 's/.*"KAMIWAZA_VERSION_MINOR":"\([^"]*\)".*/\1/')
        local patch=$(grep -o '"KAMIWAZA_VERSION_PATCH":"[^"]*"' "$version_file" | sed 's/.*"KAMIWAZA_VERSION_PATCH":"\([^"]*\)".*/\1/')
        local build_date=$(grep -o '"BUILD_DATE":"[^"]*"' "$version_file" 2>/dev/null | sed 's/.*"BUILD_DATE":"\([^"]*\)".*/\1/' || echo "Unknown")
        local git_commit=$(grep -o '"GIT_COMMIT":"[^"]*"' "$version_file" 2>/dev/null | sed 's/.*"GIT_COMMIT":"\([^"]*\)".*/\1/' || echo "Unknown")
        
        if [ -n "$major" ] && [ -n "$minor" ] && [ -n "$patch" ]; then
            local full_version="${major}.${minor}.${patch}"
            cecho 2 "  ✓ Version: ${full_version}"
        else
            cecho 1 "  ✗ Unable to parse version from file"
        fi
        
        if [ "$build_date" != "Unknown" ]; then
            cecho 7 "  ℹ Build Date: $build_date"
        fi
        
        if [ "$git_commit" != "Unknown" ]; then
            local short_commit=$(echo "$git_commit" | cut -c1-8)
            cecho 7 "  ℹ Git Commit: $short_commit"
        fi
        
    else
        cecho 1 "  ✗ Version file not found: $version_file"
        cecho 3 "  ⚠ This may indicate an incomplete installation"
    fi
    
    echo
    cecho 6 "═══ System Information ═══"
    echo
    
    # Installation details
    if [ -n "${KAMIWAZA_ROOT:-}" ]; then
        cecho 2 "  ✓ Install Path: $KAMIWAZA_ROOT"
        
        # Check installation type
        if [ -f "${KAMIWAZA_ROOT}/.kamiwaza_install_community" ]; then
            cecho 7 "  ℹ Edition: Community"
        else
            cecho 7 "  ℹ Edition: Enterprise"
        fi
        
        # Check if lite mode
        if [ "${KAMIWAZA_LITE:-}" = "true" ]; then
            cecho 7 "  ℹ Mode: Lite"
        else
            cecho 7 "  ℹ Mode: Full"
        fi
    else
        cecho 1 "  ✗ KAMIWAZA_ROOT not set"
    fi
    
    # System details
    local os_info=$(uname -s 2>/dev/null || echo "Unknown")
    local arch=$(uname -m 2>/dev/null || echo "Unknown")
    cecho 7 "  ℹ Platform: $os_info $arch"
    
    # Python version (via uv/kw_py)
    if command -v "${PYTHON_INTERPRETER_COMMAND[0]}" >/dev/null 2>&1; then
        local python_version=$("${PYTHON_INTERPRETER_COMMAND[@]}" -c 'import platform; print(platform.python_version())' 2>/dev/null || echo "Unknown")
        cecho 7 "  ℹ Python: $python_version"
    else
        cecho 1 "  ✗ Python interpreter not available"
    fi
    
    # Docker version (if available)
    if command -v docker >/dev/null 2>&1; then
        local docker_version=$(docker --version 2>/dev/null | awk '{print $3}' | sed 's/,//' || echo "Unknown")
        if [ "$docker_version" != "Unknown" ]; then
            cecho 7 "  ℹ Docker: $docker_version"
        fi
    fi
    
    echo
    cecho 6 "═══ License & Support ═══"
    echo
    
    # License information
    local license_file="${KAMIWAZA_ROOT:-}/.kamiwaza_license"
    if [ -f "$license_file" ]; then
        cecho 2 "  ✓ License: Active"
    else
        cecho 3 "  ⚠ License: Not found (using default)"
    fi
    
    # Support links
    cecho 7 "  ℹ Documentation: https://docs.kamiwaza.ai"
    cecho 7 "  ℹ Support: support@kamiwaza.ai"
    
    echo
    cecho 6 "╚═══════════════════════════════════════════════════════════════════════════════╝"
    echo
}

show_help() {
    echo
    cecho 6 "╔═══════════════════════════════════════════════════════════════╗"
    cecho 6 "║                      KAMIWAZA CONTROL                         ║"
    cecho 6 "╚═══════════════════════════════════════════════════════════════╝"
    echo
    
    cecho 7 "  Service management tool for Kamiwaza AI platform"
    echo
    
    cecho 6 "═══ Commands ═══"
    echo
    cecho 2 "  start [--missing]"
    cecho 7 "    Start all services or only missing/stopped services"
    cecho 7 "    Example: $0 start"
    cecho 7 "    Example: $SCRIPT_PATH start"
    echo
    
    cecho 2 "  stop"
    cecho 7 "    Stop all services gracefully"
    cecho 7 "    Example: $0 stop"
    cecho 7 "    Example: $SCRIPT_PATH stop"
    echo
    
    cecho 2 "  restart"
    cecho 7 "    Stop all services, then start them again"
    cecho 7 "    Example: $0 restart"
    cecho 7 "    Example: $SCRIPT_PATH restart"
    echo

    if is_dev_env; then
        cecho 2 "  restart-core"
        cecho 7 "    Restart Ray + backend core (fast dev helper)"
        cecho 7 "    Example: $0 restart-core"
        cecho 7 "    Example: $SCRIPT_PATH restart-core"
        echo
    cecho 2 "  restart-core"
    cecho 7 "    Restart Ray + backend core (fast restart)"
    cecho 7 "    Example: $SCRIPT_PATH restart-core"
    echo

        cecho 2 "  restart-frontend"
        cecho 7 "    Rebuild and restart the frontend in production via PM2"
        cecho 7 "    Example: $0 restart-frontend"
        cecho 7 "    Example: $SCRIPT_PATH restart-frontend"
        echo
    fi
    cecho 2 "  restart-frontend"
    cecho 7 "    Rebuild and restart the frontend in production via PM2"
    cecho 7 "    Example: $SCRIPT_PATH restart-frontend"
    echo
    
    cecho 2 "  status [options]"
    cecho 7 "    Show current status of all services"
    cecho 3 "    Options:"
    cecho 7 "      -a    Check all services (even if daemon is down)"
    cecho 7 "      -q    Quiet mode (minimal shell output)"
    cecho 7 "      -j    JSON output format"
    cecho 7 "      -w    Watch mode for startup monitoring"
    cecho 7 "      -l    Show recent logs after status"
    cecho 7 "      -t N  Timeout for watch mode (default: 180s)"
    cecho 7 "    Examples:"
    cecho 7 "      $0 status       # Human-readable status"
    cecho 7 "      $0 status -j    # JSON format"
    cecho 7 "      $0 status -w    # Watch startup progress"
    cecho 7 "      $SCRIPT_PATH status       # Human-readable status"
    cecho 7 "      $SCRIPT_PATH status -j    # JSON format"
    cecho 7 "      $SCRIPT_PATH status -w    # Watch startup progress"
    echo
    
    cecho 2 "  doctor"
    cecho 7 "    Run comprehensive diagnostic checks"
    cecho 7 "    Example: $0 doctor"
    cecho 7 "    Example: $SCRIPT_PATH doctor"
    echo
    
    cecho 2 "  version"
    cecho 7 "    Show version and system information"
    cecho 7 "    Example: $0 version"
    cecho 7 "    Example: $SCRIPT_PATH version"
    echo

    cecho 2 "  extensions [subcommand]"
    cecho 7 "    Manage extensions and templates"
    cecho 3 "    Subcommands:"
    cecho 7 "      sync      - Sync templates from registry"
    cecho 7 "      install   - Install registry package"
    cecho 7 "      status    - Show extensions configuration"
    cecho 7 "      help      - Show extensions help"
    cecho 7 "    Example: $SCRIPT_PATH extensions sync"
    echo

    cecho 2 "  help"
    cecho 7 "    Show this help message"
    cecho 7 "    Example: $0 help"
    cecho 7 "    Example: $SCRIPT_PATH help"
    echo
    
    cecho 6 "═══ Common Usage Patterns ═══"
    echo
    cecho 3 "  Check if system is running:"
    cecho 7 "    $0 status"
    cecho 7 "    $SCRIPT_PATH status"
    echo
    cecho 3 "  Monitor system startup:"
    cecho 7 "    $0 status -w"
    cecho 7 "    $SCRIPT_PATH status -w"
    echo
    cecho 3 "  Troubleshoot issues:"
    cecho 7 "    $0 doctor"
    cecho 7 "    $0 status -l"
    cecho 7 "    $SCRIPT_PATH doctor"
    cecho 7 "    $SCRIPT_PATH status -l"
    echo
    cecho 3 "  Full restart:"
    cecho 7 "    $0 restart"
    cecho 7 "    $SCRIPT_PATH restart"
    echo
    cecho 3 "  Get machine-readable status:"
    cecho 7 "    $0 status -j"
    cecho 7 "    $SCRIPT_PATH status -j"
    echo
    
    cecho 6 "═══ Support ═══"
    echo
    cecho 7 "  Documentation: https://docs.kamiwaza.ai"
    cecho 7 "  Support: support@kamiwaza.ai"
    cecho 7 "  Logs: \$KAMIWAZA_LOG_DIR (currently: ${KAMIWAZA_LOG_DIR:-/tmp})"
    echo
    cecho 6 "╚═══════════════════════════════════════════════════════════════════════════════╝"
    echo
}

case "${1:-}" in
    start)
        start "${2:-}"  # Pass second argument to start
        ;;
    stop)
        stop
        ;;
    restart-frontend)
        # Restart the React frontend (dev or container mode)
        restart_frontend() {
            # Determine mode from environment
            local devmode="${KAMIWAZA_FRONTEND_DEVMODE:-false}"
            local mode="container"
            if [ "$devmode" = "true" ]; then
                mode="development"
            fi

            cecho 6 "═══════════════════════════════════════════════════════════════"
            cecho 6 "  RESTARTING FRONTEND (mode: ${mode})"
            cecho 6 "═══════════════════════════════════════════════════════════════"

            # Ensure KAMIWAZA_ROOT
            cd "${KAMIWAZA_ROOT}" || { cecho 1 "KAMIWAZA_ROOT not set or invalid"; return 1; }

            # Stop any running frontend
            if [ -f "${KAMIWAZA_ROOT}/frontend/stop-frontend.sh" ]; then
                cecho 3 "⏹  Stopping existing frontend..."
                bash "${KAMIWAZA_ROOT}/frontend/stop-frontend.sh" || true
            fi

            # Prepare logs
            local FRONTEND_LOG_FILE="${LOG_DIR}/kamiwazad-frontend.log"
            touch "$FRONTEND_LOG_FILE" 2>/dev/null || true

            # Start frontend using the new script
            cd "${KAMIWAZA_ROOT}/frontend" || { cecho 1 "frontend directory not found"; return 1; }

            cecho 3 "🚀 Starting frontend..."
            if bash start-frontend.sh 1>>"$FRONTEND_LOG_FILE" 2>&1; then
                cecho 2 "✓ Frontend restart successful (mode: ${mode})"
                cecho 7 "  Logs: $FRONTEND_LOG_FILE"
            else
                cecho 1 "✗ Frontend restart failed (see $FRONTEND_LOG_FILE)"
                return 1
            fi

            cecho 6 "───────────────────────────────────────────────────────────────"
        }

        restart_frontend
        ;;
    restart-core)
        # Restart only the Python core backend quickly
        restart_core() {
            cecho 6 "═══════════════════════════════════════════════════════════════"
            cecho 6 "  RESTARTING KAMIWAZA CORE + RAY"
            cecho 6 "═══════════════════════════════════════════════════════════════"

            # Ensure we are in KAMIWAZA_ROOT
            cd "${KAMIWAZA_ROOT}" || { cecho 1 "KAMIWAZA_ROOT not set or invalid"; return 1; }

            # Stop core processes FIRST to allow ephemeral deployment cleanup
            # (cleanup requires Ray to be running to stop containers)
            if [ -f "${KAMIWAZA_ROOT}/stop-core.sh" ]; then
                cecho 3 "⏹  Stopping core (with cleanup)..."
                bash "${KAMIWAZA_ROOT}/stop-core.sh" || true
            else
                cecho 3 "⚠  stop-core.sh not found; skipping stop"
            fi

            # Stop Ray after core has cleaned up
            cecho 3 "⏹  Stopping Ray..."
            if ! run_py_cli ray stop; then
                # If ray stop fails, try to continue anyway (may not be running)
                cecho 3 "⚠  Ray stop failed or Ray not running (continuing)"
            fi

            # Brief pause to allow processes to fully stop
            sleep 2

            # Start Ray
            cecho 3 "🚀 Starting Ray..."
            if [ -f "${KAMIWAZA_ROOT}/start-ray.sh" ]; then
                bash "${KAMIWAZA_ROOT}/start-ray.sh" || { cecho 1 "✗ Ray start failed"; return 1; }
                cecho 2 "✓ Ray started successfully"
            else
                cecho 1 "✗ start-ray.sh not found in $KAMIWAZA_ROOT"
                return 1
            fi

            # Prepare log file
            CORE_LOG_FILE="${LOG_DIR}/kamiwazad-core.log"
            touch "$CORE_LOG_FILE" 2>/dev/null || true

            # Start core in background via launch.sh (non-blocking)
            if [ -f "${KAMIWAZA_ROOT}/launch.sh" ]; then
                cecho 3 "🚀 Starting core (background)..."
                nohup bash "${KAMIWAZA_ROOT}/launch.sh" 1>>"$CORE_LOG_FILE" 2>&1 &
                core_wrapper_pid=$!
                echo "$core_wrapper_pid" > /tmp/kamiwaza-core-wrapper.pid
                cecho 2 "✓ Core start initiated (wrapper PID: $core_wrapper_pid)"
                cecho 7 "  Logs: $CORE_LOG_FILE"
            else
                cecho 1 "✗ launch.sh not found in $KAMIWAZA_ROOT"
                return 1
            fi

            cecho 6 "───────────────────────────────────────────────────────────────"
        }

        restart_core
        ;;
    restart)
        stop
        start
        ;;
    status)
        status "${2:-}"  # Pass the second argument to status
        ;;
    doctor)
        doctor
        ;;
    version)
        version
        exit 0
        ;;

    extensions)
        # Extensions management commands
        extensions() {
            local subcommand="$1"
            local arg="$2"

            case "${subcommand:-}" in
                sync)
                    cecho 6 "═══ EXTENSIONS SYNC ═══"
                    echo

                    # Check for stage override
                    local stage=""
                    if [ -n "${arg:-}" ]; then
                        stage="--stage ${arg}"
                        cecho 7 "Using stage: ${arg}"
                    elif [ -n "${KAMIWAZA_EXTENSION_STAGE:-}" ]; then
                        cecho 7 "Using configured stage: ${KAMIWAZA_EXTENSION_STAGE}"
                        if [ "${KAMIWAZA_EXTENSION_STAGE}" = "LOCAL" ] && [ -n "${KAMIWAZA_EXTENSION_LOCAL_STAGE_URL:-}" ]; then
                            cecho 7 "Local URL: ${KAMIWAZA_EXTENSION_LOCAL_STAGE_URL}"
                        fi
                    fi

                    # Run the sync script
                    if [ -f "${KAMIWAZA_ROOT}/scripts/app_garden_remote_sync.py" ]; then
                        cecho 3 "⏳ Syncing extensions from registry..."
                        "${PYTHON_INTERPRETER_COMMAND[@]}" "${KAMIWAZA_ROOT}/scripts/app_garden_remote_sync.py" $stage
                        local exit_code=$?

                        if [ $exit_code -eq 0 ]; then
                            cecho 2 "✓ Extensions sync completed successfully"
                        else
                            cecho 1 "✗ Extensions sync failed"
                            exit $exit_code
                        fi
                    else
                        cecho 1 "✗ Sync script not found: ${KAMIWAZA_ROOT}/scripts/app_garden_remote_sync.py"
                        exit 1
                    fi
                    ;;

                install)
                    # Install registry package
                    local registry_package="${arg:-}"
                    if [ -z "$registry_package" ]; then
                        cecho 1 "Error: Registry package path required"
                        cecho 7 "Usage: $SCRIPT_PATH extensions install <registry-package.tar.gz>"
                        exit 1
                    fi

                    if [ ! -f "$registry_package" ]; then
                        cecho 1 "Error: Registry package not found: $registry_package"
                        exit 1
                    fi

                    cecho 6 "═══ INSTALLING EXTENSIONS REGISTRY ═══"
                    echo

                    local extract_path="${KAMIWAZA_EXTENSION_INSTALL_PATH:-/opt/kamiwaza/extensions}"
                    cecho 7 "Extract path: $extract_path"

                    # Create extraction directory if needed
                    if ! mkdir -p "$extract_path" 2>/dev/null; then
                        cecho 3 "Using sudo to create extraction directory..."
                        sudo mkdir -p "$extract_path"
                        sudo chown "$USER:$(id -g)" "$extract_path"
                    fi

                    # Extract archive
                    cecho 3 "⏳ Extracting registry package..."
                    tar -xzf "$registry_package" -C "$extract_path"

                    if [ $? -eq 0 ]; then
                        cecho 2 "✓ Registry extracted successfully"
                        echo
                        cecho 7 "To use this registry, set these environment variables:"
                        cecho 7 "  export KAMIWAZA_EXTENSION_STAGE=LOCAL"
                        cecho 7 "  export KAMIWAZA_EXTENSION_LOCAL_STAGE_URL=file://$extract_path/kamiwaza-extension-registry"
                        echo
                        cecho 7 "Then run: $SCRIPT_PATH extensions sync"
                    else
                        cecho 1 "✗ Failed to extract registry"
                        exit 1
                    fi
                    ;;

                status)
                    cecho 6 "═══ EXTENSIONS STATUS ═══"
                    echo

                    # Show current configuration
                    if [ -n "${KAMIWAZA_EXTENSION_STAGE:-}" ]; then
                        cecho 2 "✓ Extension Stage: ${KAMIWAZA_EXTENSION_STAGE}"

                        if [ "${KAMIWAZA_EXTENSION_STAGE}" = "LOCAL" ] && [ -n "${KAMIWAZA_EXTENSION_LOCAL_STAGE_URL:-}" ]; then
                            cecho 2 "✓ Local URL: ${KAMIWAZA_EXTENSION_LOCAL_STAGE_URL}"

                            # Check if local path exists
                            if [[ "${KAMIWAZA_EXTENSION_LOCAL_STAGE_URL}" == file://* ]]; then
                                local local_path="${KAMIWAZA_EXTENSION_LOCAL_STAGE_URL#file://}"
                                if [ -d "$local_path" ]; then
                                    cecho 2 "✓ Local registry path exists"

                                    # Check for JSON files
                                    if [ -f "$local_path/garden/default/apps.json" ]; then
                                        local app_count=$(grep -c '"name"' "$local_path/garden/default/apps.json" || echo "0")
                                        cecho 7 "  Apps available: $app_count"
                                    fi

                                    if [ -f "$local_path/garden/default/tools.json" ]; then
                                        local tool_count=$(grep -c '"name"' "$local_path/garden/default/tools.json" || echo "0")
                                        cecho 7 "  Tools available: $tool_count"
                                    fi
                                else
                                    cecho 1 "✗ Local registry path not found: $local_path"
                                fi
                            fi
                        fi
                    else
                        cecho 3 "⚠ Extension stage not configured"
                        cecho 7 "  Set KAMIWAZA_EXTENSION_STAGE to enable extensions"
                    fi
                    ;;

                help|--help|-h|"")
                    cecho 6 "═══ EXTENSIONS MANAGEMENT ═══"
                    echo
                    cecho 7 "Commands for managing Kamiwaza extensions and templates"
                    echo
                    cecho 2 "  extensions sync [STAGE]"
                    cecho 7 "    Sync extensions from configured or specified source"
                    cecho 7 "    STAGE options: LOCAL, DEV, STAGE, PROD"
                    cecho 7 "    Example: $SCRIPT_PATH extensions sync"
                    cecho 7 "    Example: $SCRIPT_PATH extensions sync LOCAL"
                    echo
                    cecho 2 "  extensions install <registry-package.tar.gz>"
                    cecho 7 "    Install extensions from registry package"
                    cecho 7 "    Example: $SCRIPT_PATH extensions install kamiwaza-registry-v1.0.tar.gz"
                    echo
                    cecho 2 "  extensions status"
                    cecho 7 "    Show current extensions configuration and status"
                    cecho 7 "    Example: $SCRIPT_PATH extensions status"
                    echo
                    cecho 2 "  extensions help"
                    cecho 7 "    Show this help message"
                    echo
                    cecho 6 "═══ Configuration ═══"
                    echo
                    cecho 7 "Environment variables:"
                    cecho 7 "  KAMIWAZA_EXTENSION_STAGE        - Extension source (LOCAL/DEV/STAGE/PROD)"
                    cecho 7 "  KAMIWAZA_EXTENSION_LOCAL_STAGE_URL - URL for LOCAL stage (file:// or https://)"
                    echo
                    cecho 7 "For offline deployments:"
                    cecho 7 "  1. Install registry: $SCRIPT_PATH extensions install kamiwaza-registry.tar.gz"
                    cecho 7 "  2. Set environment: export KAMIWAZA_EXTENSION_STAGE=LOCAL"
                    cecho 7 "  3. Set URL: export KAMIWAZA_EXTENSION_LOCAL_STAGE_URL=file:///opt/kamiwaza/extensions/kamiwaza-extension-registry"
                    cecho 7 "  4. Sync templates: $SCRIPT_PATH extensions sync"
                    ;;

                *)
                    cecho 1 "Unknown extensions command: ${subcommand:-}"
                    echo
                    $0 extensions help
                    exit 1
                    ;;
            esac
        }

        extensions "${2:-}" "${3:-}"
        ;;

    help|--help|-h)
        show_help
        exit 0
        ;;
    *)
        show_help
        exit 1
        ;;
esac

exit 0
