#!/bin/bash

set -euo pipefail

# Resolve project root and ensure KAMIWAZA_ROOT is set
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
if [ -z "${KAMIWAZA_ROOT:-}" ]; then
  # Source the shared root detector; it can handle being sourced
  if [ -f "${SCRIPT_DIR}/../set-kamiwaza-root.sh" ]; then
    # shellcheck disable=SC1091
    source "${SCRIPT_DIR}/../set-kamiwaza-root.sh"
  fi
fi

if [ -z "${KAMIWAZA_ROOT:-}" ]; then
  echo "KAMIWAZA_ROOT is not set" >&2
  exit 1
fi

# Helper: check if a pattern exists in process list
ps_matches() {
  local pattern="$1"
  # Use grep -E with bracket trick to avoid matching the grep itself
  ps auxww | grep -E "$pattern" | grep -v grep >/dev/null 2>&1
}

# Detect layout (top-level launch.py or in-package launch.py)
TOP_LEVEL_LAUNCH="${KAMIWAZA_ROOT}/launch.py"
PKG_LAUNCH="${KAMIWAZA_ROOT}/kamiwaza/launch.py"

# Conditions to consider core healthy:
#  - Launch wrapper running (launch.py)
#  - Main app running (either module form or path form)
#  - Scheduler running (scheduler/run.py)

# Launch check: match exact paths if possible, otherwise generic launch.py
LAUNCH_OK=false
if [ -f "$TOP_LEVEL_LAUNCH" ]; then
  ps_matches "[l]aunch\.py" && LAUNCH_OK=true
elif [ -f "$PKG_LAUNCH" ]; then
  ps_matches "[l]aunch\.py" && LAUNCH_OK=true
else
  # Fallback: any launch.py
  ps_matches "[l]aunch\.py" && LAUNCH_OK=true
fi

# Main check: module (-m kamiwaza.main) or path */kamiwaza/main.py or */main.py under root
MAIN_OK=false
if ps_matches "(-m[ ]+)?kamiwaza\.main"; then
  MAIN_OK=true
elif ps_matches "kamiwaza/main\.py"; then
  MAIN_OK=true
elif ps_matches "${KAMIWAZA_ROOT//\//\\/}/kamiwaza/main\.py"; then
  MAIN_OK=true
elif ps_matches "[m]ain\.py"; then
  MAIN_OK=true
fi

# Scheduler check: path */scheduler/run.py
SCHED_OK=false
if ps_matches "scheduler/run\.py"; then
  SCHED_OK=true
fi

if $LAUNCH_OK && $MAIN_OK && $SCHED_OK; then
  exit 0
else
  exit 1
fi

