#!/bin/bash
set -exuo pipefail

# Try sourcing env.sh directly first
if [[ -f /etc/kamiwaza/env.sh ]]; then
    source /etc/kamiwaza/env.sh
else
    # If that fails, try loading set-kamiwaza-root.sh and check alternate location
    if [[ -f "$(dirname "$0")/set-kamiwaza-root.sh" ]]; then
        source "$(dirname "$0")/set-kamiwaza-root.sh"
        if [[ -f "${KAMIWAZA_ROOT}/env.sh" ]]; then
            source "${KAMIWAZA_ROOT}/env.sh"
        fi
    fi
fi

source common.sh

# Early exit for Darwin community edition
if [[ "$(uname)" == "Darwin" && "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    echo "* Kamiwaza in community edition mode on Darwin; skipping swarm setup"
    return 0
fi

if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    echo "* Kamiwaza in community edition mode; skipping swarm setup"
    return 0
fi

verify_swarm_state() {
    if docker info | grep -q "Swarm: active"; then
        local node_role=$(docker info | grep "Is Manager:" | awk '{print $3}')
        
        # If we're meant to be head, being a manager is good
        if [[ "${KAMIWAZA_SWARM_HEAD:-}" == "true" && "$node_role" == "true" ]]; then
            return 0
        # If we're meant to be worker, being a non-manager is good
        elif [[ -n "${KAMIWAZA_HEAD_IP:-}" && "$node_role" != "true" ]]; then
            return 0
        fi
        # In any other state, we need to reconfigure
        return 1
    fi
    return 1
}

initialize_head() {
    if verify_swarm_state; then
        echo "Swarm already properly initialized"
        # Still ensure token is available
        mkdir -p /etc/kamiwaza/swarm
        docker swarm join-token -q worker > /etc/kamiwaza/swarm/token
        chmod 600 /etc/kamiwaza/swarm/token
        return 0
    fi

    # If we're here, we either need to init or reconfigure
    if docker info | grep -q "Swarm: active"; then
        echo "Leaving existing swarm..."
        docker swarm leave --force
    fi

    local max_attempts=5
    local attempt=1
    while [[ $attempt -le $max_attempts ]]; do
        echo "Initializing swarm (attempt $attempt/$max_attempts)..."
        local advertise_addr="${SWARM_LISTEN_IP:-$(best_ip_for_hostname)}"

        if docker swarm init --advertise-addr "$advertise_addr"; then
            mkdir -p /etc/kamiwaza/swarm
            docker swarm join-token -q worker > /etc/kamiwaza/swarm/token
            chmod 600 /etc/kamiwaza/swarm/token
            # tiny amount of waiting here
            sleep 5
            return 0
        fi
        ((attempt++))
        sleep 5
    done
    return 1
}

initialize_worker() {
    if verify_swarm_state; then
        echo "Already joined swarm properly"
        return 0
    fi

    # Verify SSH key exists and has proper permissions
    if [[ ! -f /etc/kamiwaza/ssl/cluster.key ]]; then
        echo "Error: Missing cluster.key"
        return 1
    fi
    chmod 600 /etc/kamiwaza/ssl/cluster.key

    local max_attempts=5
    local attempt=1
    while [[ $attempt -le $max_attempts ]]; do
        echo "Joining swarm (attempt $attempt/$max_attempts)..."
        TOKEN=$(ssh -i /etc/kamiwaza/ssl/cluster.key \
            -o StrictHostKeyChecking=no \
            -o UserKnownHostsFile=/dev/null \
            ${USER}@${KAMIWAZA_HEAD_IP} \
            "cat /etc/kamiwaza/swarm/token")
            
        if [[ -n "$TOKEN" ]] && docker swarm join --token "${TOKEN}" "${KAMIWAZA_HEAD_IP}:2377"; then
            return 0
        fi
        ((attempt++))
        sleep 5
    done
    return 1
}

setup_swarm() {
    # Enable required kernel modules and sysctl settings for Docker swarm networking
    # Only run on Linux systems
    if [[ "$(uname)" == "Linux" ]]; then
        sudo modprobe br_netfilter || true
        sudo sysctl -w net.bridge.bridge-nf-call-iptables=1 || true
        sudo sysctl -w net.bridge.bridge-nf-call-ip6tables=1 || true

        # Quick verification
        if ! lsmod | grep -q br_netfilter || \
           [[ "$(sysctl -n net.bridge.bridge-nf-call-iptables)" != "1" ]] || \
           [[ "$(sysctl -n net.bridge.bridge-nf-call-ip6tables)" != "1" ]]; then
            echo "Warning: Some network settings may not be optimal for swarm operation"
        fi
    fi

    if [[ -n "${KAMIWAZA_SWARM_HEAD:-}" && "${KAMIWAZA_SWARM_HEAD:-}" == "true" ]]; then
        initialize_head
    elif [[ -n "${KAMIWAZA_HEAD_IP:-}" ]]; then
        # Check if we are the head node by comparing our IP with KAMIWAZA_HEAD_IP
        if ifconfig | grep inet | awk '{print $2}' | grep -q "^${KAMIWAZA_HEAD_IP:-}$" > /dev/null 2>&1; then
            initialize_head
        else
            initialize_worker
        fi
    else
        echo "Error: Neither KAMIWAZA_SWARM_HEAD nor KAMIWAZA_HEAD_IP is set"
        exit 1
    fi
}

# Check if .kamiwaza_install_community exists in script's directory and set KAMIWAZA_COMMUNITY
if [[ -f "$(dirname "${BASH_SOURCE[0]}")/.kamiwaza_install_community" ]]; then
    export KAMIWAZA_COMMUNITY=true
fi

# Execute main if script is run directly (not sourced)
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    setup_swarm "$@"
fi