#!/bin/bash
set -euo pipefail

create_network_if_not_exists() {
    local network_name=$1
    local network_driver="${2:-overlay}"
    local network_opts="${3:-}"
    
    # For community edition, use bridge networks
    if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
        network_driver="bridge"
        network_opts=""
    fi

    # Check if we're a manager node or using community edition
    if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]] || docker info | grep "Is Manager:" | grep -q "true"; then
        # Manager node or community - can create networks
        if ! docker network inspect "$network_name" >/dev/null 2>&1; then
            echo "Creating Docker network: $network_name"
            if [[ -n "$network_opts" ]]; then
                docker network create --driver "$network_driver" $network_opts "$network_name"
            else
                docker network create --driver "$network_driver" "$network_name"
            fi
        else
            local current_driver=$(docker network inspect "$network_name" --format '{{.Driver}}')
            # For community edition, bridge network is acceptable
            if [[ "${KAMIWAZA_COMMUNITY:-}" != "true" && "$current_driver" != "$network_driver" ]]; then
                echo "Network $network_name exists but isn't properly configured"
                return 1
            fi
            if [[ "$network_driver" == "overlay" ]]; then
                local is_attachable=$(docker network inspect "$network_name" --format '{{.Attachable}}')
                if [[ "$is_attachable" != "true" ]]; then
                    echo "Network $network_name exists but isn't properly configured"
                    return 1
                fi
            fi
        fi
    else
        # Worker node - wait for network to be available
        local max_attempts=30
        local attempt=1
        while [[ $attempt -le $max_attempts ]]; do
            if docker network inspect "$network_name" >/dev/null 2>&1; then
                echo "Network $network_name available"
                return 0
            fi
            echo "Waiting for network $network_name to be available (attempt $attempt/$max_attempts)..."
            sleep 10
            ((attempt++))
        done
        if [[ $attempt -gt $max_attempts ]]; then
            echo "Timeout waiting for network $network_name"
            return 1
        fi
    fi
}

recreate_dummy_service() {
    # Skip dummy service for community edition
    if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
        return 0
    fi

    local service_name="dummy-network-propagation"
    local networks=$@

    # Remove the existing service if it exists
    if docker service ls --filter "name=$service_name" --format '{{.Name}}' | grep -q "$service_name"; then
        echo "Removing existing dummy service: $service_name"
        docker service rm "$service_name"
    fi

    # Create a new dummy service attached to all specified networks
    echo "Creating a new dummy global service for all required networks"
    docker service create \
        --name "$service_name" \
        --mode global \
        $(for network in $networks; do echo "--network $network "; done) \
        --detach=true \
        alpine:latest \
        sh -c "while true; do sleep 3600; done"
}

setup_essential_networks() {
    # Essential networks for Kamiwaza
    local required_networks=(
        "${KAMIWAZA_ENV:-default}_kamiwaza-traefik"
        "${KAMIWAZA_ENV:-default}_kamiwaza-etcd"
        "${KAMIWAZA_ENV:-default}_kamiwaza-backend"
        "${KAMIWAZA_ENV:-default}_kamiwaza-apps"
        "${KAMIWAZA_ENV:-default}_kamiwaza-milvus"
        "${KAMIWAZA_ENV:-default}_kamiwaza-keycloak"
    )

    # Create or verify all required networks
    for network in "${required_networks[@]}"; do
        if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
            create_network_if_not_exists "$network" "bridge"
        else
            create_network_if_not_exists "$network" "overlay" "--opt encrypted=true --attachable"
        fi
    done

    # If we're a manager node and not community edition, recreate the dummy service for all networks
    if [[ "${KAMIWAZA_COMMUNITY:-}" != "true" ]] && docker info | grep "Is Manager:" | grep -q "true"; then
        recreate_dummy_service "${required_networks[@]}"
    fi
}

main() {
    # Check if .kamiwaza_install_community exists in script's directory and set KAMIWAZA_COMMUNITY
    if [[ -f "$(dirname "${BASH_SOURCE[0]}")/.kamiwaza_install_community" ]]; then
        export KAMIWAZA_COMMUNITY=true
    fi
    
    setup_essential_networks
}

# Execute main if script is run directly (not sourced)
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi