#!/bin/bash

# Continue tracking installation progress from install.sh
if [[ -z "${CURRENT_STEP:-}" ]]; then
    # If not set by install.sh, initialize tracking variables
    START_TIME=$(date +%s)
fi

# Check if offline mode is enabled via environment variable or command line argument
# Offline mode skips pip installs and venv recreation, assuming packages are preinstalled
# Can be enabled by:
#   - Passing --offline to install.sh
#   - Passing --offline to setup.sh directly
#   - Setting OFFLINE_MODE=yes environment variable
OFFLINE_MODE="${OFFLINE_MODE:-false}"
for arg in "$@"; do
    if [ "$arg" == "--offline" ]; then
        OFFLINE_MODE="true"
        echo "Offline mode enabled via command line argument"
    fi
done

# Function to detect WSL environment
is_wsl() {
    if uname_out="$(uname -a 2>/dev/null)"; then
        if echo "$uname_out" | grep -q "microsoft-standard-WSL2"; then
            return 0  # true
        fi
    fi
    return 1  # false
}

# Function to update and display progress
update_progress() {
    local step_name="$1"
    local current_time=$(date +%s)
    local elapsed_time=$((current_time - START_TIME))
    local elapsed_minutes=$((elapsed_time / 60))
    local elapsed_seconds=$((elapsed_time % 60))

    echo "┌────────────────────────────────────────────────────────────┐"
    echo "│ $step_name"
    echo "│ Time elapsed: ${elapsed_minutes}m ${elapsed_seconds}s"
    echo "└────────────────────────────────────────────────────────────┘"
}

source common.sh

# Function to print messages in colors
print_in_color() {
    case $1 in
        green)
            echo -e "\033[92m$2\033[0m"
            ;;
        red)
            echo -e "\033[91m$2\033[0m"
            ;;
        yellow)
            echo -e "\033[93m$2\033[0m"
            ;;
        blue)
            echo -e "\033[94m$2\033[0m"
            ;;
        *)
            echo "$2"
            ;;
    esac
}

version_ge() {
    local left="$1"
    local right="$2"
    if [[ -z "$left" || -z "$right" ]]; then
        return 1
    fi
    local earliest
    earliest=$(printf '%s\n%s\n' "$right" "$left" | sort -V | head -n1)
    [[ "$earliest" == "$right" ]]
}

ensure_uv() {
    if command -v uv >/dev/null 2>&1; then
        print_in_color blue "uv detected at $(command -v uv)"
        return
    fi

    print_in_color yellow "uv not found; attempting installation via https://astral.sh/uv/"
    if ! curl -LsSf https://astral.sh/uv/install.sh | sh; then
        print_in_color red "Failed to install uv automatically. Install uv manually and re-run setup.sh."
        exit 1
    fi

    if [ -d "$HOME/.local/bin" ] && [[ ":$PATH:" != *":$HOME/.local/bin:"* ]]; then
        export PATH="$HOME/.local/bin:$PATH"
    fi
    if [ -d "$HOME/Library/Application Support/uv" ] && [[ ":$PATH:" != *":$HOME/Library/Application Support/uv:"* ]]; then
        export PATH="$HOME/Library/Application Support/uv:$PATH"
    fi

    if command -v uv >/dev/null 2>&1; then
        print_in_color green "uv installed at $(command -v uv)"
    else
        print_in_color red "uv installation completed but binary not on PATH. Update PATH or install uv manually."
        exit 1
    fi
}

source common.sh

if [[ -z "${KAMIWAZA_RUN_FROM_INSTALL:-}" ]]; then
    print_in_color red "Don't run directly; run install.sh instead"
    exit 1
fi

update_progress "Installing Python dependencies"

ensure_uv

# Include dev group only when running in dev mode
# Default uv_sync.sh behavior includes dev group, so we only need to set
# KW_UV_NO_DEFAULT=1 to skip it when not in dev mode
if [[ "${KAMIWAZA_DEV_MODE:-false}" == "true" ]]; then
    print_in_color blue "Running uv sync with dev dependencies"
    ./uv_sync.sh
else
    print_in_color blue "Running uv sync with base dependencies"
    KW_UV_NO_DEFAULT=1 ./uv_sync.sh
fi
if [[ $? -ne 0 ]]; then
    print_in_color red "uv sync failed"
    exit 1
fi

update_progress "Setting up notebook environment"

# Create separate notebook-venv with jupyter + kamiwaza-sdk (NOT platform code)
if is_wsl; then
    print_in_color yellow "WSL environment detected. Creating separate notebook-venv..."
else
    print_in_color blue "Creating separate notebook-venv with jupyter + kamiwaza-sdk"
fi

# Create notebook-venv with Python version constraint (Ray requires <3.14)
if [ ! -d "notebook-venv" ]; then
    print_in_color none "Creating notebook virtual environment (uv venv)..."
    if ! uv venv --python ">=3.10,<3.13" notebook-venv; then
        print_in_color red "Failed to create notebook virtual environment"
        exit 1
    fi
else
    print_in_color yellow "Notebook virtual environment already exists at notebook-venv"
fi

# Install notebook dependencies (jupyter packages + kamiwaza-sdk)
# NOTE: notebook-venv should NOT include kamiwaza platform code, only the SDK
print_in_color none "Installing notebook dependencies into notebook-venv..."
notebook_file="notebooks/extra-requirements.txt"
if [ ! -f "$notebook_file" ]; then
    print_in_color red "Notebook requirements file not found: $notebook_file"
    exit 1
fi

if [[ -n "${KW_OFFLINE_WHEELS:-}" ]]; then
    # Offline mode: install from bundled wheels
    if ! UV_NO_INDEX=1 UV_PIP_FIND_LINKS="${KW_OFFLINE_WHEELS}" uv pip install --python notebook-venv/bin/python --find-links "${KW_OFFLINE_WHEELS}" -r "$notebook_file"; then
        print_in_color red "Failed to install notebook dependencies from offline bundle"
        exit 1
    fi
else
    # Online mode: install from curated requirements
    if ! uv pip install --python notebook-venv/bin/python -r "$notebook_file"; then
        print_in_color red "Failed to install notebook dependencies"
        exit 1
    fi
fi

print_in_color green "✓ Created notebook-venv with jupyter + kamiwaza-sdk"

if [[ "$(uname)" == "Darwin" ]]; then
    update_progress "Setting up MLX environment"
    MLX_ENV="mlx-venv"
    if [ ! -d "$MLX_ENV" ]; then
        print_in_color none "Creating MLX virtual environment (uv venv)..."
        if ! uv venv --python ">=3.10,<3.13" "$MLX_ENV"; then
            print_in_color red "Failed to create MLX virtual environment"
            exit 1
        fi
    else
        print_in_color yellow "MLX virtual environment already exists at $MLX_ENV"
    fi

    print_in_color none "Installing MLX dependencies via uv..."
    if [[ -n "${KW_OFFLINE_WHEELS:-}" ]]; then
        if ! UV_NO_INDEX=1 UV_PIP_FIND_LINKS="${KW_OFFLINE_WHEELS}" uv pip install --python "$MLX_ENV/bin/python" --find-links "${KW_OFFLINE_WHEELS}" -r requirements-mlx.txt; then
            print_in_color red "Failed to install MLX requirements"
            exit 1
        fi
    else
        if ! uv pip install --python "$MLX_ENV/bin/python" -r requirements-mlx.txt; then
            print_in_color red "Failed to install MLX requirements"
            exit 1
        fi
    fi
else
    print_in_color yellow "Skipping MLX environment setup as it's only supported on macOS."
fi


update_progress "Installing Kamiwaza package"

tmp_dir=$(mktemp -d tmp_XXXXXX)
pushd "$tmp_dir" > /dev/null

echo "pwd: `pwd`"
if ! [ -f '../compile.sh' ] && ! [ -f 'compile.sh' ] && ! kw_py -c "import kamiwaza" &> /dev/null; then
    # Skip kamiwaza package installation in WSL since virtual environments will be preinstalled
    if is_wsl; then
        print_in_color yellow "WSL environment detected. Skipping kamiwaza package installation as virtual environments will be preinstalled."
        popd > /dev/null
        rm -rf "$tmp_dir"
    else
        print_in_color yellow "The kamiwaza package is not installed - will install"
        popd > /dev/null
        rm -rf "$tmp_dir"
        # Prefer non-cache wheel files, fallback to any wheel if none found
        kamiwaza_wheel=$(find . -name 'kamiwaza*.whl' -not -path '*/.uv-cache/*' | head -n 1)
        if [[ -z "$kamiwaza_wheel" ]]; then
            # Fallback to any wheel if no non-cache one found
            kamiwaza_wheel=$(find . -name 'kamiwaza*.whl' | head -n 1)
        fi
        if [[ -z "$kamiwaza_wheel" ]]; then
            print_in_color red "kamiwaza wheel file not found. Please ensure it's available and rerun this installer."
            exit 1
        else
            print_in_color blue "Installing kamiwaza package from wheel file into main .venv: $kamiwaza_wheel"
            if ! uv --project "${KAMIWAZA_ROOT:-.}" pip install "$kamiwaza_wheel"; then
                print_in_color red "Failed to install kamiwaza package."
                exit 1
            else
                print_in_color green "Installed into main .venv. Retesting."
            fi
            
            # Also install into notebook-venv if it exists
            if [ -d "notebook-venv" ]; then
                print_in_color blue "Installing kamiwaza package from wheel file into notebook-venv: $kamiwaza_wheel"
                if ! uv pip install --python notebook-venv/bin/python "$kamiwaza_wheel"; then
                    print_in_color red "Failed to install kamiwaza package into notebook-venv."
                    exit 1
                else
                    print_in_color green "Installed into notebook-venv."
                fi
            fi
            
            tmp_dir=$(mktemp -d tmp_XXXXXX)
            pushd "$tmp_dir" > /dev/null
            if ! kw_py -c "import kamiwaza" &> /dev/null; then
                print_in_color red "Failed to verify the installation of kamiwaza package."
                popd > /dev/null
                rm -rf "$tmp_dir"
                exit 1
            else
                popd > /dev/null
                rm -rf "$tmp_dir"
                print_in_color green "kamiwaza package was successfully installed."
            fi
        fi
    fi
else
    popd > /dev/null
    rm -rf "$tmp_dir"
    print_in_color green "kamiwaza package is installed."
fi

# Deactivate the virtual environment - with Kamiwaza installed this should no longer be needed
# and having it alive can interfere with other installs (eg, nvm setup)
safe_deactivate

update_progress "Checking prerequisites"

print_in_color none "Checking for prerequisites..."
# Test if 'cockroach' command is available in PATH (not required for lite mode which uses SQLite)
if [[ "${KAMIWAZA_LITE:-false}" != "true" ]]; then
    if ! command -v cockroach &> /dev/null; then
        print_in_color red "CockroachDB command line tool 'cockroach' could not be found."
        if [[ "$(uname)" == "Darwin" ]]; then
            echo "If you are on OSX, you can install CockroachDB using Homebrew with the command: brew install cockroach"
        else
            echo "Please install CockroachDB from https://www.cockroachlabs.com/ and ensure 'cockroach' is in your PATH."
        fi
        exit 1
    else
        print_in_color green "CockroachDB command line tool 'cockroach' is available."
    fi
else
    print_in_color yellow "Skipping CockroachDB check (lite mode uses SQLite)"
fi

# Test if 'jq' command is available in PATH
if ! command -v jq &> /dev/null; then
    print_in_color red "jq command line tool could not be found."
    if [[ "$(uname)" == "Darwin" ]]; then
        echo "If you are on OSX, you can install jq using Homebrew with the command: brew install jq"
    elif [[ "$(uname)" == "Linux" ]]; then
        echo "On Linux, install with: sudo apt install -y jq"
    else
        echo "Please install jq from your package manager and ensure 'jq' is in your PATH."
    fi
    exit 1
else
    print_in_color green "jq command line tool is available."
fi

# Test if 'cfssl' command is available in PATH
if ! command -v cfssl &> /dev/null; then
    print_in_color red "CFSSL command line tool 'cfssl' could not be found."
    if [[ "$(uname)" == "Darwin" ]]; then
        echo "If you are on OSX, you can install CFSSL using Homebrew with the command: brew install cfssl"
    elif [[ "$(uname)" == "Linux" ]]; then
        echo "On Linux, the installer should have installed, but:"
        echo "sudo apt install -y python3.10 python3.10-dev libpython3.10-dev python3.10-venv golang-cfssl python-is-python3 etcd-client net-tools"
    else
        echo "Please install CFSSL from https://cfssl.org/ and ensure 'cfssl' is in your PATH."
    fi
    exit 1
else
    print_in_color green "CFSSL command line tool 'cfssl' is available."
fi


update_progress "Checking etcd and other tools"

# Test if 'etcd-client' command is available in PATH
if ! command -v etcdctl &> /dev/null; then
    print_in_color red "etcd-client command line tool 'etcdctl' could not be found."
    if [[ "$(uname)" == "Darwin" ]]; then
        echo "If you are on OSX, you can install etcd-client using Homebrew with the command: brew install etcd"
    elif [[ "$(uname)" == "Linux" ]]; then
        echo "On Linux, the installer should have installed, but:"
        echo "sudo apt install -y etcd-client"
    else
        echo "Please install etcd-client from https://etcd.io/ and ensure 'etcdctl' is in your PATH."
    fi
    exit 1
else
    etcdctl_version=$(etcdctl version 2>/dev/null | awk '/etcdctl version:/ {print $3}')
    if [[ -z "$etcdctl_version" ]]; then
        print_in_color yellow "Unable to determine etcdctl version. Assuming it needs an update."
        etcdctl_version="0.0.0"  # Set to a version that will trigger the update
    fi

    if ! version_ge "$etcdctl_version" "3.5"; then
        print_in_color yellow "etcd-client version is less than 3.5 or unknown. Attempting to update..."
        if [[ "$(uname)" == "Darwin" ]]; then
            print_in_color yellow 'Run `brew upgrade etcd` to update etcd on macOS.'
            exit 1
        else
            print_in_color yellow 'Running install-or-update-etcd.sh...'
            source install-or-update-etcd.sh
            # Re-test etcdctl version after update
            if ! command -v etcdctl &> /dev/null; then
                print_in_color red "Failed to install etcdctl. Please install manually."
                exit 1
            fi
            etcdctl_version=$(etcdctl version 2>/dev/null | awk '/etcdctl version:/ {print $3}')
            if [[ -z "$etcdctl_version" ]] || ! version_ge "$etcdctl_version" "3.5"; then
                print_in_color red "Failed to update etcd-client to version 3.5 or higher, or unable to determine version. Please update manually."
                exit 1
            fi
        fi
    fi

    if [[ -z "$etcdctl_version" ]]; then
        print_in_color yellow "etcd-client command line tool 'etcdctl' is available, but unable to determine its version."
    else
        print_in_color green "etcd-client command line tool 'etcdctl' is available and version is $etcdctl_version."
    fi
fi

# Test if 'netstat' command is available in PATH
if ! command -v netstat &> /dev/null; then
    print_in_color red "netstat command line tool 'netstat' could not be found."
    if [[ "$(uname)" == "Darwin" ]]; then
        echo "If you are on OSX, you can install netstat using Homebrew with the command: brew install net-tools"
    elif [[ "$(uname)" == "Linux" ]]; then
        echo "On Linux, the installer should have installed, but:"
        echo "sudo apt install -y net-tools"
    else
        echo "Please install netstat from your package manager and ensure 'netstat' is in your PATH."
    fi
    echo "netstat is optional, so proceeding"
else
    print_in_color green "netstat command line tool 'netstat' is available."
fi

# Test if 'openssl' command is available
if ! command -v openssl &> /dev/null; then
    print_in_color red "openssl command line tool 'openssl' could not be found."
    read -p "Press Enter to continue"
else
    print_in_color green "openssl command line tool 'openssl' is available."
fi

# Set PYTHON_BIN to python3.12 if available, else python3.10, else error
if command -v python3.12 &> /dev/null; then
    PYTHON_BIN="python3.12"
    print_in_color green "Python 3.12 is available: $($PYTHON_BIN --version)"
elif command -v python3.10 &> /dev/null; then
    PYTHON_BIN="python3.10"
    print_in_color green "Python 3.10 is available: $($PYTHON_BIN --version)"
else
    print_in_color red "Python 3.12 or 3.10 is required but not found."
    if [[ "$(uname)" == "Darwin" ]]; then
        echo "If you are on OSX, you can install Python 3.12 or 3.10 using Homebrew with the command: brew install python@3.12"
    elif [[ "$(uname)" == "Linux" ]]; then
        echo "On Linux, the installer should have installed, but:"
        echo "sudo apt install -y python3.12 python3.12-dev libpython3.12-dev python3.12-venv || sudo apt install -y python3.10 python3.10-dev libpython3.10-dev python3.10-venv"
    else
        echo "Please install Python 3.12 or 3.10 from https://www.python.org/ and ensure it's in your PATH."
    fi
    exit 1
fi

update_progress "Checking Docker configuration"

print_in_color none "Checking for Docker installation and permissions..."

# Check if Docker is installed
if ! command -v docker &> /dev/null; then
    # If docker not found in PATH, try common locations
    # This handles the case where kamiwaza user doesn't have /usr/local/bin in PATH during RPM install
    if [ -x /usr/local/bin/docker ]; then
        export PATH="/usr/local/bin:$PATH"
        print_in_color yellow "Docker found in /usr/local/bin, added to PATH"
    elif [ -x /usr/bin/docker ]; then
        export PATH="/usr/bin:$PATH"
    else
        print_in_color red "Docker could not be found. Please install Docker and try again."
        exit 1
    fi
fi

if command -v docker &> /dev/null; then
    print_in_color green "Docker is installed."
else
    print_in_color red "Docker could not be found. Please install Docker and try again."
    exit 1
fi

# Check if 'docker compose' command is available
if ! docker compose version &> /dev/null; then
    print_in_color red "'docker compose' command is not available. Please ensure you have Docker Compose v2 and try again."
    exit 1
else
    print_in_color green "'docker compose' command is available."
fi

# Test if the user has access to the Docker daemon
# Use 'sg docker -c' to activate docker group membership if available
# This handles the case where user was just added to docker group and hasn't logged out/in
# Note: Use getent instead of 'groups' because 'groups' only shows current session groups
# Note: sg and getent are Linux-only; on macOS this falls through to plain docker info
if command -v sg &> /dev/null && command -v getent &> /dev/null && getent group docker | grep -qw "$(whoami)"; then
    DOCKER_CMD="sg docker -c 'docker info'"
else
    DOCKER_CMD="docker info"
fi

if ! eval $DOCKER_CMD &> /dev/null; then
    # Check if running on WSL
    if grep -qi microsoft /proc/version 2>/dev/null; then
        print_in_color yellow "Docker daemon not accessible. Checking Docker installation in WSL..."
        
        # Check if dockerd command exists before attempting to run it
        if command -v dockerd &> /dev/null; then
            print_in_color yellow "Attempting to start Docker daemon for WSL..."
            sudo dockerd > /dev/null 2>&1 &
            sleep 3
            if eval $DOCKER_CMD &> /dev/null; then
                print_in_color green "Docker daemon started successfully for WSL."
            else
                print_in_color red "Failed to start Docker daemon. Please check Docker installation."
                exit 1
            fi
        else
            print_in_color red "Docker daemon (dockerd) not found. Please ensure Docker is properly installed in WSL."
            print_in_color yellow "For WSL2, you may need to:"
            print_in_color yellow "1. Install Docker Desktop with WSL2 backend enabled, or"
            print_in_color yellow "2. Install Docker CE in WSL2 directly"
            exit 1
        fi
    else
        print_in_color red "You do not have access to the Docker daemon. Please ensure your user is added to the 'docker' group and try again."
        USERINFO=$(id || true)
        CUR_DOCKER_PATH=$(command -v docker || true)
        if [ ! -z "$CUR_DOCKER_PATH" ]; then
            DOCKER_BIN_LISTING=$(ls -lad "$CUR_DOCKER_PATH" 2>/dev/null || true)
        fi
        print_in_color yellow "Your `id` info: ${USERINFO:-}"
        print_in_color yellow "Your docker binary path: ${CUR_DOCKER_PATH:-}"
        print_in_color yellow "Your docker binary listing: ${DOCKER_BIN_LISTING:-}"
        exit 1
    fi
else
    print_in_color green "You have access to the Docker daemon."
fi

update_progress "License verification"

print_in_color green "***********************************************************"
print_in_color green "==== Kamiwaza Installer ===="
print_in_color yellow "Your use of this software is subject to the license agreement. If you do not agree to the license terms, say no, exit this installer, and delete all copies of the software"

if [[ "${USER_ACCEPTED_KAMIWAZA_LICENSE:-}" == "yes" ]]; then
    print_in_color green "License agreement accepted via CLI flag. Continuing with installation..."
else
    if [ ! -f LICENSE-kamiwaza ] ; then
        print_in_color red "License file not found. Contact support@kamiwaza.ai"
        exit 1
    fi
    bash read_eula.sh
    if [ $? -ne 0 ]; then
        print_in_color red "You did not agree to the license terms. Exiting installer."
        exit 1
    else
        print_in_color green "You have agreed to the license terms. Continuing with installation..."
    fi
fi

update_progress "Finalizing notebook environment setup"
print_in_color green "✓ Notebook environment (notebook-venv) is ready with all dependencies"

if [[ "$(uname)" == "Darwin" ]]; then
    update_progress "Setting up MLX environment"
    print_in_color none "MLX dependencies installed via uv extras; no separate virtual environment created."
else
    print_in_color yellow "Skipping MLX-specific dependencies (non-macOS)."
fi

if [[ "$(uname)" != "Darwin" ]]; then
    print_in_color none "Checking for /opt/kamiwaza directory..."
    if [ -d "/opt/kamiwaza" ]; then
        setup_docker_root "/opt/kamiwaza/containers"
    else
        print_in_color yellow "/opt/kamiwaza directory not found. Skipping Docker root configuration."
    fi
fi

# Verify environment and cluster configuration
verify_cluster_config() {
    if [[ -f "/etc/kamiwaza/config/is_worker" ]]; then
        if [[ -z "${KAMIWAZA_HEAD_IP:-}" ]]; then
            print_in_color red "Worker node detected but KAMIWAZA_HEAD_IP not set"
            exit 1
        fi
    fi
}

verify_cluster_config

update_progress "Installing frontend and components"

# llama.cpp is now installed by default
# Use --skip_llamacpp flag or KAMIWAZA_SKIP_LLAMACPP=1 env var to opt-out
install_llamacpp="yes"

# Check environment variable first (useful when setup.sh is sourced)
if [[ "${KAMIWAZA_SKIP_LLAMACPP:-0}" == "1" ]]; then
    install_llamacpp="no"
fi

# Then check command line arguments
for arg in "$@"; do
    if [ "$arg" == "--skip_llamacpp" ]; then
        install_llamacpp="no"
        break
    fi
done


if [ -f llamacpp.commit ] ; then
    if [[ "$install_llamacpp" == "yes" ]]; then
        print_in_color green "### Installing llamacpp..."
        if [[ "$OFFLINE_MODE" == "true" ]]; then
            bash build-llama-cpp.sh --offline
        else
            bash build-llama-cpp.sh
        fi
    else
        print_in_color yellow "Skipping local llamacpp installation."
    fi
fi


update_progress "Finalizing installation"
print_in_color none "### Running Kamiwaza install.py via uv..."
kw_py install.py

# Import Docker images in offline mode
if [[ "$OFFLINE_MODE" == "true" ]]; then
    if [ -d "/usr/share/kamiwaza/kamiwaza-extension-registry/garden/default/docker-images" ]; then
        print_in_color yellow "Offline mode: Importing Docker images..."
        cd /usr/share/kamiwaza/kamiwaza-extension-registry/garden/default/docker-images
        if [ -f "./import-images.sh" ]; then
            bash ./import-images.sh
            if [ $? -eq 0 ]; then
                print_in_color green "Docker images imported successfully."
            else
                print_in_color yellow "Warning: Docker image import failed, but continuing..."
            fi
        else
            print_in_color yellow "Warning: import-images.sh not found in Docker images directory."
        fi
        cd - > /dev/null
    else
        print_in_color yellow "Warning: Docker images directory not found in offline mode."
    fi
fi


if [ $? -ne 0 ]; then
    print_in_color red "Kamiwaza install.py failed. Exiting."
    exit 1
else
    print_in_color green "Kamiwaza install.py completed successfully."
    print_in_color green "### Kamiwaza is installed. ###"
    print_in_color green "### Startup with: bash startup/kamiwazad.sh start ###"
    touch ~/.kamiwaza-installed
fi

# Display final installation status for this phase
FINAL_TIME=$(date +%s)
TOTAL_ELAPSED_TIME=$((FINAL_TIME - START_TIME))
TOTAL_MINUTES=$((TOTAL_ELAPSED_TIME / 60))
TOTAL_SECONDS=$((TOTAL_ELAPSED_TIME % 60))

echo "┌────────────────────────────────────────────────────────────┐"
echo "│ Setup Phase Complete!                                       │"
echo "│ Total time: ${TOTAL_MINUTES}m ${TOTAL_SECONDS}s                                    │"
echo "└────────────────────────────────────────────────────────────┘"

