#!/usr/bin/env python3
"""
Create or update the ForwardAuth health-check service account in Keycloak.

This script:
1. Ensures a confidential client (default: ``kamiwaza-svc``) exists in the
   target realm with service accounts enabled and interactive flows disabled.
2. (Re)generates the client secret and stores it under ``runtime/secrets`` so it
   can be injected into the Kamiwaza environment (e.g. KAMIWAZA_SERVICE_ACCOUNT_PASSWORD).
3. Grants the configured realm role to the client's service account user so
   tokens minted via ``client_credentials`` satisfy ForwardAuth RBAC checks.

Run via ``scripts/kw_py -m scripts.setup_forwardauth_service_account`` after the
cluster is up. Configure behavior through CLI flags or environment variables.
"""

from __future__ import annotations

import argparse
import asyncio
import os
import sys
from pathlib import Path
from typing import Any, Dict, Optional, cast

import httpx

def _get_keycloak_admin_url() -> str | None:
    admin_env = os.getenv("AUTH_GATEWAY_KEYCLOAK_ADMIN_URL")
    if admin_env:
        return admin_env
    kc_url = os.getenv("AUTH_GATEWAY_KEYCLOAK_URL")
    if not kc_url:
        return None
    base = kc_url.rstrip("/")
    if base.startswith("https://"):
        return f"{base}/_kc_admin"
    return f"{base}/admin"

def _load_env_file(path: Path) -> None:
    """Best-effort parser for simple ``export KEY=value`` entries."""
    try:
        content = path.read_text(encoding="utf-8")
    except FileNotFoundError:
        return

    for raw_line in content.splitlines():
        line = raw_line.strip()
        if not line or line.startswith("#"):
            continue
        if line.startswith("export "):
            line = line[len("export ") :].strip()
        if "=" not in line:
            continue
        key, value = line.split("=", 1)
        key = key.strip()
        value = value.strip().strip('"').strip("'")
        if key and key not in os.environ:
            os.environ[key] = value


def _bootstrap_env_from_files() -> None:
    """Mirror launcher behavior by sourcing env files before running."""
    candidates: list[Path] = []
    env_hint = os.getenv("KAMIWAZA_ENV_FILE_PATH")
    if env_hint:
        candidates.append(Path(env_hint))
    candidates.append(Path("/etc/kamiwaza/env.sh"))

    root_env = os.getenv("KAMIWAZA_ROOT")
    root_path = Path(root_env) if root_env else Path(__file__).resolve().parents[1]
    os.environ.setdefault("KAMIWAZA_ROOT", str(root_path))
    candidates.append(root_path / "env.sh")

    seen: set[Path] = set()
    for candidate in candidates:
        if candidate and candidate not in seen and candidate.exists():
            _load_env_file(candidate)
            seen.add(candidate)


def _env_bool(name: str, default: bool = False) -> bool:
    raw = os.getenv(name)
    if raw is None:
        return default
    return raw.strip().lower() in {"1", "true", "yes", "on"}


def _load_admin_password() -> Optional[str]:
    password = os.getenv("KEYCLOAK_ADMIN_PASSWORD")
    if password:
        return password

    root = os.getenv("KAMIWAZA_ROOT")
    if root:
        secret_path = Path(root) / "runtime" / "secrets" / "keycloak-admin-password"
        if secret_path.exists():
            return secret_path.read_text(encoding="utf-8").strip()
    return None


async def _fetch_admin_token(
    client: httpx.AsyncClient, base_url: str, username: str, password: str
) -> str:
    token_url = f"{base_url}/realms/master/protocol/openid-connect/token"
    resp = await client.post(
        token_url,
        data={
            "grant_type": "password",
            "client_id": "admin-cli",
            "username": username,
            "password": password,
        },
    )
    resp.raise_for_status()
    payload = resp.json()
    if not isinstance(payload, dict):
        raise RuntimeError("Admin token response was not a JSON object")
    token = payload.get("access_token")
    if not isinstance(token, str) or not token:
        raise RuntimeError("Admin token response missing access_token")
    return token


async def _get_client(
    client: httpx.AsyncClient,
    admin_base_url: str,
    realm: str,
    client_id: str,
    headers: Dict[str, str],
) -> Optional[Dict[str, Any]]:
    resp = await client.get(
        f"{admin_base_url}/realms/{realm}/clients",
        params={"clientId": client_id},
        headers=headers,
    )
    resp.raise_for_status()
    data = resp.json()
    if not isinstance(data, list) or not data:
        return None
    client_uuid = data[0].get("id") if isinstance(data[0], dict) else None
    if not isinstance(client_uuid, str) or not client_uuid:
        raise RuntimeError("Keycloak client lookup returned an unexpected payload")
    detail = await client.get(
        f"{admin_base_url}/realms/{realm}/clients/{client_uuid}", headers=headers
    )
    detail.raise_for_status()
    detail_payload = detail.json()
    if not isinstance(detail_payload, dict):
        raise RuntimeError("Keycloak client detail response was not a JSON object")
    return cast(Dict[str, Any], detail_payload)


def _desired_client_shape(
    client_id: str, display_name: Optional[str]
) -> Dict[str, Any]:
    return {
        "clientId": client_id,
        "name": display_name or client_id,
        "enabled": True,
        "protocol": "openid-connect",
        "publicClient": False,
        "serviceAccountsEnabled": True,
        "standardFlowEnabled": False,
        "implicitFlowEnabled": False,
        "directAccessGrantsEnabled": False,
        "authorizationServicesEnabled": False,
        "clientAuthenticatorType": "client-secret",
        "bearerOnly": False,
    }


async def _create_client(
    client: httpx.AsyncClient,
    admin_base_url: str,
    realm: str,
    client_id: str,
    display_name: Optional[str],
    headers: Dict[str, str],
) -> Dict[str, Any]:
    payload = _desired_client_shape(client_id, display_name)
    resp = await client.post(
        f"{admin_base_url}/realms/{realm}/clients",
        json=payload,
        headers=headers,
    )
    resp.raise_for_status()
    created = await _get_client(client, admin_base_url, realm, client_id, headers)  # noqa
    if created is None:
        raise RuntimeError("Client creation reported success but lookup failed")
    return created


async def _update_client(
    client: httpx.AsyncClient,
    admin_base_url: str,
    realm: str,
    current: Dict[str, Any],
    desired: Dict[str, Any],
    headers: Dict[str, str],
) -> Dict[str, Any]:
    needs_update = False
    for key, desired_value in desired.items():
        if current.get(key) != desired_value:
            current[key] = desired_value
            needs_update = True

    if needs_update:
        resp = await client.put(
            f"{admin_base_url}/realms/{realm}/clients/{current['id']}",
            json=current,
            headers=headers,
        )
        resp.raise_for_status()
    return current


async def _regenerate_secret(
    client: httpx.AsyncClient,
    admin_base_url: str,
    realm: str,
    client_uuid: str,
    headers: Dict[str, str],
) -> str:
    resp = await client.post(
        f"{admin_base_url}/realms/{realm}/clients/{client_uuid}/client-secret",
        headers=headers,
    )
    resp.raise_for_status()
    payload = resp.json()
    if not isinstance(payload, dict):
        raise RuntimeError("Keycloak did not return a JSON object for client secret")
    secret = payload.get("value")
    if not isinstance(secret, str) or not secret:
        raise RuntimeError("Keycloak did not return a client secret")
    return secret


async def _ensure_role_mapping(
    client: httpx.AsyncClient,
    admin_base_url: str,
    realm: str,
    client_uuid: str,
    role_name: str,
    headers: Dict[str, str],
) -> None:
    svc_resp = await client.get(
        f"{admin_base_url}/realms/{realm}/clients/{client_uuid}/service-account-user",
        headers=headers,
    )
    svc_resp.raise_for_status()
    service_user = svc_resp.json()
    service_user_id = service_user["id"]

    current_roles_resp = await client.get(
        f"{admin_base_url}/realms/{realm}/users/{service_user_id}/role-mappings/realm",
        headers=headers,
    )
    current_roles_resp.raise_for_status()
    if any(role.get("name") == role_name for role in current_roles_resp.json() or []):
        return

    role_resp = await client.get(
        f"{admin_base_url}/realms/{realm}/roles/{role_name}", headers=headers
    )
    if role_resp.status_code == 404:
        raise RuntimeError(f"Realm role '{role_name}' does not exist")
    role_resp.raise_for_status()
    role_repr = role_resp.json()

    assign_resp = await client.post(
        f"{admin_base_url}/realms/{realm}/users/{service_user_id}/role-mappings/realm",  # noqa
        json=[role_repr],
        headers=headers,
    )
    assign_resp.raise_for_status()


def _write_secret(secret: str, path: Path) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    path.write_text(secret + "\n", encoding="utf-8")
    os.chmod(path, 0o600)


async def _run(args: argparse.Namespace) -> None:
    base_url = args.base_url.rstrip("/")
    admin_base_url = args.admin_base_url or _get_keycloak_admin_url()
    if admin_base_url:
        admin_base_url = admin_base_url.rstrip("/")
    else:
        if base_url.startswith("https://"):
            raise SystemExit(
                "AUTH_GATEWAY_KEYCLOAK_ADMIN_URL is required when using https://<host>; admin REST is not exposed under /admin. "
                "Set AUTH_GATEWAY_KEYCLOAK_ADMIN_URL=https://<host>/_kc_admin or a valid admin prefix."
            )
        admin_base_url = f"{base_url}/admin"
    realm = args.realm
    verify_tls = not args.insecure_tls
    admin_username = args.admin_user
    admin_password = args.admin_password or _load_admin_password()

    if not admin_password:
        raise SystemExit(
            "KEYCLOAK_ADMIN_PASSWORD is not set and runtime secret was not found."
        )

    async with httpx.AsyncClient(verify=verify_tls, timeout=20.0) as client:
        token = await _fetch_admin_token(
            client, base_url, admin_username, admin_password
        )
        headers = {
            "Authorization": f"Bearer {token}",
            "Content-Type": "application/json",
        }

        existing = await _get_client(
            client, admin_base_url, realm, args.client_id, headers
        )
        desired = _desired_client_shape(args.client_id, args.client_name)

        if existing is None:
            client_repr = await _create_client(
                client,
                admin_base_url,
                realm,
                args.client_id,
                args.client_name,
                headers,
            )
            created = True
        else:
            client_repr = await _update_client(
                client,
                admin_base_url,
                realm,
                existing,
                desired,
                headers,
            )
            created = False

        secret = await _regenerate_secret(
            client, admin_base_url, realm, client_repr["id"], headers
        )
        _write_secret(secret, args.secret_path)

        await _ensure_role_mapping(
            client, admin_base_url, realm, client_repr["id"], args.role, headers
        )

    action = "created" if created else "updated"
    print(
        f"[forwardauth] {action} client '{args.client_id}' in realm '{realm}', "
        f"secret written to {args.secret_path}"
    )
    print(
        "No env vars required: serving components read the secret directly from "
        f"{args.secret_path}"
    )


def _default_secret_path(client_id: str) -> Path:
    root = os.getenv("KAMIWAZA_ROOT", os.getcwd())
    return Path(root) / "runtime" / "secrets" / f"{client_id}-secret"


def _build_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        description="Provision the ForwardAuth health-check service account in Keycloak."
    )
    parser.add_argument(
        "--base-url",
        default=os.getenv("AUTH_GATEWAY_KEYCLOAK_URL", "http://localhost:8080"),
        help="Keycloak public base URL (default: %(default)s)",
    )
    parser.add_argument(
        "--admin-base-url",
        default=os.getenv("AUTH_GATEWAY_KEYCLOAK_ADMIN_URL"),
        help="Keycloak admin REST base URL; defaults to <base-url>/admin if unset",
    )
    parser.add_argument(
        "--realm",
        default=os.getenv("AUTH_GATEWAY_KEYCLOAK_REALM", "kamiwaza"),
        help="Target realm for the service account (default: %(default)s)",
    )
    parser.add_argument(
        "--client-id",
        default=os.getenv("KAMIWAZA_SERVICE_ACCOUNT_ID", "kamiwaza-svc"),
        help="OIDC client ID to create/update (default: %(default)s)",
    )
    parser.add_argument(
        "--client-name",
        default="ForwardAuth Probe",
        help="Display name for the client (default: %(default)s)",
    )
    parser.add_argument(
        "--role",
        default=os.getenv("KAMIWAZA_SERVICE_ACCOUNT_ROLE", "user"),
        help="Realm role to assign to the service account (default: %(default)s)",
    )
    parser.add_argument(
        "--secret-path",
        type=Path,
        default=None,
        help="Where to store the generated client secret (default: runtime/secrets/<client-id>-secret)",
    )
    parser.add_argument(
        "--admin-user",
        default=os.getenv("KEYCLOAK_ADMIN", "admin"),
        help="Keycloak bootstrap admin username (default: %(default)s)",
    )
    parser.add_argument(
        "--admin-password",
        default=None,
        help="Keycloak bootstrap admin password (defaults to KEYCLOAK_ADMIN_PASSWORD or runtime secret)",
    )
    parser.add_argument(
        "--insecure-tls",
        action="store_true",
        default=_env_bool("AUTH_GATEWAY_TLS_INSECURE", False),
        help="Disable TLS verification when talking to Keycloak (default: %(default)s)",
    )
    return parser


def main() -> None:
    _bootstrap_env_from_files()
    parser = _build_parser()
    args = parser.parse_args()
    if args.insecure_tls:
        print(
            "[WARN] TLS verification disabled for Keycloak bootstrap. Use only in dev environments.",
            file=sys.stderr,
        )
    if args.secret_path is None:
        args.secret_path = _default_secret_path(args.client_id)
    try:
        asyncio.run(_run(args))
    except KeyboardInterrupt:
        sys.exit(130)


if __name__ == "__main__":
    main()
