#!/usr/bin/env python3
"""Seed dev-only users into Keycloak.

This script exists to support local developer workflows without shipping
plaintext default passwords in the checked-in Keycloak realm import JSON.

It can seed the default convenience users or a single custom user and writes
passwords under ``runtime/secrets/``. Password resolution precedence:
``--password`` > ``${USERNAME}_PASSWORD`` > generated.

Usage examples:

  scripts/kw_py scripts/seed_keycloak_users.py
  scripts/kw_py scripts/seed_keycloak_users.py --username demo --roles admin,user --password Passw0rd!

Environment:
  - AUTH_GATEWAY_KEYCLOAK_URL (default: http://localhost:8080) — public/OIDC base
  - AUTH_GATEWAY_KEYCLOAK_ADMIN_URL (default: <base-url>/admin) — admin REST base (or https://<host>/_kc_admin via Traefik)
  - AUTH_GATEWAY_KEYCLOAK_REALM (default: kamiwaza)
  - KEYCLOAK_ADMIN / KEYCLOAK_ADMIN_PASSWORD (or runtime secret)
"""

from __future__ import annotations

import argparse
import os
import secrets
import string
from dataclasses import dataclass
from pathlib import Path
from typing import Iterable, Mapping, Sequence

import httpx


@dataclass(frozen=True)
class DevUser:
    username: str
    email: str
    first_name: str
    last_name: str
    realm_roles: tuple[str, ...] = ()
    client_roles: Mapping[str, Sequence[str]] | None = None


DEFAULT_USERS: tuple[DevUser, ...] = (
    DevUser(
        username="admin",
        email="admin@kamiwaza.local",
        first_name="Kamiwaza",
        last_name="Admin",
        realm_roles=("admin", "user"),
    ),
    DevUser(
        username="testuser",
        email="testuser@kamiwaza.local",
        first_name="Test",
        last_name="User",
        realm_roles=("viewer",),
    ),
    DevUser(
        username="testadmin",
        email="testadmin@kamiwaza.local",
        first_name="Test",
        last_name="Admin",
        realm_roles=("admin", "user"),
    ),
)


def _env_bool(name: str, default: bool = False) -> bool:
    raw = os.getenv(name)
    if raw is None:
        return default
    return raw.strip().lower() in {"1", "true", "yes", "on"}


def _load_keycloak_admin_password() -> str:
    password = os.getenv("KEYCLOAK_ADMIN_PASSWORD")
    if password:
        return password

    root = os.getenv("KAMIWAZA_ROOT")
    if root:
        secret_path = Path(root) / "runtime" / "secrets" / "keycloak-admin-password"
        if secret_path.exists():
            return secret_path.read_text(encoding="utf-8").strip()

    raise SystemExit(
        "KEYCLOAK_ADMIN_PASSWORD not set and runtime/secrets/keycloak-admin-password not found"
    )


def _generate_password(length: int) -> str:
    alphabet = string.ascii_letters + string.digits
    return "".join(secrets.choice(alphabet) for _ in range(length))


def _runtime_secrets_dir() -> Path:
    root = os.getenv("KAMIWAZA_ROOT")
    if not root:
        root = str(Path(__file__).resolve().parents[1])
        os.environ["KAMIWAZA_ROOT"] = root
    return Path(root) / "runtime" / "secrets"


def _write_secret(path: Path, value: str) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    path.write_text(value + "\n", encoding="utf-8")
    os.chmod(path, 0o600)


def _token_url(base_url: str) -> str:
    return f"{base_url.rstrip('/')}/realms/master/protocol/openid-connect/token"


def _admin_api_base(admin_base_url: str, realm: str) -> str:
    return f"{admin_base_url.rstrip('/')}/realms/{realm}"


def _user_lookup_url(admin_base_url: str, realm: str) -> str:
    return f"{_admin_api_base(admin_base_url, realm)}/users"


def _get_admin_token(client: httpx.Client, base_url: str, username: str, password: str) -> str:
    resp = client.post(
        _token_url(base_url),
        data={
            "grant_type": "password",
            "client_id": "admin-cli",
            "username": username,
            "password": password,
        },
    )
    resp.raise_for_status()
    payload = resp.json()
    token = payload.get("access_token")
    if not isinstance(token, str) or not token:
        raise RuntimeError("Keycloak admin token response missing access_token")
    return token


def _find_user_id(
    client: httpx.Client, admin_base_url: str, realm: str, headers: Mapping[str, str], username: str
) -> str | None:
    resp = client.get(_user_lookup_url(admin_base_url, realm), params={"username": username}, headers=headers)
    resp.raise_for_status()
    users = resp.json() or []
    for user in users:
        if user.get("username") == username and user.get("id"):
            return str(user["id"])
    return None


def _create_user(
    client: httpx.Client,
    admin_base_url: str,
    realm: str,
    headers: Mapping[str, str],
    user: DevUser,
) -> str:
    payload = {
        "username": user.username,
        "email": user.email,
        "emailVerified": True,
        "enabled": True,
        "firstName": user.first_name,
        "lastName": user.last_name,
    }
    resp = client.post(_user_lookup_url(admin_base_url, realm), headers=headers, json=payload)
    resp.raise_for_status()
    user_id = _find_user_id(client, admin_base_url, realm, headers, user.username)
    if not user_id:
        raise RuntimeError(f"Created user '{user.username}' but failed to resolve user id")
    return user_id


def _set_password(
    client: httpx.Client,
    admin_base_url: str,
    realm: str,
    headers: Mapping[str, str],
    user_id: str,
    password: str,
) -> None:
    resp = client.put(
        f"{_admin_api_base(admin_base_url, realm)}/users/{user_id}/reset-password",
        headers=headers,
        json={"type": "password", "value": password, "temporary": False},
    )
    resp.raise_for_status()


def _get_realm_role(
    client: httpx.Client, admin_base_url: str, realm: str, headers: Mapping[str, str], role_name: str
) -> dict:
    resp = client.get(f"{_admin_api_base(admin_base_url, realm)}/roles/{role_name}", headers=headers)
    resp.raise_for_status()
    role = resp.json()
    if not isinstance(role, dict) or "name" not in role:
        raise RuntimeError(f"Unexpected role payload for '{role_name}'")
    return role


def _assign_realm_roles(
    client: httpx.Client,
    admin_base_url: str,
    realm: str,
    headers: Mapping[str, str],
    user_id: str,
    roles: Iterable[str],
) -> None:
    role_reprs = [_get_realm_role(client, admin_base_url, realm, headers, role_name) for role_name in roles]
    if not role_reprs:
        return
    resp = client.post(
        f"{_admin_api_base(admin_base_url, realm)}/users/{user_id}/role-mappings/realm",
        headers=headers,
        json=role_reprs,
    )
    resp.raise_for_status()


def _parse_roles(raw_roles: str | None) -> tuple[str, ...]:
    if not raw_roles:
        return ()
    return tuple(role.strip() for role in raw_roles.split(",") if role.strip())


def _password_for_user(username: str, explicit_password: str | None, length: int) -> str:
    if explicit_password:
        return explicit_password
    env_var = f"{username.upper()}_PASSWORD"
    env_password = os.getenv(env_var)
    if env_password:
        return env_password
    return _generate_password(length)


def _build_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(description="Seed Keycloak dev users (no plaintext defaults).")
    parser.add_argument(
        "--base-url",
        default=os.getenv("AUTH_GATEWAY_KEYCLOAK_URL", "http://localhost:8080"),
        help="Keycloak public base URL (OIDC/token/JWKS) (default: %(default)s)",
    )
    parser.add_argument(
        "--admin-base-url",
        default=os.getenv("AUTH_GATEWAY_KEYCLOAK_ADMIN_URL"),
        help="Keycloak admin REST base URL; defaults to base-url if unset",
    )
    parser.add_argument(
        "--realm",
        default=os.getenv("AUTH_GATEWAY_KEYCLOAK_REALM", "kamiwaza"),
        help="Target realm to seed users into (default: %(default)s)",
    )
    parser.add_argument(
        "--admin-user",
        default=os.getenv("KEYCLOAK_ADMIN", "admin"),
        help="Keycloak bootstrap admin username (default: %(default)s)",
    )
    parser.add_argument(
        "--password-length",
        type=int,
        default=20,
        help="Generated password length (default: %(default)s)",
    )
    parser.add_argument(
        "--insecure-tls",
        action="store_true",
        default=_env_bool("AUTH_GATEWAY_TLS_INSECURE", False),
        help="Disable TLS verification when talking to Keycloak (default: %(default)s)",
    )
    parser.add_argument("--username", help="Seed only this username (default: seed built-ins)")
    parser.add_argument("--email", help="Email for single-user mode (default: <username>@kamiwaza.local)")
    parser.add_argument("--first-name", dest="first_name", help="First name for single-user mode")
    parser.add_argument("--last-name", dest="last_name", help="Last name for single-user mode")
    parser.add_argument(
        "--roles",
        help="Comma-separated realm roles for single-user mode (default: viewer)",
        default="viewer",
    )
    parser.add_argument(
        "--password",
        help="Optional stable password; overrides ${USERNAME}_PASSWORD fallback",
    )
    return parser


def main(argv: list[str] | None = None) -> int:
    args = _build_parser().parse_args(argv)
    verify_tls = not args.insecure_tls
    if args.admin_base_url:
        admin_base_url = args.admin_base_url.rstrip("/")
    else:
        if args.base_url.startswith("https://"):
            admin_base_url = f"{args.base_url.rstrip('/')}/_kc_admin"
        else:
            admin_base_url = f"{args.base_url.rstrip('/')}/admin"
    admin_password = _load_keycloak_admin_password()

    secrets_dir = _runtime_secrets_dir()
    print(f"[keycloak-seed] secrets dir: {secrets_dir}")

    with httpx.Client(verify=verify_tls, timeout=20.0) as client:
        token = _get_admin_token(client, args.base_url, args.admin_user, admin_password)
        headers = {"Authorization": f"Bearer {token}", "Content-Type": "application/json"}

        if args.username:
            users: Sequence[DevUser] = (
                DevUser(
                    username=args.username,
                    email=args.email or f"{args.username}@kamiwaza.local",
                    first_name=args.first_name or args.username,
                    last_name=args.last_name or "User",
                    realm_roles=_parse_roles(args.roles) or ("viewer",),
                ),
            )
        else:
            users = DEFAULT_USERS

        for user in users:
            user_id = _find_user_id(client, admin_base_url, args.realm, headers, user.username)
            created = False
            if not user_id:
                user_id = _create_user(client, admin_base_url, args.realm, headers, user)
                created = True

            password = _password_for_user(user.username, args.password, args.password_length)
            _set_password(client, admin_base_url, args.realm, headers, user_id, password)
            _assign_realm_roles(client, admin_base_url, args.realm, headers, user_id, user.realm_roles)

            secret_path = secrets_dir / f"keycloak-kz-{user.username}-password"
            _write_secret(secret_path, password)
            action = "created" if created else "updated"
            print(f"[keycloak-seed] {action} user '{user.username}', password written to {secret_path}")

    print("[keycloak-seed] done")
    return 0


if __name__ == "__main__":
    raise SystemExit(main())
