#!/usr/bin/env bash
#
# install_docker_images.sh
#
# Standalone script to install Docker, Docker Compose, and load Docker images
# from a bundled tar.gz archive on a client server.
#
# Usage:
#   1. Copy this script and docker_images_*.tar.gz to the target server
#   2. Run: sudo bash install_docker_images.sh [path/to/docker_images.tar.gz]
#
# If no tar.gz path is provided, it will search for docker_images*.tar.gz in the current directory.

set -euo pipefail

# ============================================================================
# Configuration
# ============================================================================

DOCKER_VERSION="25.0.3"
DOCKER_COMPOSE_VERSION="v2.21.0"
INSTALL_DIR="/usr/local/bin"
DOCKER_CACHE_DIR="/var/tmp/kamiwaza-docker-cache"
APP_USER="${SUDO_USER:-root}"
APP_GROUP="${SUDO_USER:-root}"

# Ensure /usr/local/bin is in PATH for docker commands
export PATH="${INSTALL_DIR}:${PATH}"

# ============================================================================
# Logging Functions
# ============================================================================

log() {
    local level="$1"
    shift
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [${level}] $*" >&2
}

log_info() {
    log "INFO" "$@"
}

log_warn() {
    log "WARN" "$@"
}

log_error() {
    log "ERROR" "$@"
}

fatal() {
    log_error "$@"
    exit 1
}

# ============================================================================
# Architecture Detection
# ============================================================================

detect_architecture() {
    local arch
    arch="$(uname -m)"
    case "${arch}" in
        x86_64)
            echo "x86_64"
            ;;
        aarch64|arm64)
            echo "aarch64"
            ;;
        *)
            fatal "Unsupported architecture: ${arch}"
            ;;
    esac
}

# ============================================================================
# Docker Installation
# ============================================================================

check_docker() {
    if command -v docker &> /dev/null; then
        local docker_version
        docker_version="$(docker --version 2>/dev/null || echo 'unknown')"
        log_info "Docker is already installed: ${docker_version}"
        return 0
    fi
    return 1
}

download_file() {
    local url="$1"
    local dest="$2"
    
    log_info "Downloading ${url} to ${dest}"
    
    if command -v curl &> /dev/null; then
        curl -fsSL -o "${dest}" "${url}"
    elif command -v wget &> /dev/null; then
        wget -q -O "${dest}" "${url}"
    else
        fatal "Neither curl nor wget found. Please install one of them."
    fi
}

install_docker() {
    local arch="$1"
    local docker_url docker_tgz temp_dir
    
    if check_docker; then
        return 0
    fi
    
    log_info "Installing Docker ${DOCKER_VERSION} for ${arch}"
    
    case "${arch}" in
        x86_64)
            docker_url="https://download.docker.com/linux/static/stable/x86_64/docker-${DOCKER_VERSION}.tgz"
            ;;
        aarch64)
            docker_url="https://download.docker.com/linux/static/stable/aarch64/docker-${DOCKER_VERSION}.tgz"
            ;;
        *)
            fatal "Unsupported architecture for Docker download: ${arch}"
            ;;
    esac
    
    temp_dir="$(mktemp -d)"
    docker_tgz="${temp_dir}/docker-${DOCKER_VERSION}.tgz"
    
    download_file "${docker_url}" "${docker_tgz}"
    
    log_info "Extracting Docker binaries to ${INSTALL_DIR}"
    tar -xzf "${docker_tgz}" -C "${temp_dir}"
    cp "${temp_dir}/docker/"* "${INSTALL_DIR}/"
    chmod +x "${INSTALL_DIR}"/docker*
    
    rm -rf "${temp_dir}"
    
    # Configure Docker daemon
    configure_docker_daemon
    
    # Create systemd service if systemd is available
    if command -v systemctl &> /dev/null; then
        create_docker_service
    fi
    
    # Fix socket permissions for non-root access
    fix_docker_socket_permissions
    
    log_info "Docker installed successfully"
}

configure_docker_daemon() {
    local daemon_config="/etc/docker/daemon.json"
    
    if [[ -f "${daemon_config}" ]]; then
        log_info "Docker daemon config already exists: ${daemon_config}"
        return 0
    fi
    
    log_info "Creating Docker daemon configuration"
    mkdir -p /etc/docker
    cat > "${daemon_config}" <<'EOF'
{
  "data-root": "/var/lib/docker",
  "group": "docker"
}
EOF
    
    log_info "Docker daemon.json created"
}

fix_docker_socket_permissions() {
    local docker_socket="/var/run/docker.sock"
    
    if [[ -S "${docker_socket}" ]]; then
        log_info "Fixing Docker socket permissions for non-root access"
        chmod 666 "${docker_socket}"
        log_info "Docker socket permissions updated"
    else
        log_warn "Docker socket ${docker_socket} not found, skipping permission fix"
    fi
}

create_docker_service() {
    local service_file="/etc/systemd/system/docker.service"
    
    if [[ -f "${service_file}" ]]; then
        log_info "Docker systemd service already exists"
        return 0
    fi
    
    log_info "Creating Docker systemd service"
    
    cat > "${service_file}" <<'EOF'
[Unit]
Description=Docker Application Container Engine
Documentation=https://docs.docker.com
After=network-online.target firewalld.service
Wants=network-online.target

[Service]
Type=notify
ExecStart=/usr/local/bin/dockerd
ExecReload=/bin/kill -s HUP $MAINPID
TimeoutSec=0
RestartSec=2
Restart=always
StartLimitBurst=3
StartLimitInterval=60s
LimitNOFILE=infinity
LimitNPROC=infinity
LimitCORE=infinity
TasksMax=infinity
Delegate=yes
KillMode=process

[Install]
WantedBy=multi-user.target
EOF
    
    systemctl daemon-reload
    systemctl enable docker
    systemctl start docker
    chmod 666 /var/run/docker.sock    
    
    log_info "Docker service created and started"
}

# ============================================================================
# Docker Compose Installation
# ============================================================================

check_docker_compose() {
    if command -v docker-compose &> /dev/null || docker compose version &> /dev/null 2>&1; then
        local compose_version
        compose_version="$(docker compose version 2>/dev/null || docker-compose --version 2>/dev/null || echo 'unknown')"
        log_info "Docker Compose is already installed: ${compose_version}"
        return 0
    fi
    return 1
}

install_docker_compose() {
    local arch="$1"
    local compose_url compose_binary
    
    if check_docker_compose; then
        return 0
    fi
    
    log_info "Installing Docker Compose ${DOCKER_COMPOSE_VERSION} for ${arch}"
    
    case "${arch}" in
        x86_64)
            compose_url="https://github.com/docker/compose/releases/download/${DOCKER_COMPOSE_VERSION}/docker-compose-linux-x86_64"
            ;;
        aarch64)
            compose_url="https://github.com/docker/compose/releases/download/${DOCKER_COMPOSE_VERSION}/docker-compose-linux-aarch64"
            ;;
        *)
            fatal "Unsupported architecture for Docker Compose download: ${arch}"
            ;;
    esac
    
    compose_binary="${INSTALL_DIR}/docker-compose"
    
    download_file "${compose_url}" "${compose_binary}"
    chmod +x "${compose_binary}"
    
    # Create docker CLI plugin directory if using newer Docker
    if docker version &> /dev/null; then
        local plugin_dir="/usr/local/lib/docker/cli-plugins"
        mkdir -p "${plugin_dir}"
        cp "${compose_binary}" "${plugin_dir}/docker-compose"
    fi
    
    log_info "Docker Compose installed successfully"
}

# ============================================================================
# Docker Image Loading
# ============================================================================

find_docker_images_archive() {
    local search_dir="${1:-.}"
    local archive
    
    archive="$(find "${search_dir}" -maxdepth 1 -type f -name "docker_images*.tar.gz" | head -n1 || true)"
    
    if [[ -z "${archive}" ]]; then
        # Also try .tgz extension
        archive="$(find "${search_dir}" -maxdepth 1 -type f -name "docker_images*.tgz" | head -n1 || true)"
    fi
    
    echo "${archive}"
}

extract_and_load_docker_images() {
    local archive="$1"
    local cache_dir="${DOCKER_CACHE_DIR}/$(basename "${archive}" .tar.gz)"
    local marker="${cache_dir}/.extracted"
    
    if [[ ! -f "${archive}" ]]; then
        fatal "Docker images archive not found: ${archive}"
    fi
    
    log_info "Processing Docker images from $(basename "${archive}")"
    
    # Extract the aggregated archive if not already done
    if [[ -f "${marker}" ]]; then
        log_info "Archive already extracted (marker ${marker})"
    else
        log_info "Extracting aggregated Docker images archive into ${cache_dir}"
        mkdir -p "${cache_dir}"
        tar -xzf "${archive}" -C "${cache_dir}"
        touch "${marker}"
        
        if [[ -n "${APP_USER}" ]] && [[ -n "${APP_GROUP}" ]]; then
            chown -R "${APP_USER}:${APP_GROUP}" "${cache_dir}" 2>/dev/null || true
        fi
    fi
    
    # Inflate any .tar.gz/.tgz files to .tar
    log_info "Inflating compressed image bundles in ${cache_dir}"
    shopt -s nullglob
    for gz in "${cache_dir}"/*.tar.gz "${cache_dir}"/*.tgz; do
        [[ -f "${gz}" ]] || continue
        
        local base dest
        base="$(basename "${gz}")"
        
        # Skip certain files
        case "${base}" in
            docker-*.tgz|docker-compose*|docker_images*.tar.gz)
                continue
                ;;
        esac
        
        if [[ "${gz}" == *.tar.gz ]]; then
            dest="${gz%.tar.gz}.tar"
        else
            dest="${gz%.tgz}.tar"
        fi
        
        if [[ -f "${dest}" ]]; then
            continue
        fi
        
        log_info "Inflating Docker image bundle ${base}"
        if gunzip -c "${gz}" > "${dest}"; then
            chmod 640 "${dest}" 2>/dev/null || true
            if [[ -n "${APP_USER}" ]] && [[ -n "${APP_GROUP}" ]]; then
                chown "${APP_USER}:${APP_GROUP}" "${dest}" 2>/dev/null || true
            fi
        else
            log_warn "Failed to unpack ${gz}; removing incomplete ${dest}"
            rm -f "${dest}"
        fi
    done
    shopt -u nullglob
    
    # Load .tar files into Docker
    log_info "Loading Docker images from ${cache_dir}"
    local images_found="false"
    local loaded=0
    
    shopt -s nullglob
    for tar_file in "${cache_dir}"/*.tar; do
        [[ -f "${tar_file}" ]] || continue
        
        local base
        base="$(basename "${tar_file}")"
        
        # Skip Docker binary archives
        case "${base}" in
            docker-*.tar)
                continue
                ;;
        esac
        
        images_found="true"
        log_info "Loading Docker image: ${base}"
        
        if docker load -i "${tar_file}"; then
            loaded=$((loaded + 1))
        else
            log_warn "Failed to load Docker image ${tar_file}"
        fi
    done
    shopt -u nullglob
    
    if [[ "${images_found}" == "false" ]]; then
        log_warn "No Docker image archives found in ${cache_dir}"
    else
        log_info "Successfully loaded ${loaded} Docker images"
    fi
    
    # Cleanup
    log_info "Cleaning up extracted cache directory ${cache_dir}"
    rm -rf "${cache_dir}"
    
    # Remove cache root if empty
    if [[ -d "${DOCKER_CACHE_DIR}" ]]; then
        local residual
        residual="$(find "${DOCKER_CACHE_DIR}" -mindepth 1 -print -quit 2>/dev/null || true)"
        if [[ -z "${residual}" ]]; then
            log_info "Removing empty Docker image cache root ${DOCKER_CACHE_DIR}"
            rm -rf "${DOCKER_CACHE_DIR}"
        fi
    fi
}

# ============================================================================
# Main
# ============================================================================

main() {
    local arch archive
    
    # Check if running as root
    if [[ "${EUID}" -ne 0 ]]; then
        fatal "This script must be run as root (use sudo)"
    fi
    
    log_info "Starting Docker installation and image loading"
    
    # Detect architecture
    arch="$(detect_architecture)"
    log_info "Detected architecture: ${arch}"
    
    # Install Docker if needed
    if ! check_docker; then
        install_docker "${arch}"
    fi
    
    # Install Docker Compose if needed
    if ! check_docker_compose; then
        install_docker_compose "${arch}"
    fi
    
    # Find Docker images archive
    if [[ $# -gt 0 ]]; then
        archive="$1"
    else
        archive="$(find_docker_images_archive "$(pwd)")"
    fi
    
    if [[ -z "${archive}" ]]; then
        log_warn "No docker_images*.tar.gz file found in current directory"
        log_info "Usage: $0 [path/to/docker_images.tar.gz]"
        log_info "Docker and Docker Compose have been installed successfully"
        exit 0
    fi
    
    # Ensure Docker daemon is running
    if ! docker ps &> /dev/null; then
        log_warn "Docker daemon is not running. Attempting to start..."
        if command -v systemctl &> /dev/null; then
            systemctl start docker || fatal "Failed to start Docker daemon"
            sleep 3
        else
            fatal "Docker daemon is not running and systemd is not available"
        fi
    fi
    
    # Fix socket permissions (in case Docker was already installed)
    fix_docker_socket_permissions
    
    # Extract and load Docker images
    extract_and_load_docker_images "${archive}"
    
    # Show loaded images
    log_info "Currently loaded Docker images:"
    docker images
    
    log_info "Installation complete!"
    log_info "Docker version: $(docker --version)"
    log_info "Docker Compose version: $(docker compose version 2>/dev/null || docker-compose --version 2>/dev/null || echo 'not available')"
}

main "$@"


