#!/bin/bash
# Import Docker images from tar files

set -e

echo "Docker Image Import Tool"
echo "========================"
echo ""

# Check if Docker is available
if ! command -v docker &> /dev/null; then
    echo "Error: Docker is not installed or not in PATH"
    exit 1
fi

# Set EXTENSIONS_DIR from command line argument or use default
EXTENSIONS_DIR="${1:-/opt/kamiwaza/extensions/kamiwaza-extension-registry/}"

# Ensure EXTENSIONS_DIR ends with a trailing slash
[[ "$EXTENSIONS_DIR" != */ ]] && EXTENSIONS_DIR="${EXTENSIONS_DIR}/"

# Check if we need sudo for docker
DOCKER_CMD="docker"
if ! docker ps >/dev/null 2>&1; then
    DOCKER_CMD="sudo docker"
fi

# Count tar files (no sudo - if you installed it, you can read it)
TAR_COUNT=$(find "$EXTENSIONS_DIR" -name "*.tar" -type f 2>/dev/null | wc -l)

if [ "$TAR_COUNT" -eq 0 ]; then
    echo "Error: No tar files found in $EXTENSIONS_DIR"
    echo "Make sure you've installed the extension registry first:"
    echo "  kamiwazad.sh extensions install <registry-package.tar.gz>"
    exit 1
fi

echo "Found $TAR_COUNT image tar files"
echo ""

# Import each tar file
SUCCESS=0
FAILED=0

while IFS= read -r tar_file; do
    echo "Importing $tar_file..."
    if $DOCKER_CMD load < "$tar_file" > /dev/null 2>&1; then
        echo "  ✓ Success"
        SUCCESS=$((SUCCESS + 1))
    else
        echo "  ✗ Failed"
        FAILED=$((FAILED + 1))
    fi
    echo ""
done < <(find "$EXTENSIONS_DIR" -name "*.tar" -type f 2>/dev/null)

# Summary
echo "========================"
echo "Import Summary:"
echo "  Success: $SUCCESS"
echo "  Failed: $FAILED"

if [ "$FAILED" -gt 0 ]; then
    exit 1
fi
