#!/usr/bin/env python
"""Scoped remote App Garden sync helper.

Authentication:
    Credentials are obtained securely via:
    1. Environment variables (recommended for automation):
       - KAMIWAZA_ADMIN_USER (defaults to 'admin' if only password provided)
       - KAMIWAZA_ADMIN_PASSWORD
    2. Interactive prompt (when running in terminal)

Stage Selection:
    Stage defaults to KAMIWAZA_EXTENSION_STAGE environment variable, or PROD if not set.
    Can be overridden with --stage argument.

Usage examples:
    # Automated/CI: environment variables
    export KAMIWAZA_ADMIN_USER=admin
    export KAMIWAZA_ADMIN_PASSWORD=your_password
    export KAMIWAZA_EXTENSION_STAGE=LOCAL
    python scripts/app_garden_remote_sync.py

    # Interactive: will prompt for credentials
    python scripts/app_garden_remote_sync.py --stage LOCAL
    # Prompts: Username [admin]:
    #          Password for admin:

    # Different stages
    python scripts/app_garden_remote_sync.py --stage DEV
    python scripts/app_garden_remote_sync.py --stage PROD

Authenticates via the production /auth/token endpoint, then issues a sync request
with stage override (LOCAL, DEV, STAGE, PROD). The server handles the host selection;
no persistent config changes are made.
"""

from __future__ import annotations

import argparse
import getpass
import json
import os
import sys
from typing import Dict, Optional

import ssl
import urllib.error
import urllib.parse
import urllib.request

from kamiwaza.lib.util import env_flag_enabled

STAGE_MAP: Dict[str, str] = {
    "LOCAL": "http://localhost:58888",
    "DEV": "https://dev-info.kamiwaza.ai",
    "STAGE": "https://stage-info.kamiwaza.ai",
    "PROD": "https://info.kamiwaza.ai",
}

DEFAULT_API_URL = "https://localhost/api"
OFFLINE_FLAG_CANDIDATES = ("KAMIWAZA_OFFLINE_MODE", "OFFLINE_MODE")


def get_credentials() -> tuple[str, str]:
    """Get credentials securely from environment or interactive prompt.

    Priority order:
    1. Environment variables (KAMIWAZA_ADMIN_USER, KAMIWAZA_ADMIN_PASSWORD)
    2. Interactive getpass prompt (if running in a TTY)

    Returns:
        (username, password) tuple

    Raises:
        RuntimeError: If no credentials available (non-interactive without env vars)
    """
    username = os.getenv("KAMIWAZA_ADMIN_USER")
    password = os.getenv("KAMIWAZA_ADMIN_PASSWORD")

    # If password is set, use username or default to "admin"
    if password:
        username = username or "admin"
        return username, password

    # Interactive prompt if running in terminal
    if sys.stdin.isatty():
        if not username:
            username = input("Username [admin]: ").strip() or "admin"
        password = getpass.getpass(f"Password for {username}: ")
        return username, password

    # No credentials available in non-interactive mode
    raise RuntimeError(
        "Authentication required. Set KAMIWAZA_ADMIN_USER and KAMIWAZA_ADMIN_PASSWORD "
        "environment variables, or run interactively."
    )


def authenticate(
    username: str,
    password: str,
    api_url: str,
    verify_ssl: bool,
) -> str:
    """Authenticate using production /auth/token endpoint.

    Uses the standard OAuth2 password flow via /auth/token.
    Returns the JWT access token for API requests.
    """
    # Prepare form-encoded data
    auth_data = f"username={urllib.parse.quote(username)}&password={urllib.parse.quote(password)}"

    token_data = _request_json(
        method="POST",
        url=f"{api_url}/auth/token",
        headers={"Content-Type": "application/x-www-form-urlencoded"},
        form_data=auth_data,
        verify_ssl=verify_ssl,
    )
    if not token_data.get("access_token"):
        raise RuntimeError(
            "Authentication succeeded but access_token missing in response"
        )

    return token_data["access_token"]


def run_remote_sync(
    token: str, api_url: str, stage: str | None, verify_ssl: bool
) -> Dict[str, object]:
    params = {"stage": stage} if stage else {}
    return _request_json(
        method="POST",
        url=f"{api_url}/apps/remote/sync",
        headers={"Authorization": f"Bearer {token}"},
        params=params,
        json_body={},
        verify_ssl=verify_ssl,
    )


def _request_json(
    *,
    method: str,
    url: str,
    headers: Optional[Dict[str, str]] = None,
    params: Optional[Dict[str, str]] = None,
    json_body: Optional[Dict[str, object]] = None,
    form_data: Optional[str] = None,
    verify_ssl: bool,
) -> Dict[str, object]:
    headers = headers.copy() if headers else {}
    final_url = url
    if params:
        query = urllib.parse.urlencode(params)
        separator = "&" if urllib.parse.urlparse(url).query else "?"
        final_url = f"{url}{separator}{query}"

    data: Optional[bytes] = None
    if json_body is not None:
        headers.setdefault("Content-Type", "application/json")
        data = json.dumps(json_body).encode("utf-8")
    elif form_data is not None:
        data = form_data.encode("utf-8")

    request = urllib.request.Request(
        final_url, data=data, headers=headers, method=method.upper()
    )

    context = None
    scheme = urllib.parse.urlparse(final_url).scheme
    if scheme == "https" and not verify_ssl:
        context = ssl._create_unverified_context()

    try:
        with urllib.request.urlopen(request, context=context) as response:
            response_data = response.read().decode("utf-8")
    except urllib.error.HTTPError as exc:
        error_payload = exc.read().decode("utf-8", errors="replace")
        if exc.code == 401 and "Missing authentication signature" in error_payload:
            raise RuntimeError(
                "Received 'Missing authentication signature' (401). "
                "Call the API through Traefik (e.g., https://localhost/api) "
                "or update --api-url."
            ) from exc
        raise RuntimeError(
            f"HTTP {exc.code} error while calling {final_url}: {error_payload}"
        ) from exc
    except urllib.error.URLError as exc:
        raise RuntimeError(f"Failed to reach {final_url}: {exc.reason}") from exc

    if not response_data:
        return {}

    try:
        return json.loads(response_data)
    except json.JSONDecodeError as exc:
        raise RuntimeError(
            f"Failed to decode JSON response from {final_url}: {response_data}"
        ) from exc


def parse_args(argv: list[str]) -> argparse.Namespace:
    parser = argparse.ArgumentParser(description="Scoped App Garden remote sync")
    parser.add_argument(
        "--stage",
        choices=list(STAGE_MAP.keys()),
        default=os.getenv("KAMIWAZA_EXTENSION_STAGE", "PROD"),
        help="Remote template stage to sync against (default: KAMIWAZA_EXTENSION_STAGE env var or PROD)",
    )
    parser.add_argument(
        "--api-url", default=DEFAULT_API_URL, help="Kamiwaza API base URL"
    )
    parser.add_argument(
        "--verify-ssl", action="store_true", help="Enable TLS certificate verification"
    )
    return parser.parse_args(argv)


def main(argv: list[str]) -> int:
    args = parse_args(argv)

    offline_enabled, offline_source = env_flag_enabled(OFFLINE_FLAG_CANDIDATES)
    if offline_enabled and args.api_url == DEFAULT_API_URL:
        print(
            f"[app_garden_remote_sync] {offline_source} is set; skipping remote sync in offline mode.",
            file=sys.stderr,
        )
        return 0

    username, password = get_credentials()
    token = authenticate(username, password, args.api_url, args.verify_ssl)
    print(f"remote-url: {STAGE_MAP[args.stage]}", file=sys.stderr)
    result = run_remote_sync(token, args.api_url, args.stage, args.verify_ssl)
    print(json.dumps(result, indent=2))
    return 0


if __name__ == "__main__":
    sys.exit(main(sys.argv[1:]))
