#!/bin/bash
set -ex

# Set the environment variable
export PROTOCOL_BUFFERS_PYTHON_IMPLEMENTATION='python'

# Get the directory of the BASH_SOURCE
dir=$(dirname "${BASH_SOURCE[0]}")
dir=$(realpath "${dir}")

# Change to the notebooks directory
cd "${dir}/notebooks"

# Get PIDs of JupyterLab processes
jupyter_pids=$(pgrep -f "jupyter-lab" || true)

# If PIDs were found, kill the processes
if [ -n "$jupyter_pids" ]; then
    for pid in $jupyter_pids; do
        echo "Killing JupyterLab process with PID: $pid"
        kill "$pid"
        # Optionally, wait for the process to be killed
        while kill -0 "$pid" 2> /dev/null; do
            echo "Waiting for JupyterLab process PID $pid to terminate..."
            sleep 1
        done
    done
else
    echo "No JupyterLab process found."
fi

# Use the tracked Jupyter configuration directory (notebooks/.jupyter/)
# This config is version-controlled and should be the source of truth
jupyter_config_dir="${dir}/notebooks/.jupyter"
jupyter_config_file="${jupyter_config_dir}/jupyter_lab_config.py"

if [ ! -f "${jupyter_config_file}" ]; then
    echo "ERROR: Jupyter configuration file not found at ${jupyter_config_file}"
    echo "This file should be tracked in git. Please restore it from version control."
    exit 1
fi

if [ -f "${dir}/kamiwaza-shibboleth" ]; then
    # Only add notebooks directory to avoid conflict with pip-installed kamiwaza package
    export PYTHONPATH="${PYTHONPATH}:${dir}/notebooks"
fi

# Verify the configuration
echo "Jupyter configuration file content:"
cat "${jupyter_config_file}"

# Start JupyterLab with the custom configuration directory
echo "Starting JupyterLab..."
NOTEBOOK_ENV="${dir}/notebook-venv"
PYTHON_BIN="${NOTEBOOK_ENV}/bin/python"
if [[ ! -x "$PYTHON_BIN" ]]; then
    echo "Notebook environment not found at $NOTEBOOK_ENV. Please run setup.sh to create it."
    exit 1
fi

export JUPYTER_CONFIG_DIR="${jupyter_config_dir}"
nohup "$PYTHON_BIN" -m jupyter lab --ip=0.0.0.0 --log-level=ERROR --no-browser > jupyter_lab.log 2>&1 &
sleep 4 # Give it a moment to start and write to the log

# Tokens are removed in 0.3.2+ in favor of JWT tokens; lab port is no longer available on hosts
#grep -E "http://.*token=|https://.*token=" jupyter_lab.log
