#!/bin/bash
set -euo pipefail

# Post uv-sync hook to ensure kamiwaza stays installed from its wheel.
# Prevents uv from re-installing the workspace as editable when only partial
# source files (missing compiled extensions) are present on production hosts.

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="${SCRIPT_DIR}"

# Default to disabling uv auto-sync unless explicitly overridden.
if [[ -z "${KW_PY_NO_SYNC:-}" ]]; then
    export KW_PY_NO_SYNC=1
fi

# Gather candidate wheel files, preferring dist/ but falling back to project root.
WHEEL_PATH="$(find "${PROJECT_ROOT}/dist" -maxdepth 1 -type f -name "kamiwaza-*.whl" 2>/dev/null | sort -r | head -n 1 || true)"
if [[ -z "${WHEEL_PATH}" ]]; then
    WHEEL_PATH="$(find "${SCRIPT_DIR}" -maxdepth 1 -type f -name "kamiwaza-*.whl" 2>/dev/null | sort -r | head -n 1 || true)"
fi

if [[ -z "${WHEEL_PATH}" ]]; then
    echo "[post-uv-sync] No kamiwaza wheel located; skipping reinstallation."
    exit 0
fi

echo "[post-uv-sync] Reinstalling Kamiwaza from wheel: $(basename "$WHEEL_PATH")"

if command -v uv >/dev/null 2>&1; then
    if ! uv --project "${PROJECT_ROOT}" pip install --no-deps --force-reinstall "$WHEEL_PATH" >/dev/null 2>&1; then
        echo "[post-uv-sync] Warning: uv failed to install wheel ${WHEEL_PATH}" >&2
        exit 1
    fi
    exit 0
fi

PIP_BIN="${PROJECT_ROOT}/.venv/bin/pip"
if [[ -x "$PIP_BIN" ]]; then
    if ! "$PIP_BIN" install --no-deps --force-reinstall "$WHEEL_PATH" >/dev/null 2>&1; then
        echo "[post-uv-sync] Warning: pip failed to install wheel ${WHEEL_PATH}" >&2
        exit 1
    fi
else
    echo "[post-uv-sync] Warning: uv and .venv/bin/pip unavailable; cannot reinstall Kamiwaza wheel." >&2
    exit 1
fi
