#!/bin/bash

# Add strict mode flags
set -euo pipefail

# Trap to restore directory on exit
trap 'cd "$OLDPWD" >/dev/null 2>&1 || true' EXIT

# Set cwd to script dir
cd "$(dirname "${BASH_SOURCE[0]}")"

# Detect docker compose version
if command -v docker-compose &>/dev/null; then
  DC="docker-compose"
else
  DC="docker compose"
fi

# Ensure DC is always set
: "${DC:=docker compose}"

echo "Using Docker Compose via: $DC"

source set-kamiwaza-root.sh
source common.sh

ensure_nofile_min 65536 || echo "warn: could not set soft nofile to 65536" >&2

# from common.sh
setup_environment

# Source the env.sh file determined by setup_environment
# Fall back to local env.sh if KAMIWAZA_ENV_FILE_PATH is not set
if [[ -n "${KAMIWAZA_ENV_FILE_PATH:-}" && -f "${KAMIWAZA_ENV_FILE_PATH}" ]]; then
    source "${KAMIWAZA_ENV_FILE_PATH}"
elif [[ -f /etc/kamiwaza/env.sh ]]; then
    source /etc/kamiwaza/env.sh
elif [[ -f env.sh ]]; then
    source env.sh
fi

# Set up environment variables (ARCH, KAMIWAZA_LOG_DIR, OTEL/Loki/Jaeger defaults)
# Note: This function handles both OTEL enabled and disabled states properly
setup_common_environment


show_help() {
    echo "Usage: $(basename "$0") [options]"
    echo ""
    echo "Options:"
    echo "  -h, --help            Show this help message and exit"
    echo "  --ray-head IP         Set the Ray head IP"
    echo "  --ray-port PORT       Set the Ray head port"
    echo "  [other options]       Other options to pass to launch.py"
}

# Check for help flag
for arg in "$@"; do
    case $arg in
        -h|--help)
            show_help
            exit 0
            ;;
    esac
done

# Collect flags to pass to launch.py
flags="$@"

# Add ray head IP and port if set
if [ -n "${KAMIWAZA_HEAD_IP:-}" ] && [ -n "${KAMIWAZA_HEAD_PORT:-}" ]; then
    # Check if we are the head node by comparing our IP with KAMIWAZA_HEAD_IP
    if ifconfig | grep inet | awk '{print $2}' | grep -q "^${KAMIWAZA_HEAD_IP}$" > /dev/null 2>&1; then
        flags+=" --ray-head ${KAMIWAZA_HEAD_IP} --ray-port ${KAMIWAZA_HEAD_PORT}"
    fi
fi

launch_python_process() {
    # Run python in background and capture its PID
    if [ "$1" = "dev" ]; then
        kw_py kamiwaza/launch.py $flags &
    else
        kw_py launch.py $flags &
    fi
    # Capture the Python process PID
    echo $! > /tmp/kamiwaza-core.pid
    # Wait for the Python process to complete
    wait $!
    # Clean up PID file on exit
    rm -f /tmp/kamiwaza-core.pid
}

if [ -f launch.py ]; then
    launch_python_process "prod"
elif [ -f kamiwaza/launch.py ]; then
    # dev layout
    # If KAMIWAZA_ANALYTICS_URL is not set and kamiwaza-shibboleth exists, set the default analytics URL
    if [ -f "kamiwaza-shibboleth" ] && [ -z "${KAMIWAZA_ANALYTICS_URL:-}" ]; then
        export KAMIWAZA_ANALYTICS_URL="https://analytics-dev.kamiwaza.ai/v1/events"
    fi

    # If running in a repo layout where package is in-tree (kamiwaza/ exists), add CWD to PYTHONPATH
    if [ -f "kamiwaza-shibboleth" ] && [ -z "${KAMIWAZA_NO_SET_PATH:-}" ]; then
        if [ -d "kamiwaza" ]; then
            case ":${PYTHONPATH:-}:" in
                *:"$(pwd)":*) : ;;
                *) export PYTHONPATH="${PYTHONPATH:+${PYTHONPATH}:}$(pwd)" ;;
            esac
        fi
    fi
    launch_python_process "dev"
else
    echo " ### KAMIWAZA_STARTUP_ERROR: No launch.py found"
    exit 1
fi
