import os
from kamiwaza.lib.util import get_kamiwaza_root, set_env_from_env_sh

# TODO: investigate a better way to deal with python etcd3 lib being dated.
os.environ['PROTOCOL_BUFFERS_PYTHON_IMPLEMENTATION'] = 'python'

import logging
import sys
from kamiwaza.util.config import RuntimeConfig
from kamiwaza.util.config import RuntimeClusterManager
from kamiwaza.lib.logging.utils import get_debug_level

logger = logging.getLogger(__name__)
logger.setLevel(get_debug_level())

KAMIWAZA_ROOT = get_kamiwaza_root()
os.environ["KAMIWAZA_ROOT"] = KAMIWAZA_ROOT
os.environ["KAMIWAZA_LAUNCH_HOST"] = "true"

# Set environment variables from env.sh
env_sh_path = os.path.join(KAMIWAZA_ROOT, 'env.sh')
set_env_from_env_sh(env_sh_path)

# Backward compatibility: If KAMIWAZA_DEBUG_MODE is set but KAMIWAZA_DEBUG is not, use KAMIWAZA_DEBUG_MODE value
if "KAMIWAZA_DEBUG" not in os.environ and "KAMIWAZA_DEBUG_MODE" in os.environ:
    os.environ["KAMIWAZA_DEBUG"] = os.environ["KAMIWAZA_DEBUG_MODE"]

# Check if PYTHONPATH contains a path element with 'kamiwaza' that does not match the current working directory
pythonpath = os.environ.get('PYTHONPATH', '')
cwd = os.getcwd()
parent_cwd = os.path.dirname(cwd) if cwd.endswith('kamiwaza') else None
allow_extra_kamiwaza_path = '--allow-extra-kamiwaza-path' in sys.argv

for path in pythonpath.split(os.pathsep):
    if 'kamiwaza' in path and not (path.startswith(cwd) or (parent_cwd and path.startswith(parent_cwd))):
        message = f"PYTHONPATH contains a path element with 'kamiwaza' that does not match the current working directory: {path}"
        if allow_extra_kamiwaza_path:
            print(f"Warning: {message}")
            logger.warning(f"Warning: {message}")
        else:
            print(f"Fatal error: {message}")
            logger.error(f"Fatal error: {message}")
            sys.exit(1)

rc = RuntimeConfig()
rc.set_config('initialized', True)

# If we have updated the db config we will need that, otherwise,
# flush the cache. Should really only matter in development or
# upgrades
rcm = RuntimeClusterManager()
rcm.invalidate_config_cache(excluded_config_keys=['cluster'])

# Path to main.py
main_py_path = None
if os.path.exists(os.path.join(KAMIWAZA_ROOT, 'main.py')):
    main_py_path = os.path.join(KAMIWAZA_ROOT, 'main.py')
elif os.path.exists(os.path.join(KAMIWAZA_ROOT, 'kamiwaza', 'main.py')):
    main_py_path = os.path.join(KAMIWAZA_ROOT, 'kamiwaza', 'main.py')
else:
    logger.error("main.py not found in KAMIWAZA_ROOT or KAMIWAZA_ROOT/kamiwaza")
    raise FileNotFoundError("main.py not found in KAMIWAZA_ROOT or KAMIWAZA_ROOT/kamiwaza")

# Function to display help message
def show_help():
    print("Usage: launch.py [options]")
    print("")
    print("Options:")
    print("  -h, --help            Show this help message and exit")
    print("  --standalone          Run in standalone mode without clustering")
    print("  --ray-host=HOST       Specify the Ray head node host")
    print("  --ray-port=PORT       Specify the Ray head node port")
    print("     Note: if neither --standalone or -ray-host is passed, Kamiwaza starts ray with this node as the head node")
    print("  --allow-extra-kamiwaza-path")
    print("                        Developer only: Allow extra 'kamiwaza' path in PYTHONPATH")

# Check for the help flag
if '-h' in sys.argv or '--help' in sys.argv:
    show_help()
    sys.exit(0)

#  Develope the --standalone flag and other arguments in the command line
standalone_flag = '--standalone' if '--standalone' in sys.argv else ''
ray_host = next((arg for arg in sys.argv if arg.startswith('--ray-host=')), '')
ray_port = next((arg for arg in sys.argv if arg.startswith('--ray-port=')), '')

if os.path.exists(os.path.join(KAMIWAZA_ROOT, 'ray-is-worker')):
    if not ray_host:
        logger.critical("This node is not the Kamiwaza head node, but ray_host is not set; set KAMIWAZA_RAY_HOST (and if needed KAMIWAZA_RAY_PORT) in start! Exiting.")
        sys.exit(1)

# Build arguments for main.py with the standalone flag and other arguments if present
env = os.environ.copy()
exec_args = ['python', main_py_path]
if standalone_flag:
    exec_args.append(standalone_flag)
if ray_host:
    exec_args.append(ray_host)
if ray_port:
    exec_args.append(ray_port)

# Use exec to replace this process with main.py
# This ensures signals (SIGTERM, SIGINT) are received directly by main.py
# enabling graceful shutdown and ephemeral deployment cleanup
logger.info(f"Executing main.py: {' '.join(exec_args)}")
os.execvpe(exec_args[0], exec_args, env)
