#!/usr/bin/env python3

import os
import sys
import re
from typing import List, Dict
import importlib.util
import yaml

# Use importlib to import remapper module
remapper_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'remapper.py'))
spec = importlib.util.spec_from_file_location("remapper", remapper_path)
remapper = importlib.util.module_from_spec(spec)
spec.loader.exec_module(remapper)

def process_traefik_service(service_config: Dict) -> Dict:
    """
    Process the traefik service to add the required networks.

    :param service_config: Dictionary containing the traefik service configuration.
    :return: Processed traefik service configuration.
    """
    networks = service_config.get('networks', [])
    
    # Ensure the host-services network is present
    if 'host-services' not in networks:
        networks.append('host-services')
    
    # Ensure the kamiwaza-backend network is present
    if 'kamiwaza-backend' not in networks:
        networks.append('kamiwaza-backend')

    # Ensure the kamiwaza-apps network is present
    if 'kamiwaza-apps' not in networks:
        networks.append('kamiwaza-apps')
    
    # Update the networks in the service configuration
    service_config['networks'] = networks
    
    return service_config

def process_file(input_file: str, output_file: str) -> None:
    try:
        with open(input_file, "r", encoding="utf-8") as infile:
            data = yaml.safe_load(infile)

        # Process services
        if 'services' in data:
            for service_name, service_config in data['services'].items():
                if service_name == 'traefik':
                    data['services'][service_name] = process_traefik_service(service_config)
                else:
                    # For other services, we don't modify the networks
                    pass

        # Update networks section
        data['networks'] = {
            'etcd': {
                'name': '${KAMIWAZA_ENV:-default}_kamiwaza-etcd',
                'external': True
            },
            'host-services': {
                'name': '${KAMIWAZA_ENV:-default}_host-services',
                'driver': 'bridge'
            },
            'kamiwaza-backend': {
                'name': '${KAMIWAZA_ENV:-default}_kamiwaza-backend',
                'external': True
            },
            'kamiwaza-apps': {
                'name': '${KAMIWAZA_ENV:-default}_kamiwaza-apps',
                'external': True
            }
        }

        # Write the updated YAML
        with open(output_file, "w", encoding="utf-8") as outfile:
            yaml.dump(data, outfile, default_flow_style=False)

    except Exception as e:
        print(f"An error occurred while processing the file: {str(e)}", file=sys.stderr)
        raise

def main() -> None:
    """
    Main function to process the vend-docker-compose.yml file.
    """
    input_file = "vend-docker-compose.yml"
    output_file = "remapped-docker-compose.yml"
    remapper.file_exists(input_file)
    process_file(input_file, output_file)
    print("Processing complete. Output saved in", output_file)

if __name__ == "__main__":
    main()