#!/bin/bash
set -euo pipefail

# Set component-specific log prefix
LOG_PREFIX="TRAEFIK-PRELAUNCH"

# Source the common logging functions
if [ -f "./common-logging.sh" ]; then
    source ./common-logging.sh
else
    # Fallback if common-logging.sh not found
    log_info() { echo "[INFO] $1"; }
    log_warn() { echo "[WARN] $1"; }
    log_error() { echo "[ERROR] $1"; }
    log_success() { echo "[SUCCESS] $1"; }
    log_step() { echo ""; echo "========== $1 =========="; }
fi

render_dynamic_config() {
    local script_path="./render_dynamic_config.py"
    if [ ! -f "$script_path" ] && [ -f "../render_dynamic_config.py" ]; then
        script_path="../render_dynamic_config.py"
    fi
    if [ ! -f "$script_path" ] && [ -f "../../render_dynamic_config.py" ]; then
        script_path="../../render_dynamic_config.py"
    fi
    local root_path="${KAMIWAZA_ROOT:-}"
    if [ -z "$root_path" ]; then
        root_path="$(pwd)"
        while [[ "$root_path" != "/" && ! -f "$root_path/launch.py" && ! -f "$root_path/install.py" ]]; do
            root_path="$(dirname "$root_path")"
        done
    fi

    if [ ! -f "$script_path" ]; then
        local fallback_script="${root_path}/deployment/kamiwaza-traefik/render_dynamic_config.py"
        if [ -f "$fallback_script" ]; then
            script_path="$fallback_script"
        else
            log_info "render_dynamic_config.py not found; skipping dynamic config render"
            return
        fi
    fi

    local runtime_config_path="${root_path}/runtime/runtime_config.json"
    if [ ! -f "$runtime_config_path" ]; then
        runtime_config_path="${root_path}/runtime_config.json"
    fi

    if [ ! -f "$runtime_config_path" ]; then
        log_warn "runtime_config.json not found at $runtime_config_path; skipping dynamic config render"
        return
    fi

    local python_cmd="python3"
    if ! ${python_cmd} - <<'PY' >/dev/null 2>&1
import yaml
PY
    then
        if [ -x "${root_path}/scripts/kw_py" ]; then
            python_cmd="${root_path}/scripts/kw_py"
        else
            log_warn "PyYAML not available; skipping dynamic config render"
            return
        fi
    fi

    log_step "Rendering Traefik dynamic config"
    ${python_cmd} "$script_path" \
        --base ./traefik-dynamic.full.yml \
        --output ./traefik-dynamic.yml \
        --runtime-config "$runtime_config_path"
}

render_dynamic_config

log_step "Setting up Traefik certificates"

# Define variables
CERT_DIR="./certs"
DOMAIN="*.default.deployment.kamiwaza.ai"
CERT_FILE="$CERT_DIR/domain.crt"
KEY_FILE="$CERT_DIR/domain.key"
DAYS_VALID=3650
RSA_LENGTH=4096

# Check if the certs directory and the necessary files exist
if [ ! -d "$CERT_DIR" ] || [ ! -f "$CERT_FILE" ] || [ ! -f "$KEY_FILE" ]; then
    log_info "Certificates not found for Traefik, generating new self-signed certificate..."

    # Create certs directory if it doesn't exist
    log_info "Creating certificates directory..."
    mkdir -p "$CERT_DIR"

    if [[ "$OSTYPE" == "darwin"* ]]; then
        chown ${USER}:staff "$CERT_DIR"
    else
        sudo chown ${USER}:${USER} "$CERT_DIR"
    fi
    
    # Generate the private key and self-signed certificate
    log_info "Generating self-signed certificate for domain: $DOMAIN"
    openssl req -newkey rsa:$RSA_LENGTH -nodes -keyout "$KEY_FILE" -x509 -sha256 -days $DAYS_VALID \
    -subj "/C=US/ST=State/L=City/O=Organization/OU=Unit/CN=$DOMAIN" \
    -addext "subjectAltName=DNS:$DOMAIN" \
    -out "$CERT_FILE"
    
    if [ $? -eq 0 ]; then
        log_success "Certificate and key have been generated and stored in $CERT_DIR"
    else
        log_error "Failed to generate certificate"
        exit 1
    fi
else
    log_info "Certificates already exist in $CERT_DIR"
fi

log_success "Traefik certificate setup completed"
