#!/bin/bash
set -euo pipefail

# Set component-specific log prefix
LOG_PREFIX="TRAEFIK-POSTLAUNCH"

# Source the common logging functions
if [ -f "./common-logging.sh" ]; then
    source ./common-logging.sh
else
    # Fallback if common-logging.sh not found
    log_info() { echo "[INFO] $1"; }
    log_warn() { echo "[WARN] $1"; }
    log_error() { echo "[ERROR] $1"; }
    log_success() { echo "[SUCCESS] $1"; }
    log_step() { echo ""; echo "========== $1 =========="; }
fi

log_info "Traefik postlaunch script - currently disabled"
# For now we are no longer pushing keys into etcd
exit 0

# Get the absolute path of the current script
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" &> /dev/null && pwd)"

# Find the KAMIWAZA_ROOT folder by navigating up the directory tree
KAMIWAZA_ROOT="$SCRIPT_DIR"
while [[ "$KAMIWAZA_ROOT" != "/" && ! -f "$KAMIWAZA_ROOT/install.py" ]]; do
    KAMIWAZA_ROOT="$(dirname "$KAMIWAZA_ROOT")"
done

# Check if KAMIWAZA_ROOT was found
if [[ ! -f "$KAMIWAZA_ROOT/launch.py" && ! -f "$KAMIWAZA_ROOT/install.py" ]]; then
    echo "Error: KAMIWAZA_ROOT not found. Please ensure launch.py or install.py exists in the expected directory structure."
    exit 1
fi

# one more folder test
if [[ -d "$KAMIWAZA_ROOT/deployment" && ! -d "$KAMIWAZA_ROOT/kamiwaza/deployment" ]]; then
    KAMIWAZA_ROOT="$(dirname "$KAMIWAZA_ROOT")"
fi

# set this so Linux systems don't try using v2 and throwing errors
export ETCDCTL_API=3

# Define the etcd certs directory
ETCD_CERTS_DIR="${KAMIWAZA_ROOT}/runtime/etcd/certs"

# Define etcd endpoint and TLS parameters
ETCD_ENDPOINT="https://127.0.0.1:12379"
ETCD_CACERT="${ETCD_CERTS_DIR}/ca.pem"
ETCD_CERT="${ETCD_CERTS_DIR}/peer-${KAMIWAZA_ETCD_NODE_NAME:-kamiwaza0}.pem"
ETCD_KEY="${ETCD_CERTS_DIR}/peer-${KAMIWAZA_ETCD_NODE_NAME:-kamiwaza0}-key.pem"

# Check for virtual environment
VENV_PATHS=(
    "${KAMIWAZA_ROOT}/.venv/bin/activate"
    "../../../../../../.venv/bin/activate"
)

VENV_ACTIVATED=false
for VENV_PATH in "${VENV_PATHS[@]}"; do
    if [ -f "$VENV_PATH" ]; then
        echo "Activating virtual environment at $VENV_PATH..."
        source "$VENV_PATH"
        VENV_ACTIVATED=true
        break
    fi
done

if [ "$VENV_ACTIVATED" = false ]; then
    echo "No virtual environment found. Proceeding with system Python."
fi

# Check if the Python script exists
if [ ! -f remap-traefik.py ]; then
    echo "#### ERROR: remap-traefik.py not found. Exiting."
    exit 1
fi

# Check if traefik-dynamic.yml exists
if [ ! -f traefik-dynamic.yml ]; then
    echo "#### ERROR: traefik-dynamic.yml not found. Exiting."
    exit 1
fi

# Use the Python script to generate etcd put commands
echo "Generating etcd put commands and verifying insertions..."
while IFS= read -r cmd; do
    # Split the command into key and value, preserving spaces in the value
    key=$(echo "$cmd" | cut -d' ' -f1)
    value="${cmd#* }"
    
    # Add the 'traefik/' prefix to the key
    full_key="traefik/$key"
    
    # Exclude the definition of the jwt middleware but not its application
    if [[ "$full_key" == "traefik/http/middlewares/jwt"* ]]; then
        echo "Skipping jwt middleware definition: $full_key"
        continue
    fi
    
    # Put the key-value pair into etcd
    etcdctl --endpoints="$ETCD_ENDPOINT" \
        --cacert="$ETCD_CACERT" \
        --cert="$ETCD_CERT" \
        --key="$ETCD_KEY" \
        put "$full_key" "$value"
    
    # Verify the insertion
    retrieved_value=$(etcdctl --endpoints="$ETCD_ENDPOINT" \
        --cacert="$ETCD_CACERT" \
        --cert="$ETCD_CERT" \
        --key="$ETCD_KEY" \
        get "$full_key" --print-value-only)
    
    if [ -n "$retrieved_value" ]; then
        echo "Verified insert to etcd: $full_key: $retrieved_value"
    else
        echo "#### WARNING: Failed to verify insert for key: $full_key"
    fi
done < <(python3 remap-traefik.py traefik-dynamic.yml)

echo "Configuration upload and verification complete."

# Deactivate virtual environment if it was activated
if [ "$VENV_ACTIVATED" = true ]; then
    deactivate
fi