#!/usr/bin/env python3

import os
import sys
import yaml
import importlib.util

# Use importlib to import remapper module
remapper_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'remapper.py'))
spec = importlib.util.spec_from_file_location("remapper", remapper_path)
remapper = importlib.util.module_from_spec(spec)
spec.loader.exec_module(remapper)

def process_file(input_file: str, output_file: str) -> None:
    """
    Processes the input file, adding kamiwaza-backend network configuration,
    and writes the output to a new file.

    :param input_file: Path to the input file.
    :param output_file: Path to the output file.
    """
    try:
        with open(input_file, "r", encoding="utf-8") as infile:
            data = yaml.safe_load(infile)

        # Add kamiwaza-backend network to each service
        if 'services' in data:
            for service_name, service_config in data['services'].items():
                if 'networks' not in service_config:
                    service_config['networks'] = []
                service_config['networks'].append("${KAMIWAZA_ENV}_kamiwaza-backend")

        # Add kamiwaza-backend network section
        if 'networks' not in data:
            data['networks'] = {}
        data['networks']['kamiwaza-backend'] = {
            'name': '${KAMIWAZA_ENV}_kamiwaza-backend',
            'external': True
        }

        # Write the updated YAML
        with open(output_file, "w", encoding="utf-8") as outfile:
            yaml.dump(data, outfile, default_flow_style=False)

    except Exception as e:
        print(f"An error occurred while processing the file: {str(e)}", file=sys.stderr)
        raise

def main() -> None:
    """
    Main function to process the vend-docker-compose.yml file.
    """
    input_file = "vend-docker-compose.yml"
    output_file = "remapped-docker-compose.yml"
    remapper.file_exists(input_file)
    process_file(input_file, output_file)
    print("Processing complete. Output saved in", output_file)

if __name__ == "__main__":
    main()
