#!/usr/bin/env python3
"""
OTEL Collector & Jaeger vend-remapper.py
Transforms vendor docker-compose.yml with Kamiwaza-specific business logic
"""
import os
import sys
import importlib.util
import yaml

# Use importlib to import remapper module
remapper_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'remapper.py'))
spec = importlib.util.spec_from_file_location("remapper", remapper_path)
remapper = importlib.util.module_from_spec(spec)
spec.loader.exec_module(remapper)

# Custom YAML representer for proper indentation
class MyDumper(yaml.Dumper):
    def increase_indent(self, flow=False, indentless=False):
        return super(MyDumper, self).increase_indent(flow, indentless)

def str_presenter(dumper, data):
    if len(data.splitlines()) > 1:  # check for multiline string
        return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')
    return dumper.represent_scalar('tag:yaml.org,2002:str', data)

yaml.add_representer(str, str_presenter, Dumper=MyDumper)

def process_file(input_file: str, output_file: str) -> None:
    try:
        with open(input_file, "r", encoding="utf-8") as infile:
            data = yaml.safe_load(infile)

        # Process services
        if 'services' in data:
            for service_name, service_config in data['services'].items():

                if service_name == 'otel-collector':
                    # Update container name with KAMIWAZA_ENV prefix
                    service_config['container_name'] = '${KAMIWAZA_ENV:-default}_kamiwaza-otel-collector'

                    # Add additional volumes for Kamiwaza log access
                    if 'volumes' not in service_config:
                        service_config['volumes'] = []
                    service_config['volumes'].extend([
                        '/var/lib/docker/containers:/var/lib/docker/containers:ro',
                        '${KAMIWAZA_LOG_DIR:-/tmp}:/tmp:rw',
                        'otel-logs:/var/log/otel:rw'
                    ])

                    # Update config path to match Kamiwaza's location
                    for i, vol in enumerate(service_config['volumes']):
                        if 'otelcol-contrib/config.yaml' in vol:
                            service_config['volumes'][i] = './config.yml:/etc/otel-collector-config.yaml:ro'

                    # Update command to match new config path
                    service_config['command'] = ['--config=/etc/otel-collector-config.yaml']

                    # Add Kamiwaza-specific environment variables
                    service_config['environment'] = [
                        'KAMIWAZA_VERSION=${KAMIWAZA_VERSION:-1.0.0}',
                        'KAMIWAZA_SESSION_ID=${KAMIWAZA_SESSION_ID:-}',
                        'KAMIWAZA_LICENSE_TYPE=${KAMIWAZA_LICENSE_TYPE:-Community}',
                        'KAMIWAZA_ENV=${KAMIWAZA_ENV:-development}',
                        'OTEL_CONTAINER_LOG_DIR=/tmp',
                        'OTEL_SERVICE_NAME=${OTEL_SERVICE_NAME:-kamiwaza}',
                        'OTEL_EXPORTER_INSECURE=${OTEL_EXPORTER_INSECURE:-false}',
                        'OTEL_EXPORTER_INSECURE_JAEGER=${OTEL_EXPORTER_INSECURE_JAEGER:-true}',
                        'CUSTOMER_OTLP_ENDPOINT=${CUSTOMER_OTLP_ENDPOINT:-http://jaeger:14250}',
                        'CUSTOMER_OTLP_AUTH=${CUSTOMER_OTLP_AUTH:-}',
                        'SPLUNK_ENDPOINT=${SPLUNK_ENDPOINT:-http://localhost:8088}',
                        'SPLUNK_TOKEN=${SPLUNK_TOKEN:-disabled}',
                        'LOKI_ENDPOINT=${LOKI_ENDPOINT:-http://loki:3100/loki/api/v1/push}',
                        'JAEGER_OTLP_ENDPOINT=${JAEGER_OTLP_ENDPOINT:-jaeger:4317}',
                        'KAMIWAZA_ANALYTICS_URL=${KAMIWAZA_ANALYTICS_URL:-https://analytics.kamiwaza.ai/v1/events}',
                        'KAMIWAZA_TELEMETRY_AUTH=${KAMIWAZA_TELEMETRY_AUTH:-}'
                    ]

                    # Add resource limits
                    service_config['deploy'] = {
                        'resources': {
                            'limits': {
                                'memory': '1.5G',
                                'cpus': '0.75'
                            },
                            'reservations': {
                                'memory': '512M',
                                'cpus': '0.5'
                            }
                        }
                    }

                    # Disable health check (distroless container lacks curl/nc/wget)
                    service_config['healthcheck'] = {'disable': True}

                    # Add logging configuration
                    service_config['logging'] = {
                        'driver': 'json-file',
                        'options': {
                            'max-size': '10m',
                            'max-file': '3',
                            'labels': 'service,version,environment'
                        }
                    }

                    # Add labels
                    service_config['labels'] = [
                        'service=otel-collector',
                        'version=${KAMIWAZA_VERSION:-1.0.0}',
                        'environment=${KAMIWAZA_ENV:-development}',
                        'component=observability'
                    ]

                elif service_name == 'jaeger':
                    # Update container name with KAMIWAZA_ENV prefix
                    service_config['container_name'] = '${KAMIWAZA_ENV:-default}_kamiwaza-jaeger'

                    # Update restart policy to be controlled by environment variable
                    service_config['restart'] = '${KAMIWAZA_JAEGER_RESTART:-no}'

                    # Add labels
                    service_config['labels'] = [
                        'service=jaeger',
                        'version=1.52',
                        'environment=${KAMIWAZA_ENV:-development}',
                        'component=observability'
                    ]

        # Add volumes section
        if 'volumes' not in data:
            data['volumes'] = {}
        data['volumes']['otel-logs'] = {'driver': 'local'}

        # Add networks section
        if 'networks' not in data:
            data['networks'] = {}
        data['networks']['default'] = {
            'name': 'kamiwaza-observability'
        }

        # Write remapped file
        with open(output_file, "w", encoding="utf-8") as outfile:
            yaml.dump(data, outfile, Dumper=MyDumper, default_flow_style=False, sort_keys=False)

        print(f"Successfully remapped {input_file} to {output_file}")

    except Exception as e:
        print(f"Error processing file: {e}", file=sys.stderr)
        sys.exit(1)

if __name__ == "__main__":
    script_dir = os.path.dirname(os.path.abspath(__file__))
    input_file = os.path.join(script_dir, "vend-docker-compose.yml")
    output_file = os.path.join(script_dir, "remapped-docker-compose.yml")

    process_file(input_file, output_file)
