#!/usr/bin/env python3
import os
import sys
import importlib.util
import yaml

# Use importlib to import remapper module
remapper_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'remapper.py'))
spec = importlib.util.spec_from_file_location("remapper", remapper_path)
remapper = importlib.util.module_from_spec(spec)
spec.loader.exec_module(remapper)

# Custom YAML representer for proper indentation
class MyDumper(yaml.Dumper):
    def increase_indent(self, flow=False, indentless=False):
        return super(MyDumper, self).increase_indent(flow, indentless)

def str_presenter(dumper, data):
    if len(data.splitlines()) > 1:  # check for multiline string
        return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')
    return dumper.represent_scalar('tag:yaml.org,2002:str', data)

yaml.add_representer(str, str_presenter, Dumper=MyDumper)

def process_file(input_file: str, output_file: str) -> None:
    try:
        with open(input_file, "r", encoding="utf-8") as infile:
            data = yaml.safe_load(infile)

        # Process services
        if 'services' in data:
            for service_name, service_config in data['services'].items():
                # Keep original network keys but ensure they exist
                service_config['networks'] = ["milvus"]
                
                if service_name == 'standalone':
                    service_config['networks'].append("kamiwaza-backend")
                    # Update container name and environment variables
                    service_config['container_name'] = '${KAMIWAZA_ENV:-default}_milvus-standalone'
                    # Update both ETCD_ENDPOINTS and MINIO_ADDRESS environment variables
                    if 'environment' in service_config:
                        new_env = []
                        for env_var in service_config['environment']:
                            if env_var.startswith('ETCD_ENDPOINTS'):
                                new_env.append('ETCD_ENDPOINTS=etcd:2379')
                            elif env_var.startswith('MINIO_ADDRESS'):
                                new_env.append('MINIO_ADDRESS=minio:9000')
                            else:
                                new_env.append(env_var)
                        service_config['environment'] = new_env
                elif service_name == 'etcd':
                    # Update the container name for etcd service
                    service_config['container_name'] = '${KAMIWAZA_ENV:-default}_milvus-etcd'
                elif service_name == 'minio':
                    # Update the container name for minio service
                    service_config['container_name'] = '${KAMIWAZA_ENV:-default}_milvus-minio'

                # Fix healthcheck command format if it's a list
                if 'healthcheck' in service_config and 'test' in service_config['healthcheck']:
                    test = service_config['healthcheck']['test']
                    if isinstance(test, list) and len(test) > 1 and test[0] == "CMD":
                        service_config['healthcheck']['test'] = ["CMD"] + test[1:]

        # Update networks section - keep original keys but update names
        data['networks'] = {
            'milvus': {
                'name': '${KAMIWAZA_ENV:-default}_kamiwaza-milvus',
                'external': True
            },
            'kamiwaza-backend': {
                'name': '${KAMIWAZA_ENV:-default}_kamiwaza-backend',
                'external': True
            }
        }

        # Remove version if present
        if 'version' in data:
            del data['version']

        # Write the updated YAML with proper indentation and formatting
        with open(output_file, "w", encoding="utf-8") as outfile:
            yaml.dump(data, outfile, Dumper=MyDumper, default_flow_style=False, indent=2, width=float("inf"))

    except Exception as e:
        print(f"An error occurred while processing the file: {str(e)}", file=sys.stderr)
        raise

def main() -> None:
    """
    Main function to process the vend-docker-compose.yml file.
    """
    input_file = "vend-docker-compose.yml"
    output_file = "remapped-docker-compose.yml"

    remapper.file_exists(input_file)
    process_file(input_file, output_file)
    print("Processing complete. Output saved in", output_file)

if __name__ == "__main__":
    main()