#!/usr/bin/env python3
"""
Loki & Grafana vend-remapper.py
Transforms vendor docker-compose.yml with Kamiwaza-specific business logic
"""
import os
import sys
import importlib.util
import yaml

# Use importlib to import remapper module
remapper_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'remapper.py'))
spec = importlib.util.spec_from_file_location("remapper", remapper_path)
remapper = importlib.util.module_from_spec(spec)
spec.loader.exec_module(remapper)

# Custom YAML representer for proper indentation
class MyDumper(yaml.Dumper):
    def increase_indent(self, flow=False, indentless=False):
        return super(MyDumper, self).increase_indent(flow, indentless)

def str_presenter(dumper, data):
    if len(data.splitlines()) > 1:  # check for multiline string
        return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')
    return dumper.represent_scalar('tag:yaml.org,2002:str', data)

yaml.add_representer(str, str_presenter, Dumper=MyDumper)

def process_file(input_file: str, output_file: str) -> None:
    try:
        with open(input_file, "r", encoding="utf-8") as infile:
            data = yaml.safe_load(infile)

        # Process services
        if 'services' in data:
            for service_name, service_config in data['services'].items():

                if service_name == 'loki':
                    # Update container name with KAMIWAZA_ENV prefix
                    service_config['container_name'] = '${KAMIWAZA_ENV:-default}_kamiwaza-loki'

                    # Update volumes to add Kamiwaza log directory and fix config path
                    if 'volumes' not in service_config:
                        service_config['volumes'] = []

                    # Update config path
                    for i, vol in enumerate(service_config['volumes']):
                        if 'local-config.yaml' in vol:
                            service_config['volumes'][i] = './loki-config.yml:/etc/loki/loki-config.yml:ro'

                    # Add additional volumes
                    service_config['volumes'].extend([
                        '${KAMIWAZA_LOG_DIR:-/tmp}/loki:/var/log/loki:rw'
                    ])

                    # Update command to match new config path
                    service_config['command'] = ['-config.file=/etc/loki/loki-config.yml']

                    # Add Kamiwaza-specific environment variables
                    service_config['environment'] = [
                        'KAMIWAZA_VERSION=${KAMIWAZA_VERSION:-1.0.0}',
                        'KAMIWAZA_ENV=${KAMIWAZA_ENV:-development}',
                        'KAMIWAZA_LOG_DIR=${KAMIWAZA_LOG_DIR:-/tmp}'
                    ]

                    # Add resource limits
                    service_config['deploy'] = {
                        'resources': {
                            'limits': {
                                'memory': '1G',
                                'cpus': '0.5'
                            },
                            'reservations': {
                                'memory': '256M',
                                'cpus': '0.25'
                            }
                        }
                    }

                    # Add health check
                    service_config['healthcheck'] = {
                        'test': ['CMD', 'wget', '--no-verbose', '--tries=1', '--spider', 'http://localhost:3100/ready'],
                        'interval': '30s',
                        'timeout': '10s',
                        'retries': 3,
                        'start_period': '15s'
                    }

                    # Add logging configuration
                    service_config['logging'] = {
                        'driver': 'json-file',
                        'options': {
                            'max-size': '10m',
                            'max-file': '3',
                            'labels': 'service,version,environment'
                        }
                    }

                    # Add labels
                    service_config['labels'] = [
                        'service=loki',
                        'version=2.9.2',
                        'environment=${KAMIWAZA_ENV:-development}',
                        'component=observability'
                    ]

                elif service_name == 'grafana':
                    # Update container name with KAMIWAZA_ENV prefix
                    service_config['container_name'] = '${KAMIWAZA_ENV:-default}_kamiwaza-grafana'

                    # Add Grafana provisioning directory and log directory
                    if 'volumes' not in service_config:
                        service_config['volumes'] = []
                    service_config['volumes'].extend([
                        './grafana-provisioning:/etc/grafana/provisioning:ro',
                        '${KAMIWAZA_LOG_DIR:-/tmp}/grafana:/var/log/grafana:rw'
                    ])

                    # Add additional environment variables
                    if 'environment' not in service_config:
                        service_config['environment'] = []

                    service_config['environment'].extend([
                        'GF_PATHS_LOGS=/var/log/grafana',
                        'KAMIWAZA_VERSION=${KAMIWAZA_VERSION:-1.0.0}',
                        'KAMIWAZA_ENV=${KAMIWAZA_ENV:-development}',
                        'KAMIWAZA_LOG_DIR=${KAMIWAZA_LOG_DIR:-/tmp}',
                        'LOKI_ENDPOINT=http://loki:3100',
                        'JAEGER_ENDPOINT=${JAEGER_ENDPOINT:-http://jaeger:16686}',
                        'KAMIWAZA_JAEGER_ENABLED=${JAEGER_ENABLED:-false}'
                    ])

                    # Update depends_on to use health check
                    service_config['depends_on'] = {
                        'loki': {'condition': 'service_healthy'}
                    }

                    # Add resource limits
                    service_config['deploy'] = {
                        'resources': {
                            'limits': {
                                'memory': '512M',
                                'cpus': '0.5'
                            },
                            'reservations': {
                                'memory': '128M',
                                'cpus': '0.1'
                            }
                        }
                    }

                    # Add health check
                    service_config['healthcheck'] = {
                        'test': ['CMD', 'wget', '--no-verbose', '--tries=1', '--spider', 'http://localhost:3000/api/health'],
                        'interval': '30s',
                        'timeout': '10s',
                        'retries': 3,
                        'start_period': '30s'
                    }

                    # Add logging configuration
                    service_config['logging'] = {
                        'driver': 'json-file',
                        'options': {
                            'max-size': '10m',
                            'max-file': '3',
                            'labels': 'service,version,environment'
                        }
                    }

                    # Add labels
                    service_config['labels'] = [
                        'service=grafana',
                        'version=10.2.2',
                        'environment=${KAMIWAZA_ENV:-development}',
                        'component=observability'
                    ]

        # Update volumes with KAMIWAZA_ENV prefix
        if 'volumes' in data:
            for volume_name in list(data['volumes'].keys()):
                volume_config = data['volumes'][volume_name]
                # Add name with environment prefix
                if volume_name == 'loki-data':
                    volume_config['name'] = '${KAMIWAZA_ENV:-default}_loki-data'
                elif volume_name == 'grafana-data':
                    volume_config['name'] = '${KAMIWAZA_ENV:-default}_grafana-data'

        # Add networks section
        if 'networks' not in data:
            data['networks'] = {}
        data['networks']['default'] = {
            'name': 'kamiwaza-observability'
        }

        # Write remapped file
        with open(output_file, "w", encoding="utf-8") as outfile:
            yaml.dump(data, outfile, Dumper=MyDumper, default_flow_style=False, sort_keys=False)

        print(f"Successfully remapped {input_file} to {output_file}")

    except Exception as e:
        print(f"Error processing file: {e}", file=sys.stderr)
        sys.exit(1)

if __name__ == "__main__":
    script_dir = os.path.dirname(os.path.abspath(__file__))
    input_file = os.path.join(script_dir, "vend-docker-compose.yml")
    output_file = os.path.join(script_dir, "remapped-docker-compose.yml")

    process_file(input_file, output_file)
