#!/usr/bin/env python3

"""Remap vendored docker-compose file to use Kamiwaza container names."""

import os
import importlib.util
from typing import Dict

import yaml

REMAPPER_PATH = os.path.abspath(os.path.join(os.path.dirname(__file__), "..", "..", "remapper.py"))
spec = importlib.util.spec_from_file_location("remapper", REMAPPER_PATH)
remapper = importlib.util.module_from_spec(spec)
assert spec.loader is not None
spec.loader.exec_module(remapper)


def _container_name(service_name: str) -> str:
    normalized = service_name.replace("_", "-")
    return f"${{KAMIWAZA_ENV:-default}}_kamiwaza-{normalized}"


def _remap_services(services: Dict[str, Dict]) -> None:
    for service_name, config in services.items():
        config["container_name"] = _container_name(service_name)


def main() -> None:
    input_file = "vend-docker-compose.yml"
    output_file = "remapped-docker-compose.yml"

    remapper.file_exists(input_file)

    with open(input_file, "r", encoding="utf-8") as infile:
        data = yaml.safe_load(infile)

    services = data.get("services")
    if isinstance(services, dict):
        _remap_services(services)

    with open(output_file, "w", encoding="utf-8") as outfile:
        yaml.safe_dump(data, outfile, default_flow_style=False, sort_keys=False, indent=2)

    print(f"Remapped container names written to {output_file}")


if __name__ == "__main__":
    main()
