#!/usr/bin/env bash
set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

if [[ -z "${KAMIWAZA_ROOT:-}" ]]; then
  ROOT_SCRIPT="${SCRIPT_DIR}/set-kamiwaza-root.sh"
  if [[ -f "${ROOT_SCRIPT}" ]]; then
    # shellcheck disable=SC1090
    source "${ROOT_SCRIPT}"
  else
    echo "KAMIWAZA_ROOT is not set and set-kamiwaza-root.sh not found in ${SCRIPT_DIR}" >&2
    exit 1
  fi
fi

# shellcheck disable=SC1090
source "${KAMIWAZA_ROOT}/common.sh"

LOG_PREFIX="KEYCLOAK-PRELAUNCH"

env_file="${KAMIWAZA_ROOT}/env.sh"
if [[ -f "${env_file}" ]]; then
  # shellcheck disable=SC1090
  source "${env_file}"
fi

secret_dir="${KAMIWAZA_ROOT}/runtime/secrets"
secret_file="${secret_dir}/keycloak-admin-password"
mkdir -p "${secret_dir}"
chmod 700 "${secret_dir}"

generate_admin_secret() {
  if command -v python3 >/dev/null 2>&1; then
    python3 - <<'PY'
import secrets
print(secrets.token_urlsafe(24))
PY
  elif command -v python >/dev/null 2>&1; then
    python - <<'PY'
import secrets
print(secrets.token_urlsafe(24))
PY
  elif command -v openssl >/dev/null 2>&1; then
    openssl rand -base64 24 | tr -d '\n'
  else
    dd if=/dev/urandom bs=24 count=1 2>/dev/null | hexdump -ve '1/1 "%02x"'
  fi
}

read_secret_file() {
  if [[ -s "${secret_file}" ]]; then
    head -n 1 "${secret_file}"
    return 0
  fi
  return 1
}

write_secret_file() {
  local value="$1"
  local tmp_file="${secret_file}.tmp"
  printf '%s\n' "${value}" > "${tmp_file}"
  chmod 600 "${tmp_file}"
  mv "${tmp_file}" "${secret_file}"
}

secret_status="existing"
if [[ "${KAMIWAZA_SKIP_SECRET_BOOTSTRAP:-}" == "true" ]]; then
  if [[ -z "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD:-}" ]]; then
    if KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD="$(read_secret_file)"; then
      export KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD
    fi
  fi
  if [[ -z "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD:-}" ]]; then
    log_error "KAMIWAZA_SKIP_SECRET_BOOTSTRAP=true but no Keycloak admin password is available. Set KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD or populate ${secret_file}."
    exit 1
  fi
else
  if [[ -z "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD:-}" ]]; then
    if KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD="$(read_secret_file)"; then
      export KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD
    fi
  fi

  if [[ -z "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD:-}" ]]; then
    if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
      KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD="${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD:-${KEYCLOAK_ADMIN_PASSWORD:-kamiwaza}}"
      secret_status="existing"
    else
      KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD="$(generate_admin_secret)"
      if [[ -z "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}" ]]; then
        log_error "Failed to generate Keycloak admin password"
        exit 1
      fi
      secret_status="generated"
    fi
  fi
fi

write_secret_file "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}"

set_env_value "KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD" "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}"
export KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD="${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}"
set_env_value "KEYCLOAK_ADMIN_PASSWORD" "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}"
export KEYCLOAK_ADMIN_PASSWORD="${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}"

record_secret_source "KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD" "${secret_status}"
record_secret_source "KEYCLOAK_ADMIN_PASSWORD" "${secret_status}"
log_info "KEYCLOAK_ADMIN_PASSWORD source: $(get_secret_source KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD)"

env_target="${KAMIWAZA_ENV_FILE_PATH:-${env_file}}"
log_success "Keycloak admin user is 'admin'. Password stored in ${env_target} and ${secret_file}."

if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
  community_hint_file="${KAMIWAZA_ROOT}/keycloak-admin-password.txt"
  original_umask=$(umask)
  umask 077
  printf "Keycloak admin password (generated on %s): %s\n" \
    "$(date -u +"%Y-%m-%dT%H:%M:%SZ")" "${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}" > "${community_hint_file}"
  umask "${original_umask}"
  log_success "Keycloak admin password (copy/paste): ${KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD}"
  log_success "Password also written to ${community_hint_file} for easy retrieval."
else
  log_success "Tip: run \"grep KAMIWAZA_KEYCLOAK_ADMIN_PASSWORD ${env_target}\" to view it."
fi
