#!/bin/bash
set -euo pipefail

# Logging functions for clean output
log_info() {
    echo "[ETCD-PRELAUNCH-INFO] $1"
}

log_warn() {
    echo "[ETCD-PRELAUNCH-WARN] $1"
}

log_error() {
    echo "[ETCD-PRELAUNCH-ERROR] $1"
}

log_success() {
    echo "[ETCD-PRELAUNCH-SUCCESS] $1"
}

log_step() {
    echo ""
    echo "=========================================="
    echo "ETCD PRELAUNCH STEP: $1"
    echo "=========================================="
}

log_step "Initializing etcd prelaunch setup"

# Create necessary directories for certificates, secrets, and etcd volumes
log_info "Creating required directories..."
mkdir -p ./certs
mkdir -p ./config
mkdir -p ./secrets
mkdir -p ./volumes/etcd
log_success "Directories created successfully"

log_step "Determining reliable hostname for certificate management"

# Use hostname as single source of truth across all platforms (macOS, Linux, WSL)
CURRENT_HOSTNAME=$(hostname)
if [[ -z "$CURRENT_HOSTNAME" ]]; then
    log_error "Failed to determine hostname using 'hostname' command"
    exit 1
fi
log_info "Detected current hostname: $CURRENT_HOSTNAME"

# Check for existing certificates with different hostnames and clean them up
log_info "Checking for certificates with mismatched hostnames..."
MISMATCHED_CERTS_FOUND=false
for cert_file in certs/peer-*-key.pem; do
    if [[ -f "$cert_file" ]]; then
        # Extract hostname from certificate filename: peer-<hostname>-key.pem
        cert_hostname=$(basename "$cert_file" | sed 's/^peer-\(.*\)-key\.pem$/\1/')
        if [[ "$cert_hostname" != "$CURRENT_HOSTNAME" ]]; then
            log_warn "Found certificate for different hostname: $cert_hostname (current: $CURRENT_HOSTNAME)"
            MISMATCHED_CERTS_FOUND=true
            break
        else
            log_info "Found certificate for correct hostname: $cert_hostname"
        fi
    fi
done

# Remove all certificates if hostname mismatch detected
if [[ "$MISMATCHED_CERTS_FOUND" == "true" ]]; then
    log_warn "Hostname mismatch detected - removing all certificates to force regeneration"
    rm -f certs/peer-* certs/ca.* certs/$(hostname).* 2>/dev/null || true
    rm -f certs/peer-* certs/ca-key.* certs/$(hostname).* 2>/dev/null || true
    rm -f "${KAMIWAZA_ROOT}/runtime/etcd/certs/"* 2>/dev/null || true
    log_success "Old certificates removed successfully"
else
    log_info "No hostname mismatch detected"
fi

# Set KAMIWAZA_ETCD_NODE_NAME to current hostname
export KAMIWAZA_ETCD_NODE_NAME="$CURRENT_HOSTNAME"
log_info "Set KAMIWAZA_ETCD_NODE_NAME to: ${KAMIWAZA_ETCD_NODE_NAME}"

# Default values for CA configuration
: ${KAMIWAZA_CA_COMMON_NAME:="KAMIWAZA CA"}
: ${KAMIWAZA_CA_ORGANIZATION:="KAMIWAZA Inc."}
: ${KAMIWAZA_CA_ORGANIZATIONAL_UNIT:="Certificate Authority"}
: ${KAMIWAZA_CA_LOCALITY:="Internet"}
: ${KAMIWAZA_CA_STATE:="Digital"}
: ${KAMIWAZA_CA_COUNTRY:="US"}

log_step "Checking for existing certificates"

# Check for existing certificates (updated to use hostname)
CERT_FILES=(
 "certs/ca.pem" 
 "certs/ca-key.pem" 
 "certs/ca.csr" 
 "certs/$(hostname)-key.pem" 
 "certs/$(hostname).csr" 
 "certs/$(hostname).pem" 
 "certs/peer-$(hostname)-key.pem" 
 "certs/peer-$(hostname).csr" 
 "certs/peer-$(hostname).pem"
)
EXISTING_CERTS=()
for FILE in "${CERT_FILES[@]}"; do
 if [[ -f "$FILE" ]]; then
   EXISTING_CERTS+=("$FILE")
   log_info "Found existing certificate: $FILE"
 fi
done

if [[ ${#EXISTING_CERTS[@]} -eq 0 ]]; then
    log_info "No existing certificates found"
else
    log_info "Found ${#EXISTING_CERTS[@]} existing certificates"
fi

# Function to check if node is already in the etcd cluster
check_node_in_cluster() {
    local node_name="${KAMIWAZA_ENV:-default}_kamiwaza-etcd-${KAMIWAZA_ETCD_NODE_NAME}"
    
    # SSH into head node and check member list
    if ssh -i /etc/kamiwaza/ssl/cluster.key \
        -o StrictHostKeyChecking=no \
        -o UserKnownHostsFile=/dev/null \
        "${USER}@${KAMIWAZA_HEAD_IP}" \
        "docker exec ${KAMIWAZA_ENV:-default}_kamiwaza-etcd-\$(hostname) etcdctl \
        --cert=\"/etc/etcd/certs/peer-\$(hostname).pem\" \
        --key=\"/etc/etcd/certs/peer-\$(hostname)-key.pem\" \
        --cacert=/etc/etcd/certs/ca.pem \
        member list -w table | grep -q \"$node_name\""; then
        
        return 0  # Node found in cluster
    else
        return 1  # Node not found in cluster
    fi
}

log_step "Setting up KAMIWAZA_ROOT and runtime directories"

# Get the absolute path of the current script
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" &> /dev/null && pwd)"
log_info "Script directory: $SCRIPT_DIR"

# Find the KAMIWAZA_ROOT folder by navigating up the directory tree
KAMIWAZA_ROOT="$SCRIPT_DIR"
while [[ "$KAMIWAZA_ROOT" != "/" && ! -f "$KAMIWAZA_ROOT/launch.py" && ! -f "$KAMIWAZA_ROOT/install.py" ]]; do
   KAMIWAZA_ROOT="$(dirname "$KAMIWAZA_ROOT")"
done

# dev layout
if [ -f "${KAMIWAZA_ROOT}/../compile.sh" ]; then
 KAMIWAZA_ROOT="$(dirname "$KAMIWAZA_ROOT")"
fi

# Check if KAMIWAZA_ROOT was found
if [[ ! -f "$KAMIWAZA_ROOT/launch.py" && ! -f "$KAMIWAZA_ROOT/install.py" ]]; then
   log_error "KAMIWAZA_ROOT not found. Please ensure launch.py or install.py exists in the expected directory structure."
   exit 1
fi

log_info "KAMIWAZA_ROOT found: $KAMIWAZA_ROOT"

# Create the KAMIWAZA_ROOT/runtime folder if it doesn't exist
log_info "Creating runtime directory structure..."
mkdir -p "${KAMIWAZA_ROOT}/runtime/etcd/certs"
log_success "Runtime directories created: ${KAMIWAZA_ROOT}/runtime/etcd/certs"

#default
is_community=false

is_osx=false
if [[ "$OSTYPE" == "darwin"* ]]; then
    is_osx=true
fi

log_step "Determining installation mode"

# Determine if we are in community/single-node mode
if [[ -f "${KAMIWAZA_ROOT}/.kamiwaza_install_community" ]]; then
    is_community=true
    log_info "Community mode detected via .kamiwaza_install_community file"
else
    is_community=false
    log_info "Enterprise mode: .kamiwaza_install_community file not found"
fi

if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    is_community=true
    log_info "Community mode set via KAMIWAZA_COMMUNITY environment variable"
fi

if [[ "$is_community" == "true" ]]; then
    log_info "Operating in community/single-node mode"
else
    log_info "Operating in enterprise/cluster mode"
fi


get_system_ips() {
    local max_attempts=5
    local attempt=1
    local ips=""
    
    while [ $attempt -le $max_attempts ]; do
        if [[ "$OSTYPE" == "linux-gnu"* ]]; then
            # Linux system IP extraction with error handling
            if ips=$(ip -4 addr 2>/dev/null | grep -oP '(?<=inet\s)\d+(\.\d+){3}' | grep -v "127.0.0.1"); then
                echo "$ips"
                return 0
            fi
        elif [[ "$OSTYPE" == "darwin"* ]]; then
            # MacOS system IP extraction with error handling
            if ips=$(ifconfig 2>/dev/null | grep inet | grep -v inet6 | grep -v "127.0.0.1" | awk '{print $2}'); then
                echo "$ips"
                return 0
            fi
        else
            echo "Unsupported OS type for IP extraction" >&2
            return 1
        fi
        
        echo "IP extraction failed (attempt $attempt/$max_attempts), retrying..." >&2
        sleep 2
        ((attempt++))
    done
    
    echo "Failed to get system IPs after $max_attempts attempts" >&2
    return 1
}

check_if_head_node() {
  local head_ip="$1"
  local max_attempts=5
  local attempt=1
  
  while [ $attempt -le $max_attempts ]; do
    # Use ip addr instead of ifconfig (more reliable)
    if ip -4 addr 2>/dev/null | grep -q "inet $head_ip/"; then
      return 0  # This is the head node
    fi
    
    # If the check fails, try with ifconfig as backup
    if ifconfig 2>/dev/null | grep inet | awk '{print $2}' | grep -q "^$head_ip$"; then
      return 0  # This is the head node
    fi
    
    echo "Head detection attempt $attempt/$max_attempts failed, retrying..." >&2
    sleep 2
    ((attempt++))
  done
  
  return 1  # This is not the head node
}

# Function to generate ETCD certificates
generate_etcd_certs() {
   log_info "Starting ETCD certificate generation..."
   cat <<EOF > config/ca-config.json
{
   "signing": {
       "default": {
           "expiry": "8760h"
       },
       "profiles": {
           "client": {
               "usages": ["signing", "key encipherment", "client auth"],
               "expiry": "8760h"
           },
           "peer": {
               "usages": ["signing", "key encipherment", "server auth", "client auth"],
               "expiry": "8760h"
           }
       }
   }
}
EOF

   # Write the ca-csr.json file dynamically
   if [[ ! -f "./config/ca-csr.json" ]]; then
       cat <<EOF >./config/ca-csr.json
{
   "CN": "$KAMIWAZA_CA_COMMON_NAME",
   "key": {
       "algo": "ecdsa",
       "size": 384
   },
   "names": [
       {
           "C": "$KAMIWAZA_CA_COUNTRY",
           "ST": "$KAMIWAZA_CA_STATE",
           "L": "$KAMIWAZA_CA_LOCALITY",
           "O": "$KAMIWAZA_CA_ORGANIZATION",
           "OU": "$KAMIWAZA_CA_ORGANIZATIONAL_UNIT"
       }
   ]
}
EOF
   fi

  # Create the req-csr.json file
  create_req_csr_json() {
      local node_name="${KAMIWAZA_ENV:-default}_kamiwaza-etcd-${KAMIWAZA_ETCD_NODE_NAME}"
      local ips=()
      
      # In community mode, only use localhost and container name
      if [[ "$is_community" == "true" ]]; then
          ips+=("127.0.0.1")
          ips+=("localhost")
          ips+=("${node_name}")
          ips+=("etcd")
      else
          # Original IP detection for non-community mode
          if ip_list=$(get_system_ips); then
              while read -r ip; do
                  ips+=("$ip")
              done <<< "$ip_list"
          else
              echo "Warning: Failed to get system IPs, using fallback addresses"
          fi
          ips+=("127.0.0.1")
          ips+=("localhost")
          ips+=("${node_name}")
          ips+=("etcd")
      fi

      # Start the JSON file
      echo "{" > config/req-csr.json
      echo "  \"CN\": \"etcd\"," >> config/req-csr.json
      echo "  \"hosts\": [" >> config/req-csr.json

      # Add each IP and DNS entry as SANs
      local first=true
      for entry in "${ips[@]}"; do
          if [ "$first" = true ]; then
              first=false
          else
              echo "," >> config/req-csr.json
          fi
          echo "    \"$entry\"" >> config/req-csr.json
      done

      # Finish the JSON structure
      echo "  ]," >> config/req-csr.json
      echo "  \"key\": {" >> config/req-csr.json
      echo "    \"algo\": \"ecdsa\"," >> config/req-csr.json
      echo "    \"size\": 384" >> config/req-csr.json
      echo "  }," >> config/req-csr.json
      echo "  \"names\": [" >> config/req-csr.json
      echo "    {" >> config/req-csr.json
      echo "      \"O\": \"autogenerated\"," >> config/req-csr.json
      echo "      \"OU\": \"etcd cluster\"," >> config/req-csr.json
      echo "      \"L\": \"the internet\"" >> config/req-csr.json
      echo "    }" >> config/req-csr.json
      echo "  ]" >> config/req-csr.json
      echo "}" >> config/req-csr.json
  }


   # Call the function to create req-csr.json
   create_req_csr_json

   # Generate CA certificate and key if they don't exist
   if [[ ! -f "./certs/ca.pem" || ! -f "./certs/ca-key.pem" ]]; then
       cfssl gencert -initca config/ca-csr.json | cfssljson -bare certs/ca
   fi

   # Generate peer certificates for this node
   cfssl gencert \
       -ca=certs/ca.pem \
       -ca-key=certs/ca-key.pem \
       -config=config/ca-config.json \
       -profile=peer \
       config/req-csr.json | cfssljson -bare "certs/peer-${KAMIWAZA_ETCD_NODE_NAME}"
}

# Simplified logic for community/single-node mode
if [[ "$is_community" == "true" ]]; then
    log_step "Setting up single-node etcd configuration"
    
    # Check if certificates exist locally
    log_info "Checking for existing certificates in ${SCRIPT_DIR}/certs"
    if ! find "${SCRIPT_DIR}/certs" -name "*.pem" | grep -q .; then
        log_info "No PEM certificates found locally, generating new certificates..."
        generate_etcd_certs
        log_success "Certificate generation completed"
    else
        log_info "Local PEM certificates found, skipping generation"
    fi

    # Check if all required certificates exist before copying
    log_info "Verifying required certificates exist before copying..."
    required_files=(
        "${SCRIPT_DIR}/certs/ca.pem"
        "${SCRIPT_DIR}/certs/ca-key.pem"
        "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}.pem"
        "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem"
    )
    
    for file in "${required_files[@]}"; do
        if [[ ! -f "$file" ]]; then
            log_error "Required certificate file missing: $file"
            log_error "Certificate generation may have failed"
            exit 1
        else
            log_info "✓ Found: $file"
        fi
    done

    # Copy certificates to runtime folder
    log_info "Copying local certificates to runtime folder (overwriting)..."
    if cp -vf "${SCRIPT_DIR}/certs/ca.pem" "${SCRIPT_DIR}/certs/ca-key.pem" "${KAMIWAZA_ROOT}/runtime/etcd/certs/"; then
        log_success "CA certificates copied successfully"
    else
        log_error "Failed to copy CA certificates"
        exit 1
    fi
    
    if cp -vf "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}.pem" "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem" \
        "${KAMIWAZA_ROOT}/runtime/etcd/certs/"; then
        log_success "Peer certificates copied successfully"
    else
        log_error "Failed to copy peer certificates"
        exit 1
    fi

    # Set single-node cluster configuration
    export KAMIWAZA_ETCD_ADVERTISE_PEER_URLS="https://${KAMIWAZA_ENV:-default}_kamiwaza-etcd-${KAMIWAZA_ETCD_NODE_NAME}:2380"
    export KAMIWAZA_ETCD_ADVERTISE_CLIENT_URLS="https://${KAMIWAZA_ENV:-default}_kamiwaza-etcd-${KAMIWAZA_ETCD_NODE_NAME}:2379"
    export KAMIWAZA_ETCD_INITIAL_CLUSTER="${KAMIWAZA_ENV:-default}_kamiwaza-etcd-${KAMIWAZA_ETCD_NODE_NAME}=https://${KAMIWAZA_ENV:-default}_kamiwaza-etcd-${KAMIWAZA_ETCD_NODE_NAME}:2380"
    export KAMIWAZA_ETCD_CLUSTER_STATE="new"
else
    # Determine if we are the head node
    if [[ -n "${KAMIWAZA_HEAD_IP:-}" ]]; then
        if check_if_head_node "${KAMIWAZA_HEAD_IP}"; then
            is_head=true
        else
            is_head=false
        fi
    else
        is_head=true  # No head IP means we're standalone/head
    fi

    if [[ "$is_head" == "true" ]]; then
        echo "Operating as head node..."
        
        # Check if certs exist in runtime but not locally
        if ! find "${SCRIPT_DIR}/certs" -name "*.pem" | grep -q . && \
           find "${KAMIWAZA_ROOT}/runtime/etcd/certs" -name "*.pem" | grep -q .; then
            echo "Found certificates in runtime, copying to local..."
            cp -v "${KAMIWAZA_ROOT}/runtime/etcd/certs/ca.pem" "${KAMIWAZA_ROOT}/runtime/etcd/certs/ca-key.pem" "${SCRIPT_DIR}/certs/"
            cp -v "${KAMIWAZA_ROOT}/runtime/etcd/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}.pem" \
                "${KAMIWAZA_ROOT}/runtime/etcd/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem" "${SCRIPT_DIR}/certs/"
        fi

        # Generate certs if they still don't exist locally
        if ! find "${SCRIPT_DIR}/certs" -name "*.pem" | grep -q .; then
            generate_etcd_certs
        fi

        # Copy local certs to runtime
        cp -v "${SCRIPT_DIR}/certs/ca.pem" "${SCRIPT_DIR}/certs/ca-key.pem" "${KAMIWAZA_ROOT}/runtime/etcd/certs/"
        cp -v "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}.pem" "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem" \
            "${KAMIWAZA_ROOT}/runtime/etcd/certs/"

    else
        echo "Operating as worker node..."
        # Ensure cluster key exists and has right permissions
        if [[ ! -f "/etc/kamiwaza/ssl/cluster.key" ]]; then
            echo "Error: cluster.key not found"
            exit 1
        fi
        chmod 600 /etc/kamiwaza/ssl/cluster.key

        # Function to check if head node's etcd is ready
        check_head_etcd_ready() {
            ssh -i /etc/kamiwaza/ssl/cluster.key \
                -o StrictHostKeyChecking=no \
                -o UserKnownHostsFile=/dev/null \
                "${USER}@${KAMIWAZA_HEAD_IP}" \
                "docker ps --filter name=kamiwaza-etcd --format '{{.Status}}' | grep -q 'Up'"
            return $?
        }

        # Wait for head node's etcd to be ready
        echo "Waiting for head node etcd to be ready..."
        max_head_attempts=60  # 30 minutes with 30-second sleep
        head_attempt=1
        while [[ $head_attempt -le $max_head_attempts ]]; do
            if check_head_etcd_ready; then
                echo "Head node etcd is ready"
                break
            fi
            echo "Head node etcd not ready yet (attempt $head_attempt/$max_head_attempts)..."
            sleep 30
            ((head_attempt++))
        done

        if [[ $head_attempt -gt $max_head_attempts ]]; then
            echo "Error: Head node etcd failed to become ready after $(( max_head_attempts * 30 )) seconds"
            exit 1
        fi

        # Get CA certificates from head node with improved retry logic
        echo "Fetching CA certificates from head node..."
        max_attempts=10
        attempt=1
        while [[ $attempt -le $max_attempts ]]; do
            if scp -i /etc/kamiwaza/ssl/cluster.key \
                -o StrictHostKeyChecking=no \
                -o UserKnownHostsFile=/dev/null \
                -o ConnectTimeout=10 \
                "${USER}@${KAMIWAZA_HEAD_IP}:${SCRIPT_DIR}/certs/ca.pem" \
                "${USER}@${KAMIWAZA_HEAD_IP}:${SCRIPT_DIR}/certs/ca-key.pem" \
                "${SCRIPT_DIR}/certs/" 2>/dev/null; then
                
                echo "Successfully retrieved CA certificates from deployment path"
                
                # Copy CA certs to runtime
                cp -v "${SCRIPT_DIR}/certs/ca.pem" "${SCRIPT_DIR}/certs/ca-key.pem" \
                    "${KAMIWAZA_ROOT}/runtime/etcd/certs/"
                
                break
            fi
            
            echo "Attempt $attempt failed. Waiting before retry..."
            sleep 15
            ((attempt++))
        done
        
        if [[ $attempt -gt $max_attempts ]]; then
            echo "Failed to retrieve CA certificates after $max_attempts attempts"
            exit 1
        fi

        if check_node_in_cluster; then
            echo "Node is already in etcd cluster, preserving existing peer certificates..."
            
            # Only fetch CA certificates if they don't exist
            if [[ ! -f "${SCRIPT_DIR}/certs/ca.pem" || ! -f "${SCRIPT_DIR}/certs/ca-key.pem" ]]; then
                # Get CA certificates from head node with retry logic
                echo "Fetching CA certificates from head node..."
            fi
            
            # Make sure peer certificates exist in runtime
            if [[ -f "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}.pem" && 
                -f "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem" ]]; then
                # Copy existing peer certs to runtime if needed
                cp -v "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}.pem" \
                    "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem" \
                    "${KAMIWAZA_ROOT}/runtime/etcd/certs/"
            else
                echo "Error: Node is in cluster but peer certificates are missing"
                exit 1
            fi
        else
            echo "Node not found in etcd cluster, generating new peer certificates..."
            generate_etcd_certs
            
            # Copy to runtime
            cp -vf "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}.pem" \
                "${SCRIPT_DIR}/certs/peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem" \
                "${KAMIWAZA_ROOT}/runtime/etcd/certs/"
        fi

    fi
fi

# Verify required certificates exist in both locations
required_certs=(
   "ca.pem"
   "peer-${KAMIWAZA_ETCD_NODE_NAME}.pem"
   "peer-${KAMIWAZA_ETCD_NODE_NAME}-key.pem"
)

for cert in "${required_certs[@]}"; do
   if [[ ! -f "${SCRIPT_DIR}/certs/$cert" ]]; then
       echo "Error: Required certificate $cert not found in local certs"
       exit 1
   fi
   if [[ ! -f "${KAMIWAZA_ROOT}/runtime/etcd/certs/$cert" ]]; then
       echo "Error: Required certificate $cert not found in runtime certs"
       echo "KAMIWAZA_ROOT: ${KAMIWAZA_ROOT}"
       exit 1
   fi
done

echo "ETCD certificates validated successfully"

# Configure etcd cluster membership
if [[ "$is_community" != "true" ]]; then
    source "${KAMIWAZA_ROOT}/etcd_cluster_manager.sh"
    build_cluster_config
fi

echo "Prelaunch check for etcd with the following configuration:"
echo "  Node Name: ${KAMIWAZA_ETCD_NODE_NAME:-Not Set}"
echo "  Advertise Peer URLs: ${KAMIWAZA_ETCD_ADVERTISE_PEER_URLS:-Not Set}"
echo "  Advertise Client URLs: ${KAMIWAZA_ETCD_ADVERTISE_CLIENT_URLS:-Not Set}" 
echo "  Initial Cluster: ${KAMIWAZA_ETCD_INITIAL_CLUSTER:-Not Set}"
echo "  Cluster State: ${KAMIWAZA_ETCD_CLUSTER_STATE:-Not Set}"
echo "  Mode: $(if [[ "$is_community" == "true" ]]; then echo "Community/Single-node"; else echo "Enterprise/Cluster"; fi)"
