#!/usr/bin/env python3
import os
import sys
from typing import Dict, Any
import importlib.util
import yaml

# Custom YAML representer to handle multi-line strings properly
class MultilineLiteral(str):
    pass

def multiline_literal_representer(dumper, data):
    return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='|')

yaml.add_representer(MultilineLiteral, multiline_literal_representer)

# Use importlib to import remapper module
remapper_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'remapper.py'))
spec = importlib.util.spec_from_file_location("remapper", remapper_path)
remapper = importlib.util.module_from_spec(spec)
spec.loader.exec_module(remapper)

# Define update services for dependency fixes
UPDATE_SERVICES = {
    'system-update-quickstart'
}

def adjust_resource_limits(service_config: Dict[str, Any], service_name: str) -> None:
    """
    Adjust resource limits based on KAMIWAZA_COMMUNITY environment variable.
    If not community edition, increase resource limits for better performance.
    """
    is_community = os.environ.get('KAMIWAZA_COMMUNITY', 'false').lower() == 'true'
    
    def update_environment(service_config: Dict[str, Any], key: str, value: str) -> None:
        """Helper to update environment variables regardless of format (list or dict)"""
        if 'environment' not in service_config:
            service_config['environment'] = {}
        
        env = service_config['environment']
        
        # If it's a list, convert to dict first
        if isinstance(env, list):
            env_dict = {}
            for item in env:
                if '=' in item:
                    k, v = item.split('=', 1)
                    env_dict[k] = v
            service_config['environment'] = env_dict
            env = env_dict
        
        # Now we can safely update the dict
        env[key] = value
    
    if not is_community:
        # Adjust MySQL resources
        if service_name == 'mysql':
            # Increase innodb buffer pool size for better performance
            update_environment(service_config, 'MYSQL_INNODB_BUFFER_POOL_SIZE', '2G')
            
        # Adjust OpenSearch resources
        elif service_name == 'opensearch':
            if 'deploy' not in service_config:
                service_config['deploy'] = {}
            if 'resources' not in service_config['deploy']:
                service_config['deploy']['resources'] = {}
            if 'limits' not in service_config['deploy']['resources']:
                service_config['deploy']['resources']['limits'] = {}
            # Increase memory limit for OpenSearch
            service_config['deploy']['resources']['limits']['memory'] = '4294967296'  # 4GB
            
            # Adjust JVM heap sizes
            update_environment(service_config, 'ES_JAVA_OPTS', '-Xms2g -Xmx2g -Dlog4j2.formatMsgNoLookups=true')
            update_environment(service_config, 'OPENSEARCH_JAVA_OPTS', '-Xms2g -Xmx2g -Dlog4j2.formatMsgNoLookups=true')
            
        # Adjust Kafka resources
        elif service_name == 'kafka-broker':
            # Increase Kafka heap
            update_environment(service_config, 'KAFKA_HEAP_OPTS', '-Xms1g -Xmx1g')
            
        # Adjust GMS resources
        elif 'gms' in service_name:
            # Increase GMS heap
            update_environment(service_config, 'JAVA_OPTS', '-Xms2g -Xmx2g')



def fix_kafka_listeners(service_config: Dict[str, Any], service_name: str) -> None:
    """
    Fix Kafka to listen on all network interfaces instead of specific hostname.
    This solves the issue where Kafka is accessible on one network but not another.
    """
    if service_name == 'kafka-broker' and 'environment' in service_config:
        env = service_config['environment']
        if isinstance(env, dict) and 'KAFKA_LISTENERS' in env:
            # Replace broker hostname with 0.0.0.0 to bind to all interfaces
            original = env['KAFKA_LISTENERS']
            env['KAFKA_LISTENERS'] = original.replace('://broker:', '://0.0.0.0:')
            if original != env['KAFKA_LISTENERS']:
                print(f"  Fixed Kafka listeners: {original} -> {env['KAFKA_LISTENERS']}")

def fix_kafka_command(service_config: Dict[str, Any], service_name: str) -> None:
    """
    Fix the Kafka command to use proper YAML multi-line format
    """
    if service_name == 'kafka-broker' and 'command' in service_config:
        command = service_config['command']
        if isinstance(command, list) and len(command) == 3 and command[0] == '/bin/bash':
            # The third element contains the script
            script = command[2]
            # Ensure the script uses literal style for proper formatting
            service_config['command'] = ['/bin/bash', '-c', MultilineLiteral(script)]

def fix_update_dependencies(service_config: Dict[str, Any], service_name: str) -> None:
    """
    Change required: true to required: false for dependencies on update services.
    This allows services to start even if update services haven't completed.
    """
    if 'depends_on' in service_config:
        depends_on = service_config['depends_on']
        if isinstance(depends_on, dict):
            for dep_name, dep_config in depends_on.items():
                if dep_name in UPDATE_SERVICES and isinstance(dep_config, dict):
                    if dep_config.get('required') is True:
                        dep_config['required'] = False
                        print(f"  Changed {service_name} dependency on {dep_name}: required: true -> false")

def fix_dependencies(service_config: Dict[str, Any], service_name: str) -> None:
    """
    Keep dependencies as-is since all services will have the setup profile.
    """
    # No changes needed to dependencies since all services have setup profile
    pass

def fix_kafka_command(service_config: Dict[str, Any], service_name: str) -> None:
    """
    Fix the Kafka command to use proper YAML multi-line format
    """
    if service_name == 'kafka-broker' and 'command' in service_config:
        command = service_config['command']
        if isinstance(command, list) and len(command) == 3 and command[0] == '/bin/bash':
            # The third element contains the script
            script = command[2]
            # Ensure the script uses literal style for proper formatting
            service_config['command'] = ['/bin/bash', '-c', MultilineLiteral(script)]

def process_file(input_file: str, output_file: str) -> None:
    try:
        with open(input_file, "r", encoding="utf-8") as infile:
            data = yaml.safe_load(infile)
        
        # Process services
        if 'services' in data:
            for service_name, service_config in data['services'].items():
                # Handle case where service_config might be None
                if service_config is None:
                    service_config = {}
                    data['services'][service_name] = service_config
                
                # Adjust resource limits based on community flag
                adjust_resource_limits(service_config, service_name)
                
                # Fix Kafka to listen on all interfaces
                fix_kafka_listeners(service_config, service_name)
                
                # Fix Kafka command formatting
                fix_kafka_command(service_config, service_name)
                
                # Fix dependencies on update services
                fix_update_dependencies(service_config, service_name)
                
                # Fix environment variables with defaults
                if 'environment' in service_config:
                    env = service_config['environment']
                    if isinstance(env, dict):
                        # Check for UI_INGESTION_DEFAULT_CLI_VERSION
                        if 'UI_INGESTION_DEFAULT_CLI_VERSION' in env:
                            # Replace with version that has a default
                            if env['UI_INGESTION_DEFAULT_CLI_VERSION'] == '${UI_INGESTION_DEFAULT_CLI_VERSION}':
                                env['UI_INGESTION_DEFAULT_CLI_VERSION'] = '${UI_INGESTION_DEFAULT_CLI_VERSION:-0.13.3}'
                
                # Fix environment variables with defaults
                if 'environment' in service_config:
                    env = service_config['environment']
                    if isinstance(env, dict):
                        # Check for UI_INGESTION_DEFAULT_CLI_VERSION
                        if 'UI_INGESTION_DEFAULT_CLI_VERSION' in env:
                            # Replace with version that has a default
                            if env['UI_INGESTION_DEFAULT_CLI_VERSION'] == '${UI_INGESTION_DEFAULT_CLI_VERSION}':
                                env['UI_INGESTION_DEFAULT_CLI_VERSION'] = '${UI_INGESTION_DEFAULT_CLI_VERSION:-0.13.3}'
                
                # Fix Kafka configuration issues
                if service_name == 'kafka-setup' and 'environment' in service_config:
                    # Remove Zookeeper reference from kafka-setup
                    if 'KAFKA_ZOOKEEPER_CONNECT' in service_config['environment']:
                        del service_config['environment']['KAFKA_ZOOKEEPER_CONNECT']
                
                # Update networks configuration
                # Handle the new format where networks might be a dict with 'default: null'
                if 'networks' in service_config:
                    # If it's a dict (new format), convert to list
                    if isinstance(service_config['networks'], dict):
                        service_config['networks'] = [
                            "datahub-network",
                            "kamiwaza-backend"
                        ]
                    # If it's already a list, replace it
                    elif isinstance(service_config['networks'], list):
                        service_config['networks'] = [
                            "datahub-network",
                            "kamiwaza-backend"
                        ]
                else:
                    # No networks defined, add them
                    service_config['networks'] = [
                        "datahub-network",
                        "kamiwaza-backend"
                    ]
        
        # Update networks section
        data['networks'] = {
            'datahub-network': {
                'name': '${KAMIWAZA_ENV:-default}_datahub-network'
            },
            'kamiwaza-backend': {
                'name': '${KAMIWAZA_ENV:-default}_kamiwaza-backend',
                'external': True
            }
        }
        
        # Update volumes section to use consistent naming
        if 'volumes' in data:
            for volume_name, volume_config in data['volumes'].items():
                if isinstance(volume_config, dict) and 'name' in volume_config:
                    # Update volume names to include the environment prefix
                    base_name = volume_config['name']
                    if not base_name.startswith('${KAMIWAZA_ENV'):
                        # Extract just the volume name part (e.g., "broker" from "datahub_broker")
                        if '_' in base_name:
                            base_name = base_name.split('_', 1)[1]
                        volume_config['name'] = f'${{KAMIWAZA_ENV:-default}}_datahub_{base_name}'
        
        # Write the updated YAML with better formatting
        with open(output_file, "w", encoding="utf-8") as outfile:
            yaml.dump(
                data, 
                outfile, 
                default_flow_style=False,
                sort_keys=False,  # Preserve order
                width=120,  # Wider lines
                indent=2  # 2-space indent
            )
        
        # Print summary of changes
        print(f"\nUpdate services dependency fixes applied")
        
    except Exception as e:
        print(f"An error occurred while processing the file: {str(e)}", file=sys.stderr)
        raise

def verify_changes(output_file: str) -> None:
    """
    Verify that the changes were applied correctly.
    """
    try:
        with open(output_file, "r", encoding="utf-8") as f:
            data = yaml.safe_load(f)
        
        print("\nVerification:")
        
        # Check Kafka listener configuration
        if 'services' in data and 'kafka-broker' in data['services']:
            kafka_env = data['services']['kafka-broker'].get('environment', {})
            if isinstance(kafka_env, dict) and 'KAFKA_LISTENERS' in kafka_env:
                listeners = kafka_env['KAFKA_LISTENERS']
                if '0.0.0.0' in listeners:
                    print("\n✓ Kafka listeners configured to bind to all interfaces")
                else:
                    print(f"\n⚠️  Kafka listeners may have binding issues: {listeners}")
        
        # Verify all services have profiles
        if other_count > 0:
            print("\n⚠️  WARNING: Some services have incorrect or missing profiles!")
        else:
            print("\n✓ All services have correct profiles")
        
        # Check network configuration
        if 'networks' in data:
            print("\nNetwork configuration:")
            if 'datahub-network' in data['networks']:
                print("✓ datahub-network configured")
            if 'kamiwaza-backend' in data['networks']:
                print("✓ kamiwaza-backend configured (external)")
        
        # Check volume configuration
        if 'volumes' in data:
            print("\nVolumes:")
            for vol_name, vol_config in data['volumes'].items():
                if isinstance(vol_config, dict) and 'name' in vol_config:
                    print(f"✓ {vol_name} -> {vol_config['name']}")
        
        # Check volume configuration
        if 'volumes' in data:
            print("\nVolumes:")
            for vol_name, vol_config in data['volumes'].items():
                if isinstance(vol_config, dict) and 'name' in vol_config:
                    print(f"✓ {vol_name} -> {vol_config['name']}")
        
    except Exception as e:
        print(f"Verification failed: {str(e)}", file=sys.stderr)

def main() -> None:
    """
    Main function to process the vend-docker-compose.yml file.
    """
    input_file = "vend-docker-compose.yml"
    output_file = "remapped-docker-compose.yml"
    
    remapper.file_exists(input_file)
    process_file(input_file, output_file)
    print("Processing complete. Output saved in", output_file)
    
    # Verify the changes
    verify_changes(output_file)

if __name__ == "__main__":
    main()
