#!/usr/bin/env python3

import os
import sys
import re
from typing import List
import importlib.util
import yaml

# Use importlib to import remapper module
remapper_path = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..', 'remapper.py'))
spec = importlib.util.spec_from_file_location("remapper", remapper_path)
remapper = importlib.util.module_from_spec(spec)
spec.loader.exec_module(remapper)

def process_file(input_file: str, output_file: str) -> None:
    try:
        with open(input_file, "r", encoding="utf-8") as infile:
            data = yaml.safe_load(infile)

        # Process services
        if 'services' in data:
            for service_name, service_config in data['services'].items():
                if 'networks' not in service_config:
                    service_config['networks'] = []
                if f"{service_name}" not in service_config['networks']:
                    service_config['networks'].append(f"{service_name}")
                if "kamiwaza-backend" not in service_config['networks']:
                    service_config['networks'].append("kamiwaza-backend")

        # Add or update networks section
        if 'networks' not in data:
            data['networks'] = {}
        for service_name in data.get('services', {}):
            data['networks'][service_name] = {
                'name': f'${{KAMIWAZA_ENV:-default}}-{service_name}'
            }
        data['networks']['kamiwaza-backend'] = {
            'name': '${KAMIWAZA_ENV:-default}_kamiwaza-backend',
            'external': True
        }

        # Write the updated YAML
        with open(output_file, "w", encoding="utf-8") as outfile:
            yaml.dump(data, outfile, default_flow_style=False)

    except Exception as e:
        print(f"An error occurred while processing the file: {str(e)}", file=sys.stderr)
        raise

def main() -> None:
    """
    Main function to process the vend-docker-compose.yml file.
    """
    input_file = "vend-docker-compose.yml"
    output_file = "remapped-docker-compose.yml"
    remapper.file_exists(input_file)
    process_file(input_file, output_file)
    print("Processing complete. Output saved in", output_file)

if __name__ == "__main__":
    main()
