#!/bin/bash
set -euo pipefail

# Set component-specific log prefix
LOG_PREFIX="COCKROACHDB-PRELAUNCH"

# Source the common logging functions
if [ -f "./common-logging.sh" ]; then
    source ./common-logging.sh
else
    # Fallback if common-logging.sh not found
    log_info() { echo "[INFO] $1"; }
    log_warn() { echo "[WARN] $1"; }
    log_error() { echo "[ERROR] $1"; }
    log_success() { echo "[SUCCESS] $1"; }
    log_step() { echo ""; echo "========== $1 =========="; }
fi

log_step "Setting up CockroachDB certificates"

# Set up directories for certificates and keys
log_info "Creating certificate directories..."
mkdir -p certs
mkdir -p secrets
mkdir -p volumes

# Set restrictive permissions on cert directories
chmod 700 certs secrets
log_success "Certificate directories created with proper permissions"

# Create the Certificate Authority (CA) if it doesn't exist
if [[ ! -f "certs/ca.crt" ]]; then
    log_info "Creating Certificate Authority..."
    cockroach cert create-ca \
    --certs-dir=certs \
    --ca-key=secrets/ca.key
    log_success "Certificate Authority created"
else
    log_info "Certificate Authority already exists"
fi

# Set proper permissions on CA files
chmod 700 secrets/ca.key
chmod 700 certs/ca.crt

# Retrieve the default hostname and ensure it's not empty
log_info "Retrieving hostname..."
HOSTNAMES=$(hostname)
if [[ -z "$HOSTNAMES" ]]; then 
    log_error "Hostname retrieval failed"
    exit 1
fi
log_info "Hostname: $HOSTNAMES"

# Check the operating system and adjust IP address retrieval and hostname accordingly
log_info "Detecting operating system and retrieving IP addresses..."
if [[ "$(uname -s)" == "Darwin" ]]; then
    # Retrieve IP addresses using ifconfig on macOS
    IP_ADDRESSES=$(ifconfig | grep inet | grep -v inet6 | awk '{print $2}' | tr '\n' ' ')
    if [[ -z "$IP_ADDRESSES" ]]; then 
        log_error "IP address retrieval failed"
        exit 1
    fi
    # Include short hostname on macOS
    HOSTNAMES="$HOSTNAMES $(hostname -s)"
    log_info "Running on macOS - IP addresses: $IP_ADDRESSES"
else
    # Retrieve IP addresses using hostname -I on Linux
    IP_ADDRESSES=$(hostname -I | tr ' ' '\n' | grep -v '^$')
    if [[ -z "$IP_ADDRESSES" ]]; then 
        log_error "IP address retrieval failed"
        exit 1
    fi
    log_info "Running on Linux - IP addresses: $IP_ADDRESSES"
fi

# Combine all hostnames and IPs into one string
ALL_NAMES="$HOSTNAMES $IP_ADDRESSES cockroachdb"
log_info "All names for certificate: $ALL_NAMES"

# Include localhost and the fully-qualified domain name as common DNS names
COMMON_NAMES="localhost 127.0.0.1 $(hostname -f)"

# Combine and deduplicate all possible names for certificate inclusion
ALL_CERT_NAMES=$(echo "$ALL_NAMES $COMMON_NAMES" | tr ' ' '\n' | sort -u | tr '\n' ' ')

# Create node certificate with all detected names if it doesn't exist
if [[ ! -f "certs/node.crt" ]]; then
    log_info "Creating node certificate..."
    cockroach cert create-node \
    $ALL_CERT_NAMES \
    --certs-dir=certs \
    --ca-key=secrets/ca.key
    log_success "Node certificate created"
else
    log_info "Node certificate already exists"
fi

# Set proper permissions on node cert files
chmod 700 certs/node.crt
chmod 700 certs/node.key

# Create a client certificate for the root user if it doesn't exist
if [[ ! -f "certs/client.root.crt" ]]; then
    log_info "Creating root client certificate..."
    cockroach cert create-client \
    root \
    --certs-dir=certs \
    --ca-key=secrets/ca.key
    log_success "Root client certificate created"
else
    log_info "Root client certificate already exists"
fi

# Set proper permissions on client cert files
chmod 740 certs/client.root.*

# Handle Docker certificate path prefix on macOS
export CERT_PATH_PREFIX=$(uname -s | grep -q 'Darwin' && echo '/host_mnt/' || echo '')

log_success "CockroachDB certificate setup completed"
