#!/bin/bash


# TODO - harden things like etcd version check
#set -euo pipefail

# unset KAMIWAZA_ROOT in case of weird reinstall cases
unset KAMIWAZA_ROOT

# Initialize variables for tracking installation progress
START_TIME=$(date +%s)
KAMIWAZA_DEV_MODE=false
OFFLINE_EXTENSION_PATH=""

# Function to update and display progress
update_progress() {
    local step_name="$1"
    local current_time=$(date +%s)
    local elapsed_time=$((current_time - START_TIME))
    local elapsed_minutes=$((elapsed_time / 60))
    local elapsed_seconds=$((elapsed_time % 60))
    
    echo "┌────────────────────────────────────────────────────────────┐"
    echo "│ $step_name"
    echo "│ Time elapsed: ${elapsed_minutes}m ${elapsed_seconds}s"
    echo "└────────────────────────────────────────────────────────────┘"
}

source set-kamiwaza-root.sh

# Source common.sh early to get access to ensure_correct_node function
source common.sh

touch .kamiwaza_setup_complete

# Function to print messages in colors
print_in_color() {
    case $1 in
        green)
            echo -e "\033[92m$2\033[0m"
            ;;
        red)
            echo -e "\033[91m$2\033[0m"
            ;;
        yellow)
            echo -e "\033[93m$2\033[0m"
            ;;
        blue)
            echo -e "\033[94m$2\033[0m"
            ;;
        *)
            echo "$2"
            ;;
    esac
}

ensure_uv() {
    if command -v uv >/dev/null 2>&1; then
        print_in_color blue "uv already installed at $(command -v uv)"
        return
    fi

    print_in_color yellow "uv not detected; installing..."
    if ! curl -LsSf https://astral.sh/uv/install.sh | sh; then
        print_in_color red "Failed to install uv automatically. Install uv manually from https://astral.sh/uv/ and re-run."
        exit 1
    fi

    if [ -d "$HOME/.local/bin" ] && [[ ":$PATH:" != *":$HOME/.local/bin:"* ]]; then
        export PATH="$HOME/.local/bin:$PATH"
    fi
    if [ -d "$HOME/Library/Application Support/uv" ] && [[ ":$PATH:" != *":$HOME/Library/Application Support/uv:"* ]]; then
        export PATH="$HOME/Library/Application Support/uv:$PATH"
    fi

    if command -v uv >/dev/null 2>&1; then
        print_in_color green "uv installed successfully at $(command -v uv)"
    else
        print_in_color red "uv installation completed but binary not found on PATH."
        print_in_color red "Add the install directory to PATH or install uv manually, then re-run install.sh."
        exit 1
    fi
}

# don't permit unless flag passed
unset USER_ACCEPTED_KAMIWAZA_LICENSE

update_progress "Initializing installation"

# Check for community install file
if [[ -f ".kamiwaza_install_community" ]]; then
    export KAMIWAZA_COMMUNITY=true
    export KAMIWAZA_SWARM_HEAD=true
fi

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --community)
            # Community edition uses local env.sh and acts as head node
            export KAMIWAZA_COMMUNITY=true
            export KAMIWAZA_SWARM_HEAD=true
            shift
            ;;
        --lite)
            export KAMIWAZA_LITE=true
            shift
            ;;
        --full)
            export KAMIWAZA_LITE=false
            shift
            ;;
        --local-auth)
            export KAMIWAZA_USE_AUTH=false
            export KAMIWAZA_LOCAL_AUTH_EXPLICIT=true  # Flag that user explicitly requested local auth
            shift
            ;;
        --full-auth)
            export KAMIWAZA_USE_AUTH=true
            export KAMIWAZA_FULL_AUTH_EXPLICIT=true  # Flag that user explicitly requested full auth
            shift
            ;;
        --head)
            export KAMIWAZA_SWARM_HEAD=true
            shift
            ;;
        --dev)
            export KAMIWAZA_DEV_MODE=true
            shift
            ;;
        --worker)
            if [[ -z "${KAMIWAZA_HEAD_IP:-}" ]]; then
                print_in_color red "KAMIWAZA_HEAD_IP must be set for worker nodes"
                exit 1
            fi
            shift
            ;;
        --i-accept-the-kamiwaza-license)
            export USER_ACCEPTED_KAMIWAZA_LICENSE='yes'
            shift
            ;;
        --offline-extension-path)
            if [[ -z "$2" ]]; then
                print_in_color red "Error: --offline-extension-path requires a path argument"
                exit 1
            fi
            OFFLINE_EXTENSION_PATH="$2"
            shift 2
            ;;
        --external-url)
            if [[ -z "$2" ]]; then
                print_in_color red "Error: --external-url requires a URL argument"
                exit 1
            fi
            export KAMIWAZA_EXTERNAL_URL_DEFAULT="$2"
            shift 2
            ;;
        --use-localhost)
            export KAMIWAZA_USE_LOCALHOST=true
            shift
            ;;
        --use-localhost=*)
            value="${1#*=}"
            value_lower="$(printf '%s' "$value" | tr '[:upper:]' '[:lower:]')"
            case "$value_lower" in
                true|1|yes) export KAMIWAZA_USE_LOCALHOST=true ;;
                false|0|no) export KAMIWAZA_USE_LOCALHOST=false ;;
                *)
                    print_in_color red "Invalid value for --use-localhost: $value"
                    echo "Use true/false, 1/0, or yes/no"
                    exit 1
                    ;;
            esac
            shift
            ;;
        *)
            print_in_color red "Unknown option: $1"
            echo "Usage: $0 (--head | --worker | --community) [--lite | --full] [--local-auth | --full-auth] [OPTIONS]"
            echo ""
            echo "Options:"
            echo "  --community                    Install community edition (acts as head node)"
            echo "  --head                         Install as cluster head node"
            echo "  --worker                       Install as cluster worker node (requires KAMIWAZA_HEAD_IP)"
            echo ""
            echo "Database Mode:"
            echo "  --lite                         SQLite database (community only)"
            echo "  --full                         CockroachDB + DataHub (requires ~4-6GB additional memory)"
            echo ""
            echo "Authentication Mode:"
            echo "  --local-auth                   Kamiwaza issues JWTs directly (simpler, no Keycloak)"
            echo "                                 Must be used with --lite mode"
            echo "  --full-auth                    Keycloak handles authentication (enterprise SSO)"
            echo "                                 Must be used with --full mode"
            echo ""
            echo "  Note: Only two combinations are supported:"
            echo "        --lite (defaults to --local-auth)"
            echo "        --full (defaults to --full-auth)"
            echo ""
            echo "Network Configuration:"
            echo "  --external-url <url>           Set external URL for remote access (hostname or IP)"
            echo "                                 Auto-detected from cloud metadata if not specified"
            echo "  --use-localhost[=true|false]   Use 127.0.0.1 for KAMIWAZA_HEAD_IP and disable cloud detection"
            echo "                                 Default: true for macOS/WSL community, false for enterprise"
            echo ""
            echo "Other Options:"
            echo "  --offline-extension-path <path> Path to offline extensions registry (sets up file:// URL)"
            echo "  --i-accept-the-kamiwaza-license Accept the Kamiwaza license"
            echo ""
            echo "Examples:"
            echo "  # Community with defaults (SQLite + local auth)"
            echo "  $0 --community --i-accept-the-kamiwaza-license"
            echo ""
            echo "  # Community with full auth and CockroachDB"
            echo "  $0 --community --full --full-auth --i-accept-the-kamiwaza-license"
            echo ""
            echo "  # Enterprise cluster head with Keycloak"
            echo "  $0 --head --full-auth --i-accept-the-kamiwaza-license"
            exit 1
            ;;
    esac
done

# Validate lite/full flags
if [[ "${KAMIWAZA_LITE:-}" == "true" && "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    print_in_color red "--lite flag is only valid with --community"
    exit 1
fi

# Set default for KAMIWAZA_LITE based on edition
# Note: Explicit flags (--lite/--full) are checked above; here we only set defaults
# when the variable wasn't set via command-line flag
if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" && -z "${KAMIWAZA_LITE:-}" ]]; then
    export KAMIWAZA_LITE=true  # Default to lite for community
elif [[ "${KAMIWAZA_COMMUNITY:-}" != "true" ]]; then
    export KAMIWAZA_LITE=false  # Enterprise is always full
fi

# Set default for KAMIWAZA_USE_AUTH based on KAMIWAZA_LITE
# IMPORTANT: For community installs, always apply the default unless explicit auth flag was passed.
# This prevents stale environment variables from causing unexpected behavior.
if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" && "${KAMIWAZA_FULL_AUTH_EXPLICIT:-}" != "true" && "${KAMIWAZA_LOCAL_AUTH_EXPLICIT:-}" != "true" ]]; then
    # Community mode: default to local auth for lite, full auth for full (unless explicitly overridden)
    if [[ "${KAMIWAZA_LITE:-}" == "true" ]]; then
        export KAMIWAZA_USE_AUTH=false  # Default to local auth for lite mode
    else
        export KAMIWAZA_USE_AUTH=true   # Full mode gets full auth
    fi
elif [[ -z "${KAMIWAZA_USE_AUTH:-}" ]]; then
    # Enterprise or no community flag: only set if not already set
    if [[ "${KAMIWAZA_LITE:-}" == "true" ]]; then
        export KAMIWAZA_USE_AUTH=false  # Default to local auth for lite mode
    else
        export KAMIWAZA_USE_AUTH=true   # Default to full auth for full mode
    fi
fi

# Validate supported auth mode combinations
# Only lite+local-auth and full+full-auth are supported and tested
if [[ "${KAMIWAZA_USE_AUTH:-}" == "true" && "${KAMIWAZA_LITE:-}" == "true" ]]; then
    print_in_color red "Error: --lite mode with --full-auth is not supported"
    print_in_color red "  - Lite mode (SQLite) is not compatible with Keycloak"
    print_in_color red "  - Use '--full --full-auth' for Keycloak authentication"
    print_in_color red "  - Or use '--lite' alone for local authentication (default)"
    exit 1
elif [[ "${KAMIWAZA_USE_AUTH:-}" == "false" && "${KAMIWAZA_LITE:-}" == "false" ]]; then
    print_in_color red "Error: --full mode with --local-auth is not supported"
    print_in_color red "  - Full mode is designed for Keycloak authentication"
    print_in_color red "  - Use '--full' alone for Keycloak (default)"
    print_in_color red "  - Or use '--lite --local-auth' for local authentication"
    exit 1
fi

if [[ "${KAMIWAZA_SWARM_HEAD:-}" != "true" && -z "${KAMIWAZA_HEAD_IP:-}" ]]; then
    print_in_color red "Must specify either --community, --head or --worker with KAMIWAZA_HEAD_IP set"
    exit 1
fi

if [[ -n "${KAMIWAZA_HEAD_IP:-}" ]]; then
    export KAMIWAZA_HEAD_IP  # Export it so child processes get it
fi

# Function to install OSX dependencies
install_osx_dependencies() {
    update_progress "Installing OSX dependencies"
    
    # Function to check if a command exists
    command_exists() {
        command -v "$1" >/dev/null 2>&1
    }

    # Function to check if a brew package is installed
    brew_package_installed() {
        brew list "$1" >/dev/null 2>&1
    }

    # Function to handle errors
    handle_error() {
        print_in_color red "ERROR: $1"
        print_in_color red "Installation failed. Please check the error message above and try again."
        exit 1
    }

    # Install Homebrew if not present
    if ! command_exists brew; then
        echo "Installing Homebrew..."
        if ! /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"; then
            handle_error "Failed to install Homebrew. Please check your internet connection and try again."
        fi
        
        # Add Homebrew to path if not already present
        if ! grep -q 'eval "$(/opt/homebrew/bin/brew shellenv)"' /Users/$USER/.zprofile; then
            echo 'eval "$(/opt/homebrew/bin/brew shellenv)"' >> /Users/$USER/.zprofile
        fi

        if ! grep -q 'eval "$(/opt/homebrew/bin/brew shellenv)"' /Users/$USER/.bash_profile; then
            echo 'eval "$(/opt/homebrew/bin/brew shellenv)"' >> /Users/$USER/.bash_profile
        fi

        eval "$(/opt/homebrew/bin/brew shellenv)" || handle_error "Failed to initialize Homebrew environment"
    else
        echo "Homebrew already installed"
    fi

    update_progress "Installing brew packages"
    
    # Install brew packages if not present
    BREW_PACKAGES="pyenv pyenv-virtualenv cockroachdb/tap/cockroach cfssl etcd cmake"
    for package in $BREW_PACKAGES; do
        if ! brew_package_installed $package; then
            echo "Installing $package..."
            if ! brew install $package; then
                handle_error "Failed to install $package. Try running 'brew doctor' to diagnose issues."
            fi
        else
            echo "$package already installed"
        fi
    done

    update_progress "Setting up Docker"
    
    # Install Docker.app if not present
    if ! [ -d "/Applications/Docker.app" ]; then
        echo "Installing Docker.app..."
        if ! brew install --cask docker; then
            handle_error "Failed to install Docker.app. Check if Homebrew cask is working properly."
        fi
    else
        echo "Docker.app already installed"
    fi

    # Configure Docker permissions
    sudo chown -R $(whoami):staff ~/.docker 2>/dev/null || true

    if ! brew_package_installed docker-compose; then
        echo "Installing docker-compose..."
        if ! brew install docker-compose; then
            handle_error "Failed to install docker-compose"
        fi
    else
        echo "docker-compose already installed"
    fi

    mkdir -p ~/.docker
    touch ~/.docker/config.json

    # Check if config.json already has the cli-plugins configuration
    if ! grep -q "cliPluginsExtraDirs" ~/.docker/config.json; then
        # Create temporary file with new config
        echo '{
    "cliPluginsExtraDirs": [
        "/opt/homebrew/lib/docker/cli-plugins"
    ]
    }' > ~/.docker/config.json.tmp

        # If config.json is empty, just move the temp file
        if [ ! -s ~/.docker/config.json ]; then
            mv ~/.docker/config.json.tmp ~/.docker/config.json
        else
            # Merge existing config with new config
            if ! jq -s '.[0] * .[1]' ~/.docker/config.json ~/.docker/config.json.tmp > ~/.docker/config.json.merged; then
                handle_error "Failed to update Docker configuration. Please check if jq is installed."
            fi
            mv ~/.docker/config.json.merged ~/.docker/config.json
            rm ~/.docker/config.json.tmp
        fi
    fi

    # Start Docker if not running
    # On macOS, check for Docker Desktop processes or if docker command works
    if [[ "$OSTYPE" == "darwin"* ]]; then
        # Check if Docker is running by looking for Docker Desktop processes or testing docker command
        if ! pgrep -f "Docker Desktop" > /dev/null && ! docker info >/dev/null 2>&1; then
            echo "Starting Docker Desktop..."
            open -a Docker.app
            print_in_color yellow "Please complete the Docker Desktop installation by following any prompts that appear."
            print_in_color yellow "Once Docker Desktop is fully installed and running, press Enter to continue..."
            read -r
        fi
    else
        # For non-macOS systems, keep the original check
        if ! pgrep -x "Docker" > /dev/null; then
            echo "Starting Docker..."
            open -a Docker.app
            print_in_color yellow "Please complete the Docker Desktop installation by following any prompts that appear."
            print_in_color yellow "Once Docker Desktop is fully installed and running, press Enter to continue..."
            read -r
        fi
    fi

    update_progress "Configuring Python environment"
    
    # Configure pyenv if not already set up
    if ! grep -q 'export PATH="$HOME/.pyenv/bin:$PATH"' ~/.zshrc; then
        echo "Configuring pyenv in ~/.zshrc..."
        echo 'export PATH="$HOME/.pyenv/bin:$PATH"' >> ~/.zshrc || handle_error "Failed to update ~/.zshrc"
        echo 'eval "$(pyenv init -)"' >> ~/.zshrc || handle_error "Failed to update ~/.zshrc"
        echo 'eval "$(pyenv virtualenv-init -)"' >> ~/.zshrc || handle_error "Failed to update ~/.zshrc"
    fi

    if ! grep -q 'export PATH="$HOME/.pyenv/bin:$PATH"' ~/.bashrc; then
        echo "Configuring pyenv in ~/.bashrc..."
        echo 'export PATH="$HOME/.pyenv/bin:$PATH"' >> ~/.bashrc || handle_error "Failed to update ~/.bashrc"
        echo 'eval "$(pyenv init -)"' >> ~/.bashrc || handle_error "Failed to update ~/.bashrc"
        echo 'eval "$(pyenv virtualenv-init -)"' >> ~/.bashrc || handle_error "Failed to update ~/.bashrc"
    fi

    # Load pyenv
    if ! eval "$(pyenv init -)"; then
        handle_error "Failed to initialize pyenv. Please check your pyenv installation."
    fi
    
    if ! eval "$(pyenv virtualenv-init -)"; then
        handle_error "Failed to initialize pyenv-virtualenv. Please check your pyenv-virtualenv installation."
    fi

    # Install Python 3.10 if not present
    if ! pyenv versions | grep -q "3.10"; then
        echo "Installing Python 3.10..."
        if ! pyenv install 3.10; then
            handle_error "Failed to install Python 3.10. Check if required dependencies are installed."
        fi
    else
        echo "Python 3.10 already installed"
    fi

    # Set local Python version if not set
    if ! [ -f ".python-version" ] || ! grep -q "3.10" .python-version; then
        echo "Setting local Python version to 3.10..."
        if ! pyenv local 3.10; then
            handle_error "Failed to set local Python version to 3.10"
        fi
    fi


    update_progress "Setting up Node.js environment"
    
    # Install nvm if not present
    if ! [ -d "$HOME/.nvm" ]; then
        echo "Installing nvm..."
        if ! curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.40.1/install.sh | bash; then
            handle_error "Failed to install nvm. Please check your internet connection and try again."
        fi
        
        # Set up NVM environment
        export NVM_DIR="$([ -z "${XDG_CONFIG_HOME-}" ] && printf %s "${HOME}/.nvm" || printf %s "${XDG_CONFIG_HOME}/nvm")"
        [ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh" || handle_error "Failed to source nvm.sh"
    else
        echo "nvm already installed"
    fi

    # Source nvm
    export NVM_DIR="$([ -z "${XDG_CONFIG_HOME-}" ] && printf %s "${HOME}/.nvm" || printf %s "${XDG_CONFIG_HOME}/nvm")"
    if [ -s "$NVM_DIR/nvm.sh" ]; then
        if ! \. "$NVM_DIR/nvm.sh"; then
            handle_error "Failed to source nvm.sh"
        fi
    else
        handle_error "nvm.sh not found. NVM installation may be corrupted."
    fi

    # Use centralized Node.js installation function
    if ! ensure_correct_node; then
        handle_error "Failed to ensure Node.js 22 is available"
    fi

    # Verify all required tools are available
    REQUIRED_TOOLS=("brew" "uv" "pyenv" "node" "docker" "docker-compose" "etcd" "cockroach" "cfssl")
    for tool in "${REQUIRED_TOOLS[@]}"; do
        if ! command_exists $tool; then
            handle_error "Required tool '$tool' is not available after installation. Please check the installation logs."
        fi
    done

    echo "Installation of dependencies complete!"
}

# Call install_osx_dependencies if on OSX
if [[ "$(uname)" == "Darwin" ]]; then
    print_in_color blue "Installing OSX dependencies..."
    install_osx_dependencies
fi

update_progress "Setting up environment"

#Proceed with the rest of the installation
setup_environment

KAMIWAZA_SECURITY_DIR="${KAMIWAZA_SECURITY_DIR:-/etc/kamiwaza/security}"
FERNET_KEY_PATH="${FERNET_KEY_PATH:-${KAMIWAZA_SECURITY_DIR}/fernet.key}"
kw_resolve_fernet_identity

# Verify initialization worked - updated to check correct location based on installation type
if [[ "$(uname)" == "Darwin" ]] || [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    if [[ ! -f "${KAMIWAZA_ROOT:-}/env.sh" ]]; then
        print_in_color red "Community/OSX cluster initialization failed"
        exit 1
    fi
else
    if ! sudo test -f /etc/kamiwaza/env.sh && [[ ! -f "${KAMIWAZA_ROOT:-}/env.sh" ]]; then
        print_in_color red "Enterprise cluster initialization failed"
        exit 1
    fi
fi

if [[ "$KAMIWAZA_DEV_MODE" == "true" ]]; then
    ENV_SH_FILE="${KAMIWAZA_ROOT:-.}/env.sh"
    ANALYTICS_LINE='export KAMIWAZA_ANALYTICS_URL=https://analytics-dev.kamiwaza.ai/v1/events'
    # Ensure frontend runs in dev mode when requested, matching macOS developer install
    set_env_value_if_not_exists "KAMIWAZA_FRONTEND_DEVMODE" "true"
    if [[ -z "${KAMIWAZA_FRONTEND_DEVMODE:-}" ]]; then
        export KAMIWAZA_FRONTEND_DEVMODE=true
    fi
    (
        if [[ ! -f "$ENV_SH_FILE" ]]; then
            echo "# Kamiwaza environment variables" > "$ENV_SH_FILE"
        fi

        if ! grep -q "^export KAMIWAZA_ANALYTICS_URL=" "$ENV_SH_FILE"; then
            echo "$ANALYTICS_LINE" >> "$ENV_SH_FILE"
            print_in_color green "Added KAMIWAZA_ANALYTICS_URL to $ENV_SH_FILE (dev mode)"
        else
            print_in_color blue "KAMIWAZA_ANALYTICS_URL already present in $ENV_SH_FILE"
        fi
    )
fi

# Configure offline extensions path if provided
if [[ -n "$OFFLINE_EXTENSION_PATH" ]]; then
    update_progress "Configuring offline extensions"

    # Validate the path exists
    if [[ ! -d "$OFFLINE_EXTENSION_PATH" ]]; then
        print_in_color red "Error: Offline extension path does not exist: $OFFLINE_EXTENSION_PATH"
        exit 1
    fi

    # Check if it looks like a valid registry (has garden/default/apps.json)
    if [[ ! -f "$OFFLINE_EXTENSION_PATH/garden/default/apps.json" ]] && [[ ! -f "$OFFLINE_EXTENSION_PATH/kamiwaza-extension-registry/garden/default/apps.json" ]]; then
        print_in_color yellow "Warning: Path doesn't appear to contain a valid registry structure"
        print_in_color yellow "Expected to find garden/default/apps.json"
    fi

    # If the path contains kamiwaza-extension-registry subdirectory, use it
    if [[ -d "$OFFLINE_EXTENSION_PATH/kamiwaza-extension-registry" ]]; then
        OFFLINE_EXTENSION_PATH="$OFFLINE_EXTENSION_PATH/kamiwaza-extension-registry"
    fi

    # Convert to absolute path
    OFFLINE_EXTENSION_PATH=$(cd "$OFFLINE_EXTENSION_PATH" && pwd)

    set_env_value "KAMIWAZA_EXTENSION_STAGE" "LOCAL"
    set_env_value "KAMIWAZA_EXTENSION_LOCAL_STAGE_URL" "\"file://${OFFLINE_EXTENSION_PATH}\""

    env_target="${KAMIWAZA_ENV_FILE_PATH:-${KAMIWAZA_ROOT:-}/env.sh}"
    print_in_color blue "Setting up offline extensions in $env_target"

    print_in_color green "✓ Offline extensions configured:"
    print_in_color green "  KAMIWAZA_EXTENSION_STAGE=LOCAL"
    print_in_color green "  KAMIWAZA_EXTENSION_LOCAL_STAGE_URL=file://${OFFLINE_EXTENSION_PATH}"

    # Export for current session
    export KAMIWAZA_EXTENSION_STAGE=LOCAL
    export KAMIWAZA_EXTENSION_LOCAL_STAGE_URL="file://${OFFLINE_EXTENSION_PATH}"
fi

update_progress "Loading environment configuration"

# Source the appropriate env file
if [[ "$(uname)" == "Darwin" ]] || [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    source "${KAMIWAZA_ROOT:-}/env.sh"
else
    if [[ -f /etc/kamiwaza/env.sh ]]; then
        source /etc/kamiwaza/env.sh
    elif [[ -f "${KAMIWAZA_ROOT:-}/env.sh" ]]; then
        source "${KAMIWAZA_ROOT:-}/env.sh"
    fi
fi

# Set default admin password if not already set (used for local auth when KAMIWAZA_USE_AUTH=false)
# Enterprise installs get a random password; community installs use "kamiwaza"
if [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    # Community: use simple password for developer convenience
    # Always set for community to override any stale env values
    set_env_value "KAMIWAZA_ADMIN_PASSWORD" "kamiwaza"
    export KAMIWAZA_ADMIN_PASSWORD="kamiwaza"
else
    # Enterprise: generate random password if not already set
    ensure_random_secret "KAMIWAZA_ADMIN_PASSWORD"
fi

# Deactivate any foreign virtual environment to prevent interference
# with uv sync and subsequent Python operations
if [[ -n "${VIRTUAL_ENV:-}" ]]; then
    _kw_our_venv="${KAMIWAZA_ROOT:-.}/.venv"
    _kw_is_foreign=true
    if [[ -d "${VIRTUAL_ENV}" ]] && [[ -d "$_kw_our_venv" ]]; then
        _kw_resolved_active="$(cd "${VIRTUAL_ENV}" && pwd)"
        _kw_resolved_ours="$(cd "$_kw_our_venv" && pwd)"
        if [[ "$_kw_resolved_active" == "$_kw_resolved_ours" ]]; then
            _kw_is_foreign=false
        fi
    fi
    if [[ "$_kw_is_foreign" == "true" ]]; then
        print_in_color yellow "Detected active virtual environment: ${VIRTUAL_ENV}"
        print_in_color yellow "Deactivating to avoid installation conflicts..."
        if type deactivate &>/dev/null; then
            deactivate
        else
            # Manually remove foreign venv bin from PATH
            _kw_foreign_bin="${VIRTUAL_ENV}/bin"
            _kw_new_path=""
            _kw_old_ifs="$IFS"
            IFS=':'
            for _kw_dir in $PATH; do
                if [[ "$_kw_dir" != "$_kw_foreign_bin" ]]; then
                    _kw_new_path="${_kw_new_path:+${_kw_new_path}:}${_kw_dir}"
                fi
            done
            IFS="$_kw_old_ifs"
            PATH="$_kw_new_path"
            unset VIRTUAL_ENV
            unset _kw_foreign_bin _kw_new_path _kw_old_ifs _kw_dir
        fi
    fi
    unset _kw_our_venv _kw_is_foreign _kw_resolved_active _kw_resolved_ours
fi

update_progress "Synchronizing Python dependencies (uv)"

ensure_uv
# Include dev group only when running in dev mode
# Default uv_sync.sh behavior includes dev group, so we only need to set
# KW_UV_NO_DEFAULT=1 to skip it when not in dev mode
if [[ "$KAMIWAZA_DEV_MODE" == "true" ]]; then
    print_in_color blue "Running uv sync with dev dependencies"
    ./uv_sync.sh
else
    print_in_color blue "Running uv sync with base dependencies"
    KW_UV_NO_DEFAULT=1 ./uv_sync.sh
fi
if [[ $? -ne 0 ]]; then
    print_in_color red "uv sync failed."
    exit 1
fi

# fix permissions if needed
chmod 774 ${KAMIWAZA_ROOT}/startup/kamiwazad.sh || true

# shellcheck disable=SC2218
update_progress "Running second phase installer"

# shellcheck disable=SC2218
print_in_color green "Running 2nd phase installer in venv..."
export KAMIWAZA_RUN_FROM_INSTALL='yes'
# Export progress tracking variables for setup.sh
export START_TIME
source setup.sh
unset KAMIWAZA_RUN_FROM_INSTALL
unset USER_ACCEPTED_KAMIWAZA_LICENSE

update_progress "Generating fernet key"

# Generate fernet key for encryption
if [[ "$(uname)" == "Darwin" ]] || [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    # Community edition or macOS: put fernet key in KAMIWAZA_ROOT/runtime
    mkdir -p ${KAMIWAZA_ROOT}/runtime
    if [[ ! -f ${KAMIWAZA_ROOT}/runtime/fernet.key ]]; then
        print_in_color blue "Creating fernet key..."
        kw_py -c "from kamiwaza.lib.util import generate_fernet_key; print(generate_fernet_key())" > \
            ${KAMIWAZA_ROOT}/runtime/fernet.key
        chmod 600 ${KAMIWAZA_ROOT}/runtime/fernet.key
        print_in_color green "Fernet key created at ${KAMIWAZA_ROOT}/runtime/fernet.key"
    else
        print_in_color blue "Fernet key already exists at ${KAMIWAZA_ROOT}/runtime/fernet.key"
    fi
else
    # Enterprise edition: put fernet key in ${KAMIWAZA_SECURITY_DIR}
    kw_resolve_fernet_identity

    sudo mkdir -p "${KAMIWAZA_SECURITY_DIR}"
    sudo chmod 750 "${KAMIWAZA_SECURITY_DIR}"
    if [[ -n "${FERNET_KEY_OWNER:-}" ]]; then
        sudo chown "${FERNET_KEY_OWNER}:${FERNET_KEY_GROUP}" "${KAMIWAZA_SECURITY_DIR}"
    else
        print_in_color yellow "Warning: unable to determine fernet key owner; leaving ${KAMIWAZA_SECURITY_DIR} with existing ownership"
    fi

    if [[ ! -f "${FERNET_KEY_PATH}" ]]; then
        print_in_color blue "Creating fernet key..."
        kw_py -c "from kamiwaza.lib.util import generate_fernet_key; print(generate_fernet_key())" | \
            sudo tee "${FERNET_KEY_PATH}" > /dev/null
        sudo chmod 600 "${FERNET_KEY_PATH}"
        print_in_color green "Fernet key created at ${FERNET_KEY_PATH}"
    else
        print_in_color blue "Fernet key already exists at ${FERNET_KEY_PATH}"
    fi

    if sudo test -f "${FERNET_KEY_PATH}"; then
        kw_ensure_fernet_permissions "${FERNET_KEY_PATH}"
        if ! kw_verify_fernet_key_readable "${FERNET_KEY_OWNER}" "${FERNET_KEY_PATH}"; then
            print_in_color yellow "Warning: ${FERNET_KEY_OWNER:-catalog user} cannot read ${FERNET_KEY_PATH}; catalog/retrieval may fail"
        fi
    fi
fi

update_progress "Provisioning retrieval gRPC TLS material"

if [[ "$(uname)" == "Darwin" ]] || [[ "${KAMIWAZA_COMMUNITY:-}" == "true" ]]; then
    tls_dir="${KAMIWAZA_ROOT}/runtime"
    mkdir -p "${tls_dir}"
    cert_path="${tls_dir}/retrieval-grpc.crt"
    key_path="${tls_dir}/retrieval-grpc.key"
    ca_path="${tls_dir}/retrieval-grpc-ca.crt"
    if [[ ! -f "${cert_path}" || ! -f "${key_path}" ]]; then
        print_in_color blue "Creating self-signed retrieval gRPC TLS certificate..."
        kw_py scripts/generate_retrieval_tls.py \
            --cert-path "${cert_path}" \
            --key-path "${key_path}" \
            --ca-path "${ca_path}"
        chmod 644 "${cert_path}" "${ca_path}"
        chmod 600 "${key_path}"
        print_in_color green "TLS material generated under ${tls_dir}"
    else
        print_in_color blue "TLS material already present under ${tls_dir}"
    fi
else
    tls_dir="/etc/kamiwaza/ssl"
    sudo mkdir -p "${tls_dir}"
    sudo chmod 750 "${tls_dir}"
    sudo chown ${USER}:${USER} "${tls_dir}"
    cert_path="${tls_dir}/retrieval-grpc.crt"
    key_path="${tls_dir}/retrieval-grpc.key"
    ca_path="${tls_dir}/retrieval-grpc-ca.crt"
    if [[ ! -f "${cert_path}" || ! -f "${key_path}" ]]; then
        print_in_color blue "Creating self-signed retrieval gRPC TLS certificate..."
        tmpdir=$(mktemp -d)
        kw_py scripts/generate_retrieval_tls.py \
            --cert-path "${tmpdir}/retrieval-grpc.crt" \
            --key-path "${tmpdir}/retrieval-grpc.key" \
            --ca-path "${tmpdir}/retrieval-grpc-ca.crt"
        sudo mv "${tmpdir}/retrieval-grpc.crt" "${cert_path}"
        sudo mv "${tmpdir}/retrieval-grpc.key" "${key_path}"
        sudo mv "${tmpdir}/retrieval-grpc-ca.crt" "${ca_path}"
        rm -rf "${tmpdir}"
        sudo chmod 640 "${cert_path}" "${ca_path}"
        sudo chmod 600 "${key_path}"
        print_in_color green "TLS material generated under ${tls_dir}"
    else
        print_in_color blue "TLS material already present under ${tls_dir}"
    fi
fi

# Sync offline extensions if configured
if [[ "${KAMIWAZA_EXTENSION_STAGE:-}" == "LOCAL" ]] && [[ "${KAMIWAZA_EXTENSION_LOCAL_STAGE_URL:-}" == file://* ]]; then
    update_progress "Syncing offline extensions"

    print_in_color blue "Syncing extensions from offline registry..."

    # Check if the kamiwazad.sh script exists
    if [[ -f "${KAMIWAZA_ROOT}/startup/kamiwazad.sh" ]]; then
        # Try to run the extensions sync
        if "${KAMIWAZA_ROOT}/startup/kamiwazad.sh" extensions sync; then
            print_in_color green "✓ Extensions synced successfully from offline registry"
        else
            print_in_color yellow "⚠ Extension sync failed or partially completed"
            print_in_color yellow "  You can manually sync later with: ${KAMIWAZA_ROOT}/startup/kamiwazad.sh extensions sync"
        fi
    else
        print_in_color yellow "⚠ Could not find kamiwazad.sh to sync extensions"
        print_in_color yellow "  After installation, run: ${KAMIWAZA_ROOT}/startup/kamiwazad.sh extensions sync"
    fi
fi

# Display final installation status
FINAL_TIME=$(date +%s)
TOTAL_ELAPSED_TIME=$((FINAL_TIME - START_TIME))
TOTAL_MINUTES=$((TOTAL_ELAPSED_TIME / 60))
TOTAL_SECONDS=$((TOTAL_ELAPSED_TIME % 60))

echo "┌────────────────────────────────────────────────────────────┐"
echo "│ Installation Complete!                                      │"
echo "│ Total time: ${TOTAL_MINUTES}m ${TOTAL_SECONDS}s                                    │"
echo "└────────────────────────────────────────────────────────────┘"
