# Steps for Successful Install

# 1. Test for requirements.txt
# 2. Test for config file
# 3. Test for Docker files
# 4. Test for/launch containers
# 5. Create databases
# 6. License stuff?

import os
import re
import importlib.metadata
import sys
import subprocess
import time
import ssl
from pathlib import Path
from urllib import request
from urllib.parse import urlparse
from typing import Optional, Tuple
from packaging.version import parse as parse_version

from kamiwaza.lib.util import get_kamiwaza_root

# Removed cryptography imports - JWT keys now handled by Keycloak

# Set PROTOCOL_BUFFERS_PYTHON_IMPLEMENTATION to use pure-Python parsing
os.environ["PROTOCOL_BUFFERS_PYTHON_IMPLEMENTATION"] = "python"


def print_in_color(color: str, message: str) -> None:
    """Print a message in the specified color."""
    colors = {
        "red": "\033[91m",
        "green": "\033[92m",
        "yellow": "\033[93m",
        "blue": "\033[94m",
        "reset": "\033[0m",
    }
    print(f"{colors.get(color, '')}{message}{colors['reset']}")


def check_container_status() -> None:
    """Check and report the status of key Kamiwaza containers."""
    print_in_color("blue", "*** Checking container status...")

    key_containers = [
        # 'cockroach',
        "etcd",
        "traefik",
    ]

    # Check if milvus should be required (default: yes, unless explicitly disabled)
    if os.getenv("KAMIWAZA_MILVUS_ENABLED", "true").lower() == "true":
        key_containers.append("milvus")
    if os.getenv("KAMIWAZA_USE_AUTH", "true").lower() in {"true", "1", "yes"}:
        key_containers.append("keycloak")

    try:
        # Get running containers
        result = subprocess.run(
            ["docker", "ps", "--format", "table {{.Names}}\t{{.Status}}\t{{.Image}}"],
            capture_output=True,
            text=True,
        )

        if result.returncode == 0:
            output_lines = result.stdout.strip().split("\n")
            print("Running containers:")
            for line in output_lines:
                print(f"  {line}")

            # Check if key containers are running
            running_containers = result.stdout.lower()
            for container in key_containers:
                if container in running_containers:
                    print_in_color("green", f"✓ {container} container is running")
                else:
                    print_in_color(
                        "yellow",
                        f"⚠ {container} container not found in running containers",
                    )
        else:
            print_in_color("red", f"Failed to check container status: {result.stderr}")

    except Exception as e:
        print_in_color("red", f"Error checking container status: {e}")

    print()  # Add spacing


def resolve_env_file() -> Optional[str]:
    explicit = os.getenv("KAMIWAZA_ENV_FILE_PATH")
    if explicit and os.path.exists(explicit):
        return explicit
    root = os.getenv("KAMIWAZA_ROOT", os.getcwd())
    community = os.getenv("KAMIWAZA_COMMUNITY", "true").lower() == "true"
    candidate = os.path.join(root, "env.sh") if community else "/etc/kamiwaza/env.sh"
    if os.path.exists(candidate):
        return candidate
    return None


def update_env_file(key: str, value: str) -> None:
    env_file = resolve_env_file()
    if not env_file:
        print_in_color("yellow", f"Warning: unable to locate env.sh to persist {key}")
        return
    try:
        lines = []
        if os.path.exists(env_file):
            with open(env_file, "r", encoding="utf-8") as handle:
                lines = handle.readlines()
        pattern = re.compile(rf"^export {re.escape(key)}=")
        replaced = False
        new_lines = []
        for line in lines:
            if pattern.match(line.strip()):
                new_lines.append(f'export {key}="{value}"\n')
                replaced = True
            else:
                new_lines.append(line)
        if not replaced:
            new_lines.append(f'export {key}="{value}"\n')
        with open(env_file, "w", encoding="utf-8") as handle:
            handle.writelines(new_lines)
    except Exception as exc:
        print_in_color(
            "yellow", f"Warning: failed to update {env_file} with {key}: {exc}"
        )


def _keycloak_verify_enabled() -> bool:
    flag = os.getenv("AUTH_GATEWAY_TLS_INSECURE")
    if flag is None:
        return True
    return flag.strip().lower() not in {"1", "true", "yes"}


def wait_for_keycloak(timeout: int = 600) -> bool:
    url = os.getenv("AUTH_GATEWAY_KEYCLOAK_URL", "https://localhost").rstrip("/")
    realm = os.getenv("AUTH_GATEWAY_KEYCLOAK_REALM", "kamiwaza")
    endpoint = f"{url}/realms/{realm}/.well-known/openid-configuration"

    parsed = urlparse(endpoint)
    if parsed.scheme not in {"http", "https"}:
        raise ValueError(f"Invalid Keycloak endpoint scheme: {parsed.scheme}")

    verify = _keycloak_verify_enabled()
    if verify:
        ssl_context = ssl.create_default_context()
    else:
        # Intentional: allow insecure TLS for local/dev installs when auth_gateway_tls_insecure=true
        ssl_context = ssl._create_unverified_context()  # nosec B323
    start = time.time()
    while time.time() - start < timeout:
        try:
            # Scheme is validated above (http/https only)
            with request.urlopen(
                endpoint, timeout=5.0, context=ssl_context
            ) as resp:  # nosec B310
                if resp.status == 200:
                    return True
        except Exception:
            time.sleep(2)
            continue
    return False


def ensure_forwardauth_service_account() -> None:
    use_auth = os.getenv("KAMIWAZA_USE_AUTH", "true").lower() in {"true", "1", "yes"}
    lite_mode = os.getenv("KAMIWAZA_LITE", "false").lower() == "true"
    if not use_auth or lite_mode:
        print_in_color(
            "blue",
            "Skipping ForwardAuth service account setup (auth disabled or lite mode).",
        )
        return
    print_in_color("blue", "Waiting for Keycloak to become available...")
    if not wait_for_keycloak():
        print_in_color(
            "yellow",
            "Keycloak did not become ready; skipping ForwardAuth service account bootstrap.",
        )
        return
    # Always use the canonical service account ID and ignore any env overrides to avoid stale secrets.
    client_id = "kamiwaza-svc"
    runner = Path(get_kamiwaza_root()) / "scripts" / "kw_py"
    cmd = [
        str(runner),
        "-m",
        "scripts.setup_kamiwaza_service_account",
        "--client-id",
        client_id,
    ]
    # Pass --insecure-tls if TLS verification is disabled (self-signed certs)
    if not _keycloak_verify_enabled():
        cmd.append("--insecure-tls")
    try:
        subprocess.run(cmd, check=True)
        print_in_color(
            "green", f"ForwardAuth service account '{client_id}' provisioned."
        )
    except subprocess.CalledProcessError as exc:
        print_in_color(
            "yellow", f"ForwardAuth service account setup failed (continuing): {exc}"
        )
        return


def ensure_keycloak_client_redirects() -> None:
    """
    Ensure Keycloak client has correct redirect URIs for the configured hostname.

    Syncs AUTH_GATEWAY_PUBLIC_URL / KAMIWAZA_ORIGIN to Keycloak's allowed
    redirect URIs. Should be called after ensure_forwardauth_service_account()
    since that already waits for Keycloak availability.
    """
    use_auth = os.getenv("KAMIWAZA_USE_AUTH", "true").lower() in {"true", "1", "yes"}
    lite_mode = os.getenv("KAMIWAZA_LITE", "false").lower() == "true"
    if not use_auth or lite_mode:
        print_in_color(
            "blue",
            "Skipping Keycloak client redirect sync (auth disabled or lite mode).",
        )
        return

    print_in_color("blue", "Syncing Keycloak client redirect URIs...")

    # Import here to avoid circular imports and heavy dependencies at module load
    from kamiwaza.services.auth.keycloak_client_setup import (
        sync_keycloak_client_redirects,
        build_redirect_config,
    )

    # Show what will be configured
    redirect_uris, web_origins = build_redirect_config()
    print_in_color("blue", f"  Redirect URIs: {redirect_uris}")
    print_in_color("blue", f"  Web Origins: {web_origins}")

    # Determine TLS verification setting
    verify_tls = _keycloak_verify_enabled()

    if sync_keycloak_client_redirects(verify_tls=verify_tls):
        print_in_color("green", "Keycloak client redirect URIs synced successfully.")
    else:
        print_in_color(
            "yellow",
            "Keycloak client redirect sync failed (continuing). "
            "You may need to manually add redirect URIs in Keycloak admin console.",
        )


def seed_keycloak_users() -> None:
    """
    Seed development users into Keycloak.

    Uses KAMIWAZA_ADMIN_PASSWORD from env.sh if set, otherwise generates random passwords.
    Should be called after ensure_forwardauth_service_account() since that already
    waits for Keycloak availability.
    """
    use_auth = os.getenv("KAMIWAZA_USE_AUTH", "true").lower() in {"true", "1", "yes"}
    lite_mode = os.getenv("KAMIWAZA_LITE", "false").lower() == "true"

    if not use_auth or lite_mode:
        print_in_color(
            "blue",
            "Skipping Keycloak dev user seeding (auth disabled or lite mode).",
        )
        return

    print_in_color("blue", "Seeding Keycloak development users...")

    # Use KAMIWAZA_ADMIN_PASSWORD from env.sh if set
    admin_password = os.getenv("KAMIWAZA_ADMIN_PASSWORD")
    if admin_password:
        os.environ["ADMIN_PASSWORD"] = admin_password
        print_in_color("blue", "Using KAMIWAZA_ADMIN_PASSWORD from env.sh for admin user")

    runner = Path(get_kamiwaza_root()) / "scripts" / "kw_py"
    cmd = [
        str(runner),
        "scripts/seed_keycloak_users.py",
    ]

    # Pass --insecure-tls if TLS verification is disabled (self-signed certs)
    if not _keycloak_verify_enabled():
        cmd.append("--insecure-tls")

    try:
        subprocess.run(cmd, check=True)
        print_in_color("green", "Keycloak dev users seeded successfully.")
    except subprocess.CalledProcessError as exc:
        print_in_color(
            "yellow", f"Keycloak dev user seeding failed (continuing): {exc}"
        )
        return


def parse_requirement(req: str) -> Tuple[str, Optional[str], Optional[str]]:
    """Parse a requirement string into name, min version, and max version."""
    # Strip platform markers first
    req_base = req.split(";")[0]

    parts = req_base.split(",")
    name = parts[0].split(">=")[0].split("==")[0].strip()
    min_version = max_version = None
    for part in parts:
        if ">=" in part:
            min_version = part.split(">=")[1].strip()
        elif "<" in part:
            max_version = part.split("<")[1].strip()
    return name, min_version, max_version


def get_installed_version(package_name: str) -> Optional[str]:
    """Get the installed version of a package."""
    try:
        # Strip out extras (e.g., 'ray[default,serve]' -> 'ray')
        package_name = re.split(r"\[|\]", package_name)[0]
        return importlib.metadata.version(package_name)
    except importlib.metadata.PackageNotFoundError:
        return None
    except Exception as e:
        print_in_color("red", f"Error parsing requirement for {package_name}: {e}")
        return None


def check_requirements(requirements_file: str = "requirements.txt") -> None:
    """Check if all packages in requirements.txt are installed and meet version constraints."""
    print_in_color("blue", "Checking installed packages against requirements.txt...")

    with open(requirements_file, "r", encoding="utf-8") as f:
        requirements = f.readlines()

    for req in requirements:
        req = req.strip()
        if not req or req.startswith("#") or "sys_platform" in req:
            continue

        name, min_version, max_version = parse_requirement(req)

        version = get_installed_version(name)

        if version:
            if min_version and max_version:
                if (
                    parse_version(min_version)
                    <= parse_version(version)
                    < parse_version(max_version)
                ):
                    print_in_color(
                        "green",
                        f"{name} {version} installed (required: >={min_version},<{max_version})",
                    )
                else:
                    print_in_color(
                        "red",
                        f"{name} {version} installed, but version >={min_version},<{max_version} is required",
                    )
            elif min_version:
                if parse_version(version) >= parse_version(min_version):
                    print_in_color(
                        "green",
                        f"{name} {version} installed (required: >={min_version})",
                    )
                else:
                    print_in_color(
                        "red",
                        f"{name} {version} installed, but version >={min_version} is required",
                    )
            else:
                print_in_color("green", f"{name} {version} installed")
        else:
            print_in_color("red", f"{name} is not installed")


# Ensure the script is invoked from install.sh or setup.sh
if not os.getenv("KAMIWAZA_RUN_FROM_INSTALL"):
    print_in_color("red", "Warning: This script should not be run directly.")
    print("Please run install.sh or setup.sh instead.")
    print(
        "If you know what you are doing and have been instructed to run this script directly, set the environment variable 'KAMIWAZA_RUN_FROM_INSTALL=yes' and try again."
    )
    sys.exit(1)

# Check if we are in a virtual environment
def in_virtualenv() -> bool:
    """Detect whether Python is executing inside an activated virtual environment."""
    return hasattr(sys, 'real_prefix') or (hasattr(sys, 'base_prefix') and sys.base_prefix != sys.prefix)


if in_virtualenv():
    print_in_color('green', f"Virtual environment detected: {sys.prefix}")
else:
    print_in_color("red", "Warning: Not running in a virtual environment - careful")
    if "-y" not in sys.argv:
        user_input = input(
            "Do you want to continue without a virtual environment? Type 'yes' to proceed: "
        )
        if user_input.lower() != "yes":
            print("Exiting the installation.")
            sys.exit(1)

# Define the paths to check for requirements.txt
paths_to_check = ["../requirements.txt", "./requirements.txt"]

# Check requirements
for path in paths_to_check:
    if os.path.isfile(path):
        check_requirements(path)
        break

# Rest of the script remains unchanged
###### 2. Test for config fiels
print("**** Testing for expected config files...")

# Define the paths to check for config files, removing the leading './' for proper concatenation later
config_paths_to_check = [
    "kamiwaza/cluster/config.py",
    "kamiwaza/serving/config.py",
    "kamiwaza/node/config.py",
    "kamiwaza/services/catalog/config.py",
    "kamiwaza/services/vectordb/config.py",
    "kamiwaza/services/models/config.py",
    "kamiwaza/services/prompts/config.py",
]


def get_site_packages_path(venv_path: Optional[str] = None) -> Optional[str]:
    """
    Retrieves the site-packages path from the virtual environment if available.

    Args:
        venv_path: The path to the virtual environment.

    Returns:
        The site-packages path if the virtual environment is detected, otherwise None.
    """
    if venv_path:
        return os.path.join(
            venv_path,
            "lib",
            f"python{sys.version_info.major}.{sys.version_info.minor}",
            "site-packages",
        )
    return None


def check_config_path(
    config_path: str, site_packages_path: Optional[str] = None
) -> None:
    """
    Checks if the config file exists at the given path or within the site-packages directory.

    Args:
        config_path: The path to the config file to check.
        site_packages_path: The site-packages directory path.
    """
    # Check directly in the provided path
    if os.path.isfile(config_path):
        print_in_color("green", f"{config_path}: Yes")
    # Construct the path within the site-packages and check
    elif site_packages_path:
        # Construct the path relative to the site-packages directory
        venv_config_path = os.path.join(site_packages_path, config_path)
        if os.path.isfile(venv_config_path):
            print_in_color("green", f"{config_path} (in venv): Yes")
        else:
            print_in_color("red", f"{config_path}: No")
    else:
        print_in_color("red", f"{config_path}: No")


# Retrieve the virtual environment path from the environment variable
venv_path = os.getenv("VIRTUAL_ENV")
site_packages_path = get_site_packages_path(venv_path)

# Iterate over the config paths
for config_path in config_paths_to_check:
    check_config_path(config_path, site_packages_path)


##### Local JWT keypair for Traefik plugin (transition period)
# Some deployments still require a local RSA public key (JWT_PUBLIC_KEY)
# for Traefik's jwt plugin. Ensure a keypair exists under runtime/.
try:
    from util.jwt_keys import generate_jwt_keys

    kamiwaza_root = os.environ.get("KAMIWAZA_ROOT") or os.getcwd()
    runtime_dir = os.path.join(kamiwaza_root, "runtime")
    print("*** Generating JWT keypair (if missing)...")
    generate_jwt_keys(runtime_dir)
except Exception as e:
    print_in_color("yellow", f"Skipping local JWT key generation: {e}")

##### 3. Install the containers
print("*** Composing docker containers... ")

# launch the containers, because we need them up for the install
print("*** First attempt to start containers...")
try:
    result = subprocess.run(
        ["./containers-up.sh"], stdout=subprocess.PIPE, stderr=subprocess.STDOUT
    )
    output = result.stdout.decode("utf-8")
    print(output)

    if result.returncode != 0:
        print_in_color(
            "red", f"containers-up.sh failed with return code {result.returncode}"
        )
        print_in_color("yellow", "Will retry after waiting for Docker to stabilize...")
        # Don't exit here, as containers might need a retry due to timing issues
    else:
        print_in_color(
            "green", "First container startup attempt completed successfully"
        )

except Exception as e:
    print_in_color("red", f"Failed to execute containers-up.sh: {e}")
    print_in_color("red", "This indicates a serious system issue. Please check:")
    print("  - Docker is running and accessible")
    print("  - containers-up.sh script exists and is executable")
    print("  - Current working directory is correct")
    exit(1)

print("*** Waiting for containers to start...")
time.sleep(15)

print("*** Ensuring containers are up (second attempt)...")
try:
    result = subprocess.run(
        ["./containers-up.sh"], stdout=subprocess.PIPE, stderr=subprocess.STDOUT
    )
    output = result.stdout.decode("utf-8")
    print(output)

    if result.returncode != 0:
        print_in_color(
            "red",
            f"containers-up.sh failed on second attempt with return code {result.returncode}",
        )
        print_in_color("red", "Container startup has failed. This is a critical error.")
        print_in_color("yellow", "Debug information:")
        print(f"  - Script output: {output}")
        print("  - Check that all required images are available")
        print("  - Verify Docker has sufficient resources")
        print("  - Check network connectivity for image pulls")
        print("Contact support@kamiwaza.ai with this error information")
        exit(1)
    else:
        print_in_color("green", "Container startup completed successfully")

except Exception as e:
    print_in_color("red", f"Failed to execute containers-up.sh on second attempt: {e}")
    print("Second pass container failure: FATAL. Contact support@kamiwaza.ai")
    exit(1)

# Check container status after startup attempts
check_container_status()
ensure_forwardauth_service_account()
ensure_keycloak_client_redirects()
seed_keycloak_users()


# import here because we need the containers up to avoid a cockroach error

from util.admin_db_reset import reset_all_databases  # noqa: E402

##### 5. Create databases
print("*** Initializing Database...")
# Initialize all databases in non-destructive mode
reset_all_databases(reset_db=False, skip_confirmation=True)
