#!/usr/bin/env bash
set -euo pipefail

# // Original script hard-coded amd64; that breaks on aarch64 (DGX). This version detects arch.

ETCD_VER="${ETCD_VER:-v3.5.15}"
GITHUB_URL="https://github.com/etcd-io/etcd/releases/download"
GOOGLE_URL="https://storage.googleapis.com/etcd"

# Use sudo only if not root
SUDO=""
if [ "${EUID:-$(id -u)}" -ne 0 ]; then SUDO="sudo"; fi

# Remove distro etcd client if present (prevents PATH confusion)
$SUDO apt-get remove -y etcd-client >/dev/null 2>&1 || true

# Map uname arch -> etcd release arch
case "$(uname -m)" in
  x86_64)   ETCD_ARCH="amd64" ;;
  aarch64|arm64) ETCD_ARCH="arm64" ;;
  *) echo "Unsupported arch: $(uname -m)"; exit 1 ;;
esac

TARBALL="etcd-${ETCD_VER}-linux-${ETCD_ARCH}.tar.gz"
TMPDIR="$(mktemp -d /tmp/etcd-download-XXXXXX)"
cleanup() { rm -rf "$TMPDIR"; }
trap cleanup EXIT

# Try GitHub first, then GCS mirror
downloaded=""
for BASE in "$GITHUB_URL" "$GOOGLE_URL"; do
  URL="${BASE}/${ETCD_VER}/${TARBALL}"
  if curl -fL --retry 3 --connect-timeout 10 "$URL" -o "${TMPDIR}/${TARBALL}"; then
    downloaded="$URL"; break
  fi
done
[ -n "$downloaded" ] || { echo "Failed to download ${TARBALL} from GitHub and GCS"; exit 1; }

tar xzf "${TMPDIR}/${TARBALL}" -C "$TMPDIR" --strip-components=1

# Quick sanity check before install
if "${TMPDIR}/etcdctl" version >/dev/null 2>&1; then
  $SUDO install -m 0755 "${TMPDIR}/etcdctl" "${TMPDIR}/etcd" /usr/local/bin/
else
  echo "### etcdctl self-test failed; not installing"; exit 1
fi

# Verify in PATH
etcdctl version
etcd --version

