#!/usr/bin/env python3
"""
Script to serialize docker-compose.yml file into a single-line string
suitable for the Kamiwaza apps.json compose_yml field.
"""

import sys
import json
import argparse
from pathlib import Path


def serialize_compose(file_path):
    """Read a docker-compose.yml file and serialize it to a single-line string."""
    try:
        with open(file_path, 'r') as f:
            content = f.read()
        
        # Convert the content to a single-line string with \n for newlines
        # This is what apps.json expects
        serialized = json.dumps(content)
        
        return serialized
    except FileNotFoundError:
        print(f"Error: File '{file_path}' not found.")
        sys.exit(1)
    except Exception as e:
        print(f"Error reading file: {e}")
        sys.exit(1)


def main():
    parser = argparse.ArgumentParser(
        description='Serialize docker-compose.yml for Kamiwaza apps.json'
    )
    parser.add_argument(
        'compose_file',
        nargs='?',
        default='docker-compose.yml',
        help='Path to docker-compose.yml file (default: docker-compose.yml)'
    )
    parser.add_argument(
        '-o', '--output',
        help='Output file (default: print to stdout)'
    )
    parser.add_argument(
        '-j', '--json',
        action='store_true',
        help='Output as JSON-escaped string (for direct copy-paste into apps.json)'
    )
    
    args = parser.parse_args()
    
    # Serialize the compose file
    serialized = serialize_compose(args.compose_file)
    
    # If JSON output is requested, properly escape for JSON
    if args.json:
        # This will add additional escaping for JSON
        output = json.dumps(serialized)
    else:
        output = serialized
    
    # Output the result
    if args.output:
        with open(args.output, 'w') as f:
            f.write(output)
        print(f"Serialized output written to: {args.output}")
    else:
        print("Serialized docker-compose.yml:")
        print("-" * 80)
        print(output)
        print("-" * 80)
        print("\nCopy the above and paste it into the 'compose_yml' field in apps.json")


if __name__ == '__main__':
    main() 