#!/bin/bash
set -e

# Source environment if exists
if [ -f "set-kamiwaza-root.sh" ]; then
  source set-kamiwaza-root.sh
elif [ -f "../set-kamiwaza-root.sh" ]; then
	source ../set-kamiwaza-root.sh
fi

KAMIWAZA_ROOT=${KAMIWAZA_ROOT:-.}
cd "${KAMIWAZA_ROOT}/frontend"

# Source env.sh to get KAMIWAZA_FRONTEND_DEVMODE
if [ -f "${KAMIWAZA_ROOT}/env.sh" ]; then
  source "${KAMIWAZA_ROOT}/env.sh"
elif [ -f "/etc/kamiwaza/env.sh" ]; then
  source "/etc/kamiwaza/env.sh"
fi

DEVMODE="${KAMIWAZA_FRONTEND_DEVMODE:-false}"
DEV_PID_FILE="${KAMIWAZA_ROOT}/frontend/.frontend-dev.pid"

normalize_bool() {
  # Use tr for bash 3.2 compatibility (macOS)
  local lower
  lower="$(echo "$1" | tr '[:upper:]' '[:lower:]')"
  case "$lower" in
    true|1|yes|y|on) echo "true" ;;
    *) echo "false" ;;
  esac
}

is_positive_int() {
  case "$1" in
    ''|*[!0-9]*|0) return 1 ;;
    *) return 0 ;;
  esac
}

DEVMODE="$(normalize_bool "${DEVMODE}")"

stop_dev_processes() {
  echo "Stopping development frontend processes..."
  FRONTEND_ROOT_PATTERN="${KAMIWAZA_ROOT}/frontend"

  # Targeted shutdown using recorded PID/PGID if available
  if [ -f "$DEV_PID_FILE" ]; then
    pid_record="$(cat "$DEV_PID_FILE" 2>/dev/null || true)"
    if [ -n "$pid_record" ]; then
      IFS=':' read -r stored_pid stored_pgid <<<"$pid_record"

      if is_positive_int "$stored_pgid"; then
        kill -TERM -- "-${stored_pgid}" 2>/dev/null || true
      fi
      if is_positive_int "$stored_pid"; then
        kill -TERM -- "$stored_pid" 2>/dev/null || true
      fi

      local deadline=$((SECONDS + 3))
      while [ "$SECONDS" -lt "$deadline" ]; do
        local pgid_alive=0
        local pid_alive=0

        if is_positive_int "$stored_pgid"; then
          if kill -0 -- "-${stored_pgid}" 2>/dev/null; then
            pgid_alive=1
          fi
        fi

        if is_positive_int "$stored_pid"; then
          if kill -0 -- "$stored_pid" 2>/dev/null; then
            pid_alive=1
          fi
        fi

        if [ "$pgid_alive" -eq 0 ] && [ "$pid_alive" -eq 0 ]; then
          break
        fi

        sleep 0.2
      done

      if is_positive_int "$stored_pgid" && kill -0 -- "-${stored_pgid}" 2>/dev/null; then
        kill -KILL -- "-${stored_pgid}" 2>/dev/null || true
      fi
      if is_positive_int "$stored_pid" && kill -0 -- "$stored_pid" 2>/dev/null; then
        kill -KILL -- "$stored_pid" 2>/dev/null || true
      fi
    fi
    rm -f "$DEV_PID_FILE"
  fi

  # Use space-separated string for bash 3.2 compatibility (no associative arrays)
  local seen_pids=""
  local script_pid="$$"
  local parent_pid="$PPID"

  add_pid() {
    local pid="$1"
    if [ -z "$pid" ]; then
      return
    fi
    if [ "$pid" = "$script_pid" ] || [ "$pid" = "$parent_pid" ]; then
      return
    fi
    # Check if already in list
    if ! echo " $seen_pids " | grep -q " $pid "; then
      seen_pids="$seen_pids $pid"
    fi
  }

  # Collect npm/node/serve commands running from frontend directory
  # Use lsof on macOS (no /proc), /proc on Linux
  for proc in npm node serve webpack; do
    for pid in $(pgrep -f "$proc" 2>/dev/null || true); do
      [ -z "$pid" ] && continue
      if [ -d "/proc" ]; then
        # Linux: check /proc/pid/cwd
        if [ -L "/proc/$pid/cwd" ]; then
          local cwd
          cwd=$(readlink -f "/proc/$pid/cwd" 2>/dev/null || true)
          if [ "$cwd" = "$FRONTEND_ROOT_PATTERN" ]; then
            add_pid "$pid"
          fi
        fi
      else
        # macOS: use lsof to find processes with cwd in frontend
        # lsof -Fn outputs: "fcwd" on one line, then "n/path" on the next line
        local cwd
        cwd=$(lsof -p "$pid" -Fn 2>/dev/null | awk '/^fcwd$/{getline; if(/^n/) print substr($0,2)}' || true)
        if [ "$cwd" = "$FRONTEND_ROOT_PATTERN" ]; then
          add_pid "$pid"
        fi
      fi
    done
  done

  # Trim leading space
  seen_pids="${seen_pids# }"

  if [ -z "$seen_pids" ]; then
    echo "No frontend development processes found"
    return 0
  fi

  for pid in $seen_pids; do
    kill "$pid" 2>/dev/null || true
  done

  sleep 1

  for pid in $seen_pids; do
    if kill -0 "$pid" 2>/dev/null; then
      kill -9 "$pid" 2>/dev/null || true
    fi
  done

  sleep 1

  local remaining=""
  for pid in $seen_pids; do
    if kill -0 "$pid" 2>/dev/null; then
      remaining="$remaining $pid"
    fi
  done

  remaining="${remaining# }"
  if [ -n "$remaining" ]; then
    echo "Warning: Some frontend development processes may still be running"
    for pid in $remaining; do
      ps -p "$pid" 2>/dev/null || true
    done
    return 1
  fi

  return 0
}

stop_container() {
  echo "Stopping frontend Docker container..."
  docker compose down 2>/dev/null || true
  docker stop kamiwaza-frontend 2>/dev/null || true
  docker rm kamiwaza-frontend 2>/dev/null || true

  if docker ps --filter "name=kamiwaza-frontend" --format "{{.Names}}" | grep -q "kamiwaza-frontend"; then
    echo "Warning: Frontend container may still be running"
    return 1
  fi

  return 0
}

stop_dev=false
stop_container_flag=false

if [ "$DEVMODE" = "true" ]; then
  stop_dev=true
else
  stop_container_flag=true
fi

if [ "$stop_dev" = true ]; then
  stop_dev_processes || WARN_DEV=true
fi

if [ "$stop_container_flag" = true ]; then
  stop_container || WARN_CONTAINER=true
else
  # Even in dev mode, gracefully stop any stray containers from previous runs
  if docker ps --filter "name=kamiwaza-frontend" --format "{{.Names}}" | grep -q "kamiwaza-frontend"; then
    stop_container || WARN_CONTAINER=true
  fi
fi

if [ "${WARN_DEV:-false}" = true ] || [ "${WARN_CONTAINER:-false}" = true ]; then
  echo "Frontend stop completed with warnings"
  exit 1
fi

echo "Frontend stopped successfully"
exit 0
