#!/bin/bash
set -euo pipefail

# Source environment
if [ -f "set-kamiwaza-root.sh" ]; then
  source set-kamiwaza-root.sh
elif [ -f "../set-kamiwaza-root.sh" ]; then
	source ../set-kamiwaza-root.sh
fi

KAMIWAZA_ROOT=${KAMIWAZA_ROOT:-.}
cd "${KAMIWAZA_ROOT}/frontend"

# Source env.sh to get configuration
if [ -f "${KAMIWAZA_ROOT}/env.sh" ]; then
  source "${KAMIWAZA_ROOT}/env.sh"
elif [ -f "/etc/kamiwaza/env.sh" ]; then
  source "/etc/kamiwaza/env.sh"
fi

DEVMODE="${KAMIWAZA_FRONTEND_DEVMODE:-false}"
KAMIWAZA_ENV="${KAMIWAZA_ENV:-default}"

# Get version for container tag
get_version() {
    if [ -f "${KAMIWAZA_ROOT}/kamiwaza.version.json" ]; then
        python3 -c "
import json
with open('${KAMIWAZA_ROOT}/kamiwaza.version.json') as f:
    v = json.load(f)
print(f\"{v['KAMIWAZA_VERSION_MAJOR']}.{v['KAMIWAZA_VERSION_MINOR']}.{v['KAMIWAZA_VERSION_PATCH']}\")
" 2>/dev/null || echo "latest"
    else
        echo "latest"
    fi
}

if [ "$DEVMODE" = "true" ]; then
  # Development mode: Run webpack-dev-server directly
  echo "Starting frontend in development mode (webpack-dev-server)..."

  # Set Node version if script exists
  if [ -f "set-node-version.sh" ]; then
    source set-node-version.sh
  fi

  # Ensure node_modules is bootstrapped before starting the dev server
  if [ ! -x "node_modules/.bin/webpack" ]; then
    echo "Installing frontend dependencies (npm install)..."
    npm install --no-progress >/tmp/kamiwaza-frontend-npm-install.log 2>&1 || {
      echo "ERROR: npm install failed. See /tmp/kamiwaza-frontend-npm-install.log" >&2
      exit 1
    }
    echo "npm install completed."
  fi

  # Run webpack dev server in the background
  npm run dev &
  DEV_PID=$!
  PID_FILE="${KAMIWAZA_ROOT}/runtime/frontend-dev.pid"
  mkdir -p "$(dirname "${PID_FILE}")"
  echo "${DEV_PID}" > "${PID_FILE}"
  # Detach the dev server from this bootstrap shell so it survives
  disown "${DEV_PID}" 2>/dev/null || true

  echo "Frontend development server started (PID: $DEV_PID)"
  echo "Access at: http://localhost:3000"

else
  # Production mode: Run Docker container via docker-compose
  echo "Starting frontend in production mode (Docker container)..."

  # Get version for container tag
  VERSION=$(get_version)
  GIT_HASH=$(git rev-parse --short HEAD 2>/dev/null || echo "")
  PLATFORM_ARCH=$(uname -m | sed 's/x86_64/amd64/; s/aarch64/arm64/')

  # Try tags in order of preference
  TAGS_TO_TRY=("${VERSION}-${GIT_HASH}" "${VERSION}" "latest" "latest-${PLATFORM_ARCH}")
  [ -z "$GIT_HASH" ] && TAGS_TO_TRY=("${VERSION}" "latest" "latest-${PLATFORM_ARCH}")

  for tag in "${TAGS_TO_TRY[@]}"; do
    if docker image inspect "kamiwazaai/frontend:${tag}" >/dev/null 2>&1; then
      IMAGE_TAG="${tag}"
      break
    fi
    echo "Image kamiwazaai/frontend:${tag} not found locally, trying next..."
  done

  if [ -z "${IMAGE_TAG:-}" ]; then
    echo "ERROR: No frontend Docker image found!"
    echo "The frontend image should be pulled by containers-up.sh"
    echo "If that didn't work, you can build it locally:"
    echo "  cd ${KAMIWAZA_ROOT}"
    echo "  ./build-scripting/build-frontend.sh --platform linux/${PLATFORM_ARCH}"
    exit 1
  fi

  echo "Using frontend image: kamiwazaai/frontend:${IMAGE_TAG}"

  # Export version for docker-compose
  export KAMIWAZA_VERSION="${IMAGE_TAG}"
  export KAMIWAZA_ENV="${KAMIWAZA_ENV}"
  export NODE_ENV="production"

  # Ensure the kamiwaza-backend network exists (with retry for startup timing)
  NETWORK_NAME="${KAMIWAZA_ENV}_kamiwaza-backend"
  MAX_NETWORK_RETRIES=12  # 12 retries * 5 seconds = 60 seconds max wait
  NETWORK_RETRY_DELAY=5

  network_ready=false
  for attempt in $(seq 1 $MAX_NETWORK_RETRIES); do
    if docker network inspect "$NETWORK_NAME" >/dev/null 2>&1; then
      network_ready=true
      if [ "$attempt" -gt 1 ]; then
        echo "Network $NETWORK_NAME is now available (after $attempt attempts)"
      fi
      break
    fi

    if [ "$attempt" -eq 1 ]; then
      echo "Waiting for network $NETWORK_NAME to be ready..."
    fi

    if [ "$attempt" -lt "$MAX_NETWORK_RETRIES" ]; then
      sleep "$NETWORK_RETRY_DELAY"
    fi
  done

  if [ "$network_ready" = false ]; then
    echo "ERROR: Network $NETWORK_NAME does not exist after ${MAX_NETWORK_RETRIES} attempts."
    echo "This network should be created by containers-up.sh"
    echo "Please ensure backend containers are running first."
    exit 1
  fi

  # Start the container using docker-compose (with retry for transient Docker issues)
  MAX_COMPOSE_RETRIES=3
  COMPOSE_RETRY_DELAY=5
  compose_success=false

  for attempt in $(seq 1 $MAX_COMPOSE_RETRIES); do
    echo "Starting frontend container (attempt $attempt/$MAX_COMPOSE_RETRIES)..."

    if docker compose up -d 2>&1; then
      # Wait for container to start
      sleep 3

      # Check if container is running
      if docker ps --filter "name=kamiwaza-frontend" --filter "status=running" --format "{{.Names}}" | grep -q "kamiwaza-frontend"; then
        compose_success=true
        break
      fi
    fi

    if [ "$attempt" -lt "$MAX_COMPOSE_RETRIES" ]; then
      echo "Container not ready, retrying in ${COMPOSE_RETRY_DELAY}s..."
      # Clean up any failed container before retry
      docker compose down 2>/dev/null || true
      sleep "$COMPOSE_RETRY_DELAY"
    fi
  done

  if [ "$compose_success" = false ]; then
    echo "ERROR: Frontend container failed to start after ${MAX_COMPOSE_RETRIES} attempts"
    echo "Container logs:"
    docker logs kamiwaza-frontend 2>&1 || true
    exit 1
  fi

  echo "Frontend container started via docker-compose"
  echo "Access at: http://localhost:3000"
  echo "Frontend container is running"
fi

exit 0
