import { isDefaultPort } from "../const";

const trimTrailingSlash = (value = "") => {
	if (typeof value !== "string") {
		return value;
	}
	return value.endsWith("/") ? value.slice(0, -1) : value;
};

const ensureLeadingSlash = (value) => {
	if (!value || typeof value !== "string") return "";
	return value.startsWith("/") ? value : `/${value}`;
};

const buildEndpoints = (base, docPathBase) => {
	if (!base) return null;
	const trimmedBase = trimTrailingSlash(base);
	const normalizedDocBase = ensureLeadingSlash(docPathBase || "")
		.replace(/\/+$/, "")
		.replace(/\/openapi\.json$/i, "");
	const openapiPath = normalizedDocBase
		? `${normalizedDocBase}/openapi.json`
		: "/openapi.json";

	return {
		baseUrl: base,
		chatUrl: `${trimmedBase}/v1/chat/completions`,
		completionsUrl: `${trimmedBase}/v1/completions`,
		modelsUrl: `${trimmedBase}/v1/models`,
		embeddingsUrl: `${trimmedBase}/v1/embeddings`,
		docsUrl: `${trimmedBase}/docs?url=${encodeURIComponent(openapiPath)}`,
		propsUrl: `${trimmedBase}/props`,
		healthUrl: `${trimmedBase}/health`,
	};
};

const resolveDeploymentId = (deployment = {}) =>
	deployment.id ||
	deployment.deployment_id ||
	deployment.deploymentId ||
	deployment.model_deployment_id ||
	"";

const normalizeRoutingMode = (value) => {
	if (value === undefined || value === null) {
		return null;
	}

	const normalized = String(value).trim().toLowerCase();
	if (!normalized) {
		return null;
	}

	if (
		["port", "ports", "legacy", "hostport", "host-port", "lb"].includes(
			normalized,
		)
	) {
		return "port";
	}

	if (["path", "paths", "prefix", "pathprefix"].includes(normalized)) {
		return "path";
	}

	if (["dual", "both", "auto", "any"].includes(normalized)) {
		return "auto";
	}

	return null;
};

const toKeyValueMap = (entries) => {
	if (!entries) {
		return {};
	}

	if (Array.isArray(entries)) {
		return entries.reduce((acc, entry) => {
			if (
				Array.isArray(entry) &&
				entry.length >= 2 &&
				typeof entry[0] === "string"
			) {
				acc[entry[0]] = entry[1];
			} else if (
				entry &&
				typeof entry === "object" &&
				typeof entry.key === "string"
			) {
				acc[entry.key] = entry.value;
			}
			return acc;
		}, {});
	}

	if (typeof entries === "object") {
		return entries;
	}

	return {};
};

const buildEnvMap = (deployment = {}) => {
	const runtimeArtifacts = deployment.runtime_artifacts || {};
	const envSources = [
		toKeyValueMap(runtimeArtifacts.system_env),
		toKeyValueMap(runtimeArtifacts.env_entries),
		toKeyValueMap(runtimeArtifacts.resolved_env),
		toKeyValueMap(deployment.env_vars),
	];

	return envSources.reduce((acc, source) => {
		Object.entries(source).forEach(([key, value]) => {
			if (typeof key === "string" && value !== undefined && value !== null) {
				acc[key] = value;
			}
		});
		return acc;
	}, {});
};

const getDefaultAppRoutingMode = () => {
	if (typeof window === "undefined") {
		return "port";
	}

	const configMode = normalizeRoutingMode(
		window.__KAMIWAZA_CONFIG__?.appRouting?.defaultMode,
	);
	if (configMode && configMode !== "auto") {
		return configMode;
	}

	try {
		const stored = window.localStorage?.getItem(
			"kamiwaza.appRoutingDefaultMode",
		);
		const storedMode = normalizeRoutingMode(stored);
		if (storedMode && storedMode !== "auto") {
			return storedMode;
		}
	} catch (_err) {
		// ignore storage access issues
	}

	return "port";
};

const getDeploymentRoutingPreference = (deployment = {}, envMap = {}) => {
	const metadata = deployment.runtime_artifacts?.compose_metadata || {};

	// First check explicit app-level preferences
	const appPreferences = [
		deployment.app_routing_mode,
		deployment.routing_mode,
		metadata.appRoutingMode,
		metadata.app_routing_mode,
		metadata.appRouting?.preferredMode,
		metadata.appRouting?.mode,
		metadata.app_routing?.preferred_mode,
		envMap.KAMIWAZA_APP_ROUTING_MODE,
		envMap.KAMIWAZA_APP_ROUTING_PREFERENCE,
	];

	for (const value of appPreferences) {
		const normalized = normalizeRoutingMode(value);
		if (normalized) {
			return normalized;
		}
	}

	// Fall back to system routing mode (already normalized by backend: "path" or "port")
	const systemRoutingMode = envMap.KAMIWAZA_ROUTING_MODE;
	if (systemRoutingMode) {
		const normalized = normalizeRoutingMode(systemRoutingMode);
		if (normalized) {
			return normalized;
		}
	}

	return null;
};

export const usingPathRouting = () => {
	if (typeof window === "undefined" || !window.location) {
		return false;
	}
	return isDefaultPort(window.location.port);
};

/**
 * Build runtime info for a model deployment.
 * @param {Object} deployment - The deployment object
 * @param {string|null} systemRoutingMode - The system routing mode ('path', 'port', or 'dual').
 *   - 'path': prefer path-based endpoints
 *   - 'port': prefer port-based endpoints
 *   - 'dual' or null: prefer port-based (dual normalizes to port for list display)
 */
export const buildModelRuntimeInfo = (
	deployment = {},
	systemRoutingMode = null,
) => {
	const protocol = window.location?.protocol || "https:";
	const hostname = window.location?.hostname || "localhost";
	const hostWithPort = window.location?.host || hostname;
	const origin = `${protocol}//${hostWithPort}`;

	const emptyResult = {
		baseUrl: "",
		chatUrl: "",
		completionsUrl: "",
		modelsUrl: "",
		embeddingsUrl: "",
		docsUrl: "",
		propsUrl: "",
		healthUrl: "",
		isPathRouting: true,
		displayEndpoint: "",
		fallbackPort: null,
		publicPath: "",
		servePath: "",
		origin,
		pathEndpoints: null,
		portEndpoints: null,
		hasLegacyPort: false,
		lbPort: null,
	};

	if (!deployment) {
		return emptyResult;
	}

	const servePathRaw = deployment.serve_path || deployment.servePath || "";
	const normalizedServePath = servePathRaw
		? ensureLeadingSlash(servePathRaw).replace(/\/+$/, "")
		: "";

	const defaultPathPrefix = "/runtime/models";
	const pathPrefixRaw =
		deployment.access_path_prefix ||
		deployment.path_prefix ||
		defaultPathPrefix;
	const normalizedPrefix =
		ensureLeadingSlash(pathPrefixRaw).replace(/\/+$/, "") || defaultPathPrefix;
	const deploymentId = resolveDeploymentId(deployment);
	const explicitAccessPath = deployment.access_path || deployment.accessPath;

	// Only generate path endpoints when path routing is applicable:
	// - Routing mode is NOT 'port' (i.e., 'path', 'dual', or null for backwards compat), OR
	// - An explicit access_path was provided by the backend
	// Normalize the routing mode to handle case variations like "Port" or "PORT"
	const normalizedSystemMode = normalizeRoutingMode(systemRoutingMode);
	const shouldGeneratePathEndpoints =
		normalizedSystemMode !== "port" || Boolean(explicitAccessPath);

	const publicPath = shouldGeneratePathEndpoints
		? explicitAccessPath
			? ensureLeadingSlash(explicitAccessPath)
			: deploymentId
				? `${normalizedPrefix}/${deploymentId}`
				: normalizedServePath || ""
		: "";

	const lbPort = Number(
		deployment.lb_port ?? deployment.lbPort ?? deployment.lbport ?? 0,
	);

	const pathBaseUrl = publicPath ? `${origin}${publicPath}` : "";
	const pathEndpoints = buildEndpoints(pathBaseUrl, publicPath);

	// Exclude port 443 (default HTTPS) - it's not a dedicated port assignment
	// TODO: Make this configurable via routing config for non-standard base ports
	const legacyPortActive = Boolean(lbPort) && lbPort !== 443;
	const portBaseUrl = legacyPortActive ? `https://${hostname}:${lbPort}` : "";
	const portEndpoints = buildEndpoints(
		portBaseUrl,
		normalizedServePath || publicPath,
	);

	// Select preferred endpoints based on system routing mode
	// - 'path': prefer path endpoints
	// - 'port', 'dual', or null: prefer port endpoints (with path fallback)
	let preferred;
	let isPathRouting;
	let displayEndpoint;

	if (systemRoutingMode === "path") {
		// Path mode: prefer path endpoints
		preferred = pathEndpoints || portEndpoints;
		isPathRouting = Boolean(pathEndpoints);
		displayEndpoint = pathEndpoints ? publicPath : portEndpoints?.baseUrl || "";
	} else {
		// Port mode (including dual which normalizes to port for display): prefer port endpoints
		preferred = portEndpoints || pathEndpoints;
		isPathRouting = !portEndpoints && Boolean(pathEndpoints);
		displayEndpoint =
			portEndpoints?.baseUrl || (pathEndpoints ? publicPath : "");
	}

	return {
		baseUrl: preferred?.baseUrl || "",
		chatUrl: preferred?.chatUrl || "",
		completionsUrl: preferred?.completionsUrl || "",
		modelsUrl: preferred?.modelsUrl || "",
		embeddingsUrl: preferred?.embeddingsUrl || "",
		docsUrl: preferred?.docsUrl || "",
		propsUrl: preferred?.propsUrl || "",
		healthUrl: preferred?.healthUrl || "",
		isPathRouting,
		displayEndpoint,
		fallbackPort: legacyPortActive ? lbPort : null,
		publicPath,
		servePath: servePathRaw || "",
		origin,
		pathEndpoints,
		portEndpoints,
		hasLegacyPort: Boolean(portEndpoints),
		lbPort: legacyPortActive ? lbPort : null,
	};
};

export const buildAppRuntimeInfo = (deployment = {}) => {
	const protocol = window.location?.protocol || "https:";
	const host = window.location?.hostname || "localhost";
	const hostWithPort = window.location?.host || host;
	const origin = `${protocol}//${hostWithPort}`;

	if (!deployment) {
		return {
			baseUrl: "",
			docsUrl: "",
			healthUrl: "",
			publicPath: "",
			displayEndpoint: "",
			origin,
			pathEndpoints: null,
			portEndpoints: null,
			hasLegacyPort: false,
			supportsPath: false,
			lbPort: null,
			primaryRouting: null,
			fallbackRouting: null,
			fallbackDisplay: "",
			fallbackUrl: "",
			pathUrl: "",
			portUrl: "",
		};
	}

	const envMap = buildEnvMap(deployment);

	const defaultPrefix = "/runtime/apps";
	const pathPrefixRaw =
		deployment.access_path_prefix || deployment.path_prefix || defaultPrefix;
	const normalizedPrefix =
		ensureLeadingSlash(pathPrefixRaw).replace(/\/+$/, "") || defaultPrefix;
	const deploymentId = resolveDeploymentId(deployment);
	const explicitAccessPath =
		deployment.access_path || deployment.accessPath || envMap.KAMIWAZA_APP_PATH;

	const publicPath = explicitAccessPath
		? ensureLeadingSlash(explicitAccessPath)
		: deploymentId
			? `${normalizedPrefix}/${deploymentId}`
			: "";

	const servePathRaw =
		deployment.serve_path || deployment.servePath || envMap.KAMIWAZA_APP_PATH;
	const normalizedServePath = servePathRaw
		? ensureLeadingSlash(servePathRaw).replace(/\/+$/, "")
		: "";

	const lbPortCandidate =
		envMap.KAMIWAZA_APP_PORT ?? deployment.lb_port ?? deployment.lbPort ?? 0;
	const lbPortNumber = Number(lbPortCandidate);
	const urlPortCandidate = (() => {
		const rawUrl = envMap.KAMIWAZA_APP_URL;
		if (!rawUrl) return null;
		try {
			const parsed = new URL(rawUrl);
			return parsed.port ? Number(parsed.port) : null;
		} catch (_err) {
			return null;
		}
	})();
	const effectivePort =
		(Number.isFinite(lbPortNumber) && lbPortNumber && lbPortNumber !== 443
			? lbPortNumber
			: null) ||
		(urlPortCandidate && urlPortCandidate !== 443 ? urlPortCandidate : null);

	const portBaseUrl = (() => {
		if (envMap.KAMIWAZA_APP_URL && effectivePort) {
			return envMap.KAMIWAZA_APP_URL;
		}
		if (effectivePort) {
			return `${protocol}//${host}:${effectivePort}`;
		}
		return "";
	})();

	const pathBaseUrl =
		envMap.KAMIWAZA_APP_PATH_URL ||
		(publicPath ? `${origin}${publicPath}` : "");

	const pathEndpoints = buildEndpoints(pathBaseUrl, publicPath);
	const portEndpoints = buildEndpoints(
		portBaseUrl,
		normalizedServePath || publicPath,
	);

	const hasPathRoute = Boolean(pathEndpoints?.baseUrl);
	const hasDedicatedPort = Boolean(portEndpoints?.baseUrl);

	const defaultMode = getDefaultAppRoutingMode();
	const preference = getDeploymentRoutingPreference(deployment, envMap);

	const candidates = [];

	if (preference && preference !== "auto") {
		candidates.push(preference);
	} else if (preference === "auto") {
		const normalizedDefault = normalizeRoutingMode(defaultMode);
		candidates.push(
			normalizedDefault && normalizedDefault !== "auto"
				? normalizedDefault
				: "port",
		);
	}

	if (defaultMode && defaultMode !== "auto") {
		candidates.push(defaultMode);
	}

	candidates.push("port", "path");

	let primaryRouting = null;
	for (const candidate of candidates) {
		if (candidate === "port" && hasDedicatedPort) {
			primaryRouting = "port";
			break;
		}
		if (candidate === "path" && hasPathRoute) {
			primaryRouting = "path";
			break;
		}
	}

	if (!primaryRouting) {
		if (hasDedicatedPort) {
			primaryRouting = "port";
		} else if (hasPathRoute) {
			primaryRouting = "path";
		}
	}

	const fallbackRouting =
		primaryRouting === "port" && hasPathRoute
			? "path"
			: primaryRouting === "path" && hasDedicatedPort
				? "port"
				: null;

	const primaryEndpoints =
		primaryRouting === "port" ? portEndpoints : pathEndpoints;
	const fallbackEndpoints =
		fallbackRouting === "port"
			? portEndpoints
			: fallbackRouting === "path"
				? pathEndpoints
				: null;

	const baseUrl = primaryEndpoints?.baseUrl || "";
	const docsUrl = primaryEndpoints?.docsUrl || "";
	const healthUrl = primaryEndpoints?.healthUrl || "";

	const displayEndpoint =
		primaryRouting === "port" ? portEndpoints?.baseUrl || "" : publicPath || "";

	const fallbackDisplay =
		fallbackRouting === "port"
			? portEndpoints?.baseUrl || ""
			: fallbackRouting === "path"
				? publicPath || ""
				: "";

	const fallbackUrl =
		fallbackRouting === "port"
			? fallbackEndpoints?.baseUrl || ""
			: fallbackRouting === "path"
				? fallbackEndpoints?.baseUrl || ""
				: "";

	return {
		baseUrl,
		docsUrl,
		healthUrl,
		publicPath,
		displayEndpoint,
		origin,
		pathEndpoints,
		portEndpoints,
		hasLegacyPort: hasDedicatedPort,
		supportsPath: hasPathRoute,
		lbPort: hasDedicatedPort ? effectivePort : null,
		primaryRouting: primaryRouting || null,
		fallbackRouting,
		fallbackDisplay,
		fallbackUrl,
		pathUrl: pathEndpoints?.baseUrl || "",
		portUrl: portEndpoints?.baseUrl || "",
	};
};
