import {
	buildAppRuntimeInfo,
	buildModelRuntimeInfo,
} from "../../utils/routing";

describe("buildModelRuntimeInfo", () => {
	beforeAll(() => {
		delete window.location;
		window.location = {
			hostname: "localhost",
			protocol: "https:",
			port: "",
			host: "localhost",
			origin: "https://localhost",
		};
	});

	const baseDeployment = {
		id: "dep-123",
		access_path_prefix: "/runtime/models",
		access_path: "/runtime/models/dep-123",
		serve_path: "/model-dep-123",
	};

	describe("routing mode selection", () => {
		it('prefers port-based endpoints when systemRoutingMode is "port"', () => {
			const deployment = {
				...baseDeployment,
				lb_port: 8080,
			};

			const info = buildModelRuntimeInfo(deployment, "port");

			expect(info.baseUrl).toBe("https://localhost:8080");
			expect(info.chatUrl).toBe("https://localhost:8080/v1/chat/completions");
			expect(info.isPathRouting).toBe(false);
			expect(info.portEndpoints).not.toBeNull();
			expect(info.pathEndpoints).not.toBeNull();
		});

		it('prefers path-based endpoints when systemRoutingMode is "path"', () => {
			const deployment = {
				...baseDeployment,
				lb_port: 8080,
			};

			const info = buildModelRuntimeInfo(deployment, "path");

			expect(info.baseUrl).toBe("https://localhost/runtime/models/dep-123");
			expect(info.chatUrl).toBe(
				"https://localhost/runtime/models/dep-123/v1/chat/completions",
			);
			expect(info.isPathRouting).toBe(true);
			expect(info.portEndpoints).not.toBeNull();
			expect(info.pathEndpoints).not.toBeNull();
		});

		it('prefers port-based endpoints when systemRoutingMode is "dual"', () => {
			const deployment = {
				...baseDeployment,
				lb_port: 8080,
			};

			const info = buildModelRuntimeInfo(deployment, "dual");

			// Dual mode prefers port for primary display
			expect(info.baseUrl).toBe("https://localhost:8080");
			expect(info.isPathRouting).toBe(false);
			// But both endpoint sets are available
			expect(info.portEndpoints).not.toBeNull();
			expect(info.portEndpoints.baseUrl).toBe("https://localhost:8080");
			expect(info.pathEndpoints).not.toBeNull();
			expect(info.pathEndpoints.baseUrl).toBe(
				"https://localhost/runtime/models/dep-123",
			);
		});

		it("defaults to port preference when systemRoutingMode is null", () => {
			const deployment = {
				...baseDeployment,
				lb_port: 8080,
			};

			const info = buildModelRuntimeInfo(deployment, null);

			expect(info.baseUrl).toBe("https://localhost:8080");
			expect(info.isPathRouting).toBe(false);
		});

		it("handles case variations in routing mode (Port, PORT)", () => {
			// Deployment without explicit access_path to verify port mode suppresses path generation
			const deployment = {
				id: "dep-case-test",
				lb_port: 8080,
			};

			// "Port" (capitalized) should be normalized to "port"
			const infoCapitalized = buildModelRuntimeInfo(deployment, "Port");
			expect(infoCapitalized.baseUrl).toBe("https://localhost:8080");
			expect(infoCapitalized.pathEndpoints).toBeNull(); // No path endpoints in port mode without access_path

			// "PORT" (uppercase) should be normalized to "port"
			const infoUppercase = buildModelRuntimeInfo(deployment, "PORT");
			expect(infoUppercase.baseUrl).toBe("https://localhost:8080");
			expect(infoUppercase.pathEndpoints).toBeNull(); // No path endpoints in port mode without access_path
		});
	});

	describe("fallback behavior", () => {
		it("falls back to path endpoints when no port is available in port mode", () => {
			const deployment = {
				...baseDeployment,
				lb_port: 0,
			};

			const info = buildModelRuntimeInfo(deployment, "port");

			expect(info.baseUrl).toBe("https://localhost/runtime/models/dep-123");
			expect(info.isPathRouting).toBe(true);
			expect(info.portEndpoints).toBeNull();
			expect(info.pathEndpoints).not.toBeNull();
		});

		it("falls back to port endpoints when no path is available in path mode", () => {
			const deployment = {
				id: "dep-123",
				lb_port: 8080,
				// No access_path, no serve_path
			};

			const info = buildModelRuntimeInfo(deployment, "path");

			// Path is computed from id and prefix
			expect(info.pathEndpoints).not.toBeNull();
			expect(info.portEndpoints).not.toBeNull();
		});
	});

	describe("endpoint generation", () => {
		it("generates all OpenAI-compatible endpoints for port routing", () => {
			const deployment = {
				...baseDeployment,
				lb_port: 8080,
			};

			const info = buildModelRuntimeInfo(deployment, "port");

			expect(info.chatUrl).toBe("https://localhost:8080/v1/chat/completions");
			expect(info.completionsUrl).toBe("https://localhost:8080/v1/completions");
			expect(info.modelsUrl).toBe("https://localhost:8080/v1/models");
			expect(info.embeddingsUrl).toBe("https://localhost:8080/v1/embeddings");
		});

		it("generates all OpenAI-compatible endpoints for path routing", () => {
			const deployment = {
				...baseDeployment,
				lb_port: 0,
			};

			const info = buildModelRuntimeInfo(deployment, "path");

			expect(info.chatUrl).toBe(
				"https://localhost/runtime/models/dep-123/v1/chat/completions",
			);
			expect(info.completionsUrl).toBe(
				"https://localhost/runtime/models/dep-123/v1/completions",
			);
			expect(info.modelsUrl).toBe(
				"https://localhost/runtime/models/dep-123/v1/models",
			);
			expect(info.embeddingsUrl).toBe(
				"https://localhost/runtime/models/dep-123/v1/embeddings",
			);
		});
	});

	describe("dual mode endpoint availability", () => {
		it("returns both port and path endpoints in dual mode for component access", () => {
			const deployment = {
				...baseDeployment,
				lb_port: 9090,
			};

			const info = buildModelRuntimeInfo(deployment, "dual");

			// Primary endpoints (port-based)
			expect(info.baseUrl).toBe("https://localhost:9090");

			// Port endpoints available
			expect(info.portEndpoints.baseUrl).toBe("https://localhost:9090");
			expect(info.portEndpoints.chatUrl).toBe(
				"https://localhost:9090/v1/chat/completions",
			);

			// Path endpoints also available
			expect(info.pathEndpoints.baseUrl).toBe(
				"https://localhost/runtime/models/dep-123",
			);
			expect(info.pathEndpoints.chatUrl).toBe(
				"https://localhost/runtime/models/dep-123/v1/chat/completions",
			);

			// lbPort is exposed for display
			expect(info.lbPort).toBe(9090);
		});
	});

	describe("edge cases", () => {
		it("returns empty result for null deployment", () => {
			const info = buildModelRuntimeInfo(null, "port");

			expect(info.baseUrl).toBe("");
			expect(info.pathEndpoints).toBeNull();
			expect(info.portEndpoints).toBeNull();
		});

		it("returns empty result for undefined deployment", () => {
			const info = buildModelRuntimeInfo(undefined, "port");

			expect(info.baseUrl).toBe("");
		});

		it("handles deployment with only id (generates path from id)", () => {
			const deployment = {
				id: "my-deployment",
				lb_port: 0,
			};

			const info = buildModelRuntimeInfo(deployment, "path");

			expect(info.publicPath).toBe("/runtime/models/my-deployment");
			expect(info.baseUrl).toBe(
				"https://localhost/runtime/models/my-deployment",
			);
		});

		it("treats port 443 as no dedicated port (falls back to path)", () => {
			// Port 443 is the default HTTPS port, not a dedicated deployment port
			const deployment = {
				...baseDeployment,
				lb_port: 443,
			};

			const info = buildModelRuntimeInfo(deployment, "port");

			// Should NOT create port endpoints for port 443
			expect(info.portEndpoints).toBeNull();
			expect(info.lbPort).toBeNull();

			// Should fall back to path endpoints (explicit access_path provided)
			expect(info.pathEndpoints).not.toBeNull();
			expect(info.baseUrl).toBe("https://localhost/runtime/models/dep-123");
		});

		it("does NOT generate path endpoints in port mode when no explicit access_path", () => {
			// In port-only mode without explicit access_path from backend,
			// path endpoints should not be generated (even from deployment ID)
			const deployment = {
				id: "port-only-deployment",
				lb_port: 9090,
				// No access_path, no access_path_prefix - backend didn't provide path info
			};

			const info = buildModelRuntimeInfo(deployment, "port");

			// Should use port endpoints
			expect(info.portEndpoints).not.toBeNull();
			expect(info.baseUrl).toBe("https://localhost:9090");

			// Should NOT generate path endpoints from deployment ID in port mode
			expect(info.pathEndpoints).toBeNull();
			expect(info.publicPath).toBe("");
		});

		it("generates path endpoints in port mode when explicit access_path is provided", () => {
			// When backend explicitly provides access_path, it should be available
			// even in port mode (for backwards compatibility or dual-mode deployments)
			const deployment = {
				id: "dep-with-path",
				lb_port: 9090,
				access_path: "/runtime/models/dep-with-path",
			};

			const info = buildModelRuntimeInfo(deployment, "port");

			// Should use port endpoints as primary
			expect(info.baseUrl).toBe("https://localhost:9090");

			// But path endpoints should also be available since backend explicitly provided them
			expect(info.pathEndpoints).not.toBeNull();
			expect(info.pathEndpoints.baseUrl).toBe(
				"https://localhost/runtime/models/dep-with-path",
			);
		});
	});
});

describe("buildAppRuntimeInfo", () => {
	const originalConfig = window.__KAMIWAZA_CONFIG__;

	afterEach(() => {
		window.__KAMIWAZA_CONFIG__ = originalConfig;
	});

	const baseDeployment = {
		id: "123e4567-e89b-12d3-a456-426614174000",
		access_path_prefix: "/runtime/apps",
		access_path: "/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
		serve_path: "/app-123e4567-e89b-12d3-a456-426614174000",
	};

	it("prefers port-based routing when a dedicated port is available", () => {
		const deployment = {
			...baseDeployment,
			lb_port: 61123,
			runtime_artifacts: {
				resolved_env: {
					KAMIWAZA_APP_URL: "https://localhost:61123",
					KAMIWAZA_APP_PATH_URL:
						"https://localhost/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PATH:
						"/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PORT: "61123",
				},
			},
		};

		const info = buildAppRuntimeInfo(deployment);

		expect(info.primaryRouting).toBe("port");
		expect(info.baseUrl).toBe("https://localhost:61123");
		expect(info.displayEndpoint).toBe("https://localhost:61123");
		expect(info.fallbackRouting).toBe("path");
		expect(info.fallbackDisplay).toBe(
			"/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
		);
	});

	it("falls back to path routing when no dedicated port is exposed", () => {
		const deployment = {
			...baseDeployment,
			lb_port: 443,
			runtime_artifacts: {
				resolved_env: {
					KAMIWAZA_APP_URL: "https://localhost:443",
					KAMIWAZA_APP_PATH_URL:
						"https://localhost/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PATH:
						"/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PORT: "443",
				},
			},
		};

		const info = buildAppRuntimeInfo(deployment);

		expect(info.primaryRouting).toBe("path");
		expect(info.baseUrl).toBe(
			"https://localhost/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
		);
		expect(info.displayEndpoint).toBe(
			"/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
		);
		expect(info.fallbackRouting).toBe(null);
	});

	it("honors explicit path routing preference when both routes are available", () => {
		const deployment = {
			...baseDeployment,
			lb_port: 61124,
			runtime_artifacts: {
				resolved_env: {
					KAMIWAZA_APP_URL: "https://localhost:61124",
					KAMIWAZA_APP_PATH_URL:
						"https://localhost/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PATH:
						"/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PORT: "61124",
					KAMIWAZA_APP_ROUTING_MODE: "path",
				},
			},
		};

		const info = buildAppRuntimeInfo(deployment);

		expect(info.primaryRouting).toBe("path");
		expect(info.baseUrl).toBe(
			"https://localhost/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
		);
		expect(info.fallbackRouting).toBe("port");
		expect(info.fallbackDisplay).toBe("https://localhost:61124");
	});

	it("falls back to default configuration preference when present", () => {
		window.__KAMIWAZA_CONFIG__ = { appRouting: { defaultMode: "path" } };

		const deployment = {
			...baseDeployment,
			lb_port: 61125,
			runtime_artifacts: {
				resolved_env: {
					KAMIWAZA_APP_URL: "https://localhost:61125",
					KAMIWAZA_APP_PATH_URL:
						"https://localhost/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PATH:
						"/runtime/apps/123e4567-e89b-12d3-a456-426614174000",
					KAMIWAZA_APP_PORT: "61125",
				},
			},
		};

		const info = buildAppRuntimeInfo(deployment);

		expect(info.primaryRouting).toBe("path");
		expect(info.fallbackRouting).toBe("port");
	});
});
