import React from 'react';
import { render } from '@testing-library/react';
import { ThemeProvider } from '@mui/material/styles';
import { BrowserRouter } from 'react-router-dom';
import { AuthContext } from './context/AuthContext';
import { ServerInfoContext } from './context/ServerInfoContext';
import { SecurityContext } from './context/SecurityContext';
import { RoutingConfigContext } from './context/RoutingConfigContext';
import theme from './Theme';

/**
 * Custom render function that includes all necessary providers
 * @param {React.ReactElement} ui - Component to render
 * @param {Object} options - Additional render options
 * @returns {Object} Render result from @testing-library/react
 */
export const renderWithProviders = (ui, options = {}) => {
  const {
    // Mock auth context value
    authValue = {
      user: null,
      loading: false,
      login: jest.fn(),
      logout: jest.fn(),
      createUser: jest.fn(),
      getWelcomeMessage: jest.fn(() => ''),
      fetchUserData: jest.fn(),
      verifyToken: jest.fn(),
      sessionToken: null
    },
    // Mock server info context value  
    serverInfoValue = {
      serverOs: 'linux',
      isLoading: false,
      error: null
    },
    // Mock security context value
    securityValue = {
      config: {
        consent_enabled: false,
        consent_content: null,
        consent_button_label: null,
        banner_enabled: false,
        banner_top_text: null,
        banner_top_color: null,
        banner_bottom_text: null,
        banner_bottom_color: null,
      },
      shouldShowBanners: false,
      shouldShowConsentGate: false,
      consentAccepted: true,
      loading: false,
      error: null,
      acceptConsent: jest.fn(),
      declineConsent: jest.fn(),
      refreshConfig: jest.fn(),
    },
    // Mock routing config context value
    routingConfigValue = {
      routingMode: 'dual',
      rawRoutingMode: 'dual',
      isLoading: false,
      error: null,
      isDualMode: true,
      normalizedMode: 'port',
    },
    // Router options
    routerProps = {},
    ...renderOptions
  } = options;

  const mergedAuthValue = {
    sessionToken: null,
    ...authValue,
  };

  // eslint-disable-next-line react/prop-types
  const AllTheProviders = ({ children }) => {
    return (
      <ThemeProvider theme={theme}>
        <BrowserRouter {...routerProps}>
          <SecurityContext.Provider value={securityValue}>
            <AuthContext.Provider value={mergedAuthValue}>
              <ServerInfoContext.Provider value={serverInfoValue}>
                <RoutingConfigContext.Provider value={routingConfigValue}>
                  {children}
                </RoutingConfigContext.Provider>
              </ServerInfoContext.Provider>
            </AuthContext.Provider>
          </SecurityContext.Provider>
        </BrowserRouter>
      </ThemeProvider>
    );
  };

  return render(ui, { wrapper: AllTheProviders, ...renderOptions });
};

/**
 * Simple render with just theme provider (for isolated component testing)
 * @param {React.ReactElement} ui - Component to render
 * @param {Object} options - Additional render options
 * @returns {Object} Render result from @testing-library/react
 */
export const renderWithTheme = (ui, options = {}) => {
  // eslint-disable-next-line react/prop-types
  const ThemeWrapper = ({ children }) => (
    <ThemeProvider theme={theme}>
      {children}
    </ThemeProvider>
  );

  return render(ui, { wrapper: ThemeWrapper, ...options });
};

/**
 * Mock fetch responses for testing
 */
export const createMockFetch = (mockResponses = {}) => {
  return jest.fn((url) => {
    const response = mockResponses[url] || { ok: true, json: () => Promise.resolve({}) };
    return Promise.resolve(response);
  });
};

/**
 * Common test data factories
 */
export const testData = {
  model: {
    id: 'test-model-1',
    name: 'Test Model',
    description: 'A test model for unit testing',
    status: 'active',
    created_at: '2023-01-01T00:00:00Z'
  },
  
  user: {
    id: 'user-1',
    username: 'testuser',
    email: 'test@example.com',
    role: 'user',
    roles: ['user'],
    is_superuser: false,
  },

  cluster: {
    id: 'cluster-1',
    name: 'Test Cluster',
    status: 'running',
    nodes: 3
  },

  vectordb: {
    id: 'vectordb-1',
    name: 'Test VectorDB',
    type: 'milvus',
    status: 'healthy'
  }
};

// Re-export everything from @testing-library/react
export * from '@testing-library/react';
export { default as userEvent } from '@testing-library/user-event';
