import axios from 'axios';
import { BASE_URL } from '../const';

/**
 * Service for interacting with the unified logging API
 */
export const logsService = {
  /**
   * List all logs, optionally filtered by deployment type
   * @param {string} deploymentType - 'all', 'vllm', 'llamacpp'
   * TODO: Future types: 'mlx', 'ampere', 'app_garden', 'tool_shed'
   */
  async listLogs(deploymentType = 'all') {
    try {
      let endpoint = '/logger/deployments/all';
      if (deploymentType && deploymentType !== 'all') {
        endpoint = `/logger/deployments/type/${deploymentType}`;
      }
      
      const response = await axios.get(`${BASE_URL}${endpoint}`);
      return response.data;
    } catch (error) {
      console.error('Error fetching logs:', error);
      throw error;
    }
  },

  /**
   * Get log content for a specific deployment
   * @param {string} deploymentType - The type of deployment
   * @param {string} deploymentId - The deployment ID
   */
  async getLogContent(deploymentType, deploymentId) {
    try {
      const response = await axios.get(`${BASE_URL}/logger/deployment/${deploymentId}/logs`);
      return response.data;
    } catch (error) {
      console.error('Error fetching log content:', error);
      throw error;
    }
  },

  /**
   * Get pattern analysis for a deployment's logs
   * @param {string} deploymentId - The deployment ID
   */
  async getLogPatterns(deploymentId) {
    try {
      const response = await axios.get(`${BASE_URL}/logger/deployment/${deploymentId}/logs/patterns`);
      return response.data;
    } catch (error) {
      console.error('Error analyzing log patterns:', error);
      throw error;
    }
  },

  /**
   * Get storage statistics for logs
   */
  async getStats() {
    try {
      const response = await axios.get(`${BASE_URL}/logger/stats`);
      return response.data;
    } catch (error) {
      console.error('Error fetching log stats:', error);
      throw error;
    }
  },

  /**
   * Delete a log file
   * @param {string} deploymentType - The type of deployment
   * @param {string} deploymentId - The deployment ID
   */
  async deleteLog(deploymentType, deploymentId) {
    try {
      const response = await axios.delete(`${BASE_URL}/logger/deployments/${deploymentType}/${deploymentId}?confirm=true`);
      return response.data;
    } catch (error) {
      console.error('Error deleting log:', error);
      throw error;
    }
  },

  /**
   * Clean up old logs
   * @param {number} daysOld - Delete logs older than this many days
   * @param {boolean} dryRun - If true, only preview what would be deleted
   */
  async cleanupOldLogs(daysOld = 30, dryRun = true) {
    try {
      const response = await axios.post(`${BASE_URL}/logger/cleanup`, {
        days_old: daysOld,
        dry_run: dryRun
      });
      return response.data;
    } catch (error) {
      console.error('Error cleaning up logs:', error);
      throw error;
    }
  },

  /**
   * Start monitoring system services (Ray, etcd, Traefik, etc.)
   */
  async startSystemMonitoring() {
    try {
      const response = await axios.post(`${BASE_URL}/logger/monitor/system`);
      return response.data;
    } catch (error) {
      console.error('Error starting system monitoring:', error);
      throw error;
    }
  },

  /**
   * Get process logs for a specific service
   * @param {string} serviceName - Name of the service
   * @param {string} serviceType - Type of service (optional)
   */
  async getProcessLogs(serviceName, serviceType = null) {
    try {
      const params = serviceType ? { service_type: serviceType } : {};
      const response = await axios.get(`${BASE_URL}/logger/process/${serviceName}`, { params });
      return response.data;
    } catch (error) {
      console.error('Error fetching process logs:', error);
      throw error;
    }
  },

  /**
   * Legacy support - get logs for a specific deployment from serving API
   * @param {string} deploymentId - The deployment ID
   */
  async getDeploymentLogsLegacy(deploymentId) {
    try {
      const response = await axios.get(`${BASE_URL}/serving/deployment/${deploymentId}/logs`);
      return response.data;
    } catch (error) {
      // Fallback to new API if legacy fails
      if (error.response?.status === 404) {
        return this.getLogContent(null, deploymentId);
      }
      throw error;
    }
  },
};