import { useState, useEffect } from 'react';
import axios from 'axios';
import { BASE_URL } from '../const';
import { useServerInfo } from '../context/ServerInfoContext';

const initialModelConfigData = {
    general: {
        name: '',
        description: '',
    },
    vLLM: {
        max_model_len: '',
        worker_use_ray: null,
        pipeline_parallel_size: '',
        tensor_parallel_size: '',
        max_parallel_loading_workers: '',
        block_size: '',
        seed: '',
        swap_space: '',
        gpu_memory_utilization: '',
        max_num_batched_tokens: '',
        max_num_seqs: '',
        cpu_offload_gb: '',
        max_paddings: '',
        quantization: '',
        enforce_eager: null,
        max_context_len_to_capture: '',
        disable_custom_all_reduce: null,
        enable_lora: null,
        max_loras: '',
        max_lora_rank: '',
        lora_extra_vocab_size: '',
        lora_dtype: '',
        max_cpu_loras: '',
        enable_auto_tool_choice: null,
        tool_call_parser: '',
        enable_reasoning: null,
        reasoning_parser: '',
    },
    llamaCPP: {
        ctx_size: '',
        threads: '',
        threads_batch: '',
        rope_scaling: '',
        rope_freq_base: '',
        rope_freq_scale: '',
        yarn_ext_factor: '',
        yarn_attn_factor: '',
        yarn_beta_slow: '',
        yarn_beta_fast: '',
        batch_size: '',
        memory_f32: null,
        mlock: null,
        n_gpu_layers: '',
        lora: '',
        lora_base: '',
        timeout: '',
        embedding: null,
        parallel: '',
        cont_batching: null,
        log_disable: null,
        slots_endpoint_disable: null,
        n_predict: '',
        override_kv: '',
        grp_attn_n: '',
        grp_attn_w: '',
        chat_template: '',
        jinja: null,
        reasoning_format: '',
    },
    mlx: {
        vision: null,
        strip_thinking: null,
    },
    custom: {},
    tooltips: {
        ctx_size: 'Size of the prompt context',
        threads: 'Number of threads to use during computation',
        threads_batch: 'Number of threads to use during batch and prompt processing',
        rope_scaling: 'RoPE frequency scaling method, defaults to linear unless specified by the model',
        rope_freq_base: 'RoPE base frequency (default: loaded from model)',
        rope_freq_scale: 'RoPE frequency scaling factor, expands context by a factor of 1/N',
        yarn_ext_factor: 'YaRN: extrapolation mix factor',
        yarn_attn_factor: 'YaRN: scale sqrt(t) or attention magnitude',
        yarn_beta_slow: 'YaRN: high correction dim or alpha',
        yarn_beta_fast: 'YaRN: low correction dim or beta',
        batch_size: 'Batch size for prompt processing',
        memory_f32: 'Use f32 instead of f16 for memory key+value (not recommended)',
        mlock: 'Force system to keep model in RAM rather than swapping or compressing',
        n_gpu_layers: 'Number of layers to store in VRAM',
        lora: 'Apply LoRA adapter (implies --no-mmap)',
        lora_base: 'Optional model to use as a base for the layers modified by the LoRA adapter',
        timeout: 'Server read/write timeout in seconds',
        embedding: 'Enable embedding vector output',
        parallel: 'Number of slots for process requests',
        cont_batching: 'Enable continuous batching (a.k.a dynamic batching)',
        log_disable: 'Disables logging to a file',
        slots_endpoint_disable: 'Disables slots monitoring endpoint',
        n_predict: 'Maximum tokens to predict',
        override_kv: 'Advanced option to override model metadata by key',
        grp_attn_n: 'Set the group attention factor to extend context size through self-extend',
        grp_attn_w: 'Set the group attention width to extend context size through self-extend',
        chat_template: 'Set custom jinja chat template',
        jinja: 'Enable jinja template processing',
        reasoning_format: 'Format for reasoning output (e.g., deepseek)',
        max_model_len: 'Model context length. If unspecified, will be automatically derived from the model. Setting to lower than model max can substantially reduce VRAM usage.',
        worker_use_ray: 'Use Ray for distributed serving, will be automatically set when using more than 1 GPU',
        pipeline_parallel_size: 'Number of pipeline stages',
        tensor_parallel_size: 'Number of tensor parallel replicas',
        max_parallel_loading_workers: 'Load model sequentially in multiple batches, to avoid RAM OOM when using tensor parallel and large models (actually not implemented by vLLM at writing)',
        block_size: 'Token block size',
        seed: 'Random seed',
        swap_space: 'CPU swap space size (GiB) per GPU (does not reduce memory requirements of vram, see cpu_offload_gb for that)',
        gpu_memory_utilization: 'The fraction of GPU memory to be used for the model executor, which can range from 0 to 1. Kamiwaza attempts to compute this but respects your setting.',
        max_num_batched_tokens: 'Maximum number of batched tokens per iteration (must be >= max_model_len)',
        max_num_seqs: 'Maximum number of sequences per iteration',
        cpu_offload_gb: 'CPU offload size (GiB) per GPU',
        max_paddings: 'Maximum number of paddings in a batch',
        quantization: 'Weight quantization method - generally autodetected',
        enforce_eager: 'Always use eager-mode PyTorch. If False, will use eager mode and CUDA graph in hybrid for maximal performance and flexibility. True reduces throughput but can reduce VRAM usage.',
        max_context_len_to_capture: 'Maximum context length covered by CUDA graphs. When a sequence has context length larger than this, we fall back to eager mode.',
        enable_lora: 'If True, enable handling of LoRA adapters.',
        max_loras: 'Max number of LoRAs in a single batch.',
        max_lora_rank: 'Max LoRA rank.',
        lora_extra_vocab_size: 'Maximum size of extra vocabulary that can be present in a LoRA adapter (added to the base model vocabulary).',
        lora_dtype: 'Data type for LoRA. If auto, will default to base model dtype.',
        max_cpu_loras: 'Maximum number of LoRAs to store in CPU memory. Must be >= than max_num_seqs. Defaults to max_num_seqs.',
        enable_auto_tool_choice: 'Enable automatic tool choice for function calling',
        tool_call_parser: 'Parser to use for tool calls (e.g., hermes)',
        enable_reasoning: 'Enable reasoning mode for models that support it',
        reasoning_parser: 'Parser to use for reasoning output (e.g., deepseek_r1)',
        vision: 'Enable vision support (if not set, will try auto-detect)',
        strip_thinking: 'Strip thinking from the output (Qwen format)',
    },
};

export const useModelConfigForm = ({ showModal, m_id, configId = null, isEditMode = false }) => {
    const { serverOs, isLoading: isLoadingServerOs, error: serverOsError } = useServerInfo();
    
    const [modelConfigData, setModelConfigData] = useState(initialModelConfigData);
    const [showVLLMFields, setShowVLLMFields] = useState(false);
    const [showLlamaCPPFields, setShowLlamaCPPFields] = useState(false);
    const [showMLXFields, setShowMLXFields] = useState(false);
    const [newConfigKey, setNewConfigKey] = useState('');
    const [newConfigValue, setNewConfigValue] = useState('');
    const [modelFiles, setModelFiles] = useState([]);
    const [selectedModelFile, setSelectedModelFile] = useState('');
    const [isGgufModelForDropdown, setIsGgufModelForDropdown] = useState(false);
    const [hasGgufFilesForSection, setHasGgufFilesForSection] = useState(false);
    const [hasSafetensorsFilesForSection, setHasSafetensorsFilesForSection] = useState(false);
    const [originalConfigData, setOriginalConfigData] = useState(null);
    const [error, setError] = useState(null);
    const [isLoading, setIsLoading] = useState(false);

    useEffect(() => {
        if (!showModal) return;

        // Reset states when modal opens
        setHasGgufFilesForSection(false);
        setHasSafetensorsFilesForSection(false);
        setIsGgufModelForDropdown(false);
        setModelFiles([]);
        setSelectedModelFile('');
        setModelConfigData(initialModelConfigData);
        setOriginalConfigData(null);

        // Fetch config data based on mode
        if (isEditMode && configId) {
            fetchExistingConfig();
        } else {
            fetchDefaultConfig();
        }

        // Always fetch model details
        fetchModelDetailAndUpdateFileTypes();
    }, [showModal, m_id, configId, isEditMode]);

    const fetchExistingConfig = async () => {
        try {
            const response = await axios.get(`${BASE_URL}/model_configs/${configId}`);
            if (response.status === 200) {
                const configData = response.data;
                setOriginalConfigData(configData);
                
                // Parse the config data - map fields to their respective sections
                const vllmFields = Object.keys(initialModelConfigData.vLLM);
                const llamaCPPFields = Object.keys(initialModelConfigData.llamaCPP);
                const mlxFields = Object.keys(initialModelConfigData.mlx);
                
                const { general, vLLM, llamaCPP, mlx, custom } = Object.entries(configData.config).reduce((acc, [key, value]) => {
                    if (vllmFields.includes(key)) acc.vLLM[key] = value;
                    else if (llamaCPPFields.includes(key)) acc.llamaCPP[key] = value;
                    else if (mlxFields.includes(key)) acc.mlx[key] = value;
                    else acc.custom[key] = value;
                    return acc;
                }, { general: {}, vLLM: {}, llamaCPP: {}, mlx: {}, custom: {} });

                setModelConfigData(prevState => ({
                    ...prevState,
                    general: { 
                        name: configData.name || '',
                        description: configData.description || ''
                    },
                    vLLM: { ...prevState.vLLM, ...vLLM },
                    llamaCPP: { ...prevState.llamaCPP, ...llamaCPP },
                    mlx: { ...prevState.mlx, ...mlx },
                    custom: custom,
                }));

                // Set selected model file if present
                if (configData.m_file_id) {
                    setSelectedModelFile(configData.m_file_id);
                }
            }
        } catch (err) {
            console.error(`[useModelConfigForm] Error fetching config data:`, err);
            setError(`Failed to fetch configuration: ${err.response?.data?.detail || err.message || 'Unknown error'}`);
        }
    };

    const fetchDefaultConfig = async () => {
        try {
            const response = await axios.get(`${BASE_URL}/models/${m_id}/configs?default=true`);
            if (response.status === 200 && response.data.length > 0) {
                const defaultConfigData = response.data[0];
                if (Object.keys(defaultConfigData.config).length > 0) {
                    // Map fields to their respective sections based on known field names
                    const vllmFields = Object.keys(initialModelConfigData.vLLM);
                    const llamaCPPFields = Object.keys(initialModelConfigData.llamaCPP);
                    const mlxFields = Object.keys(initialModelConfigData.mlx);
                    
                    const { general, vLLM, llamaCPP, mlx } = Object.entries(defaultConfigData.config).reduce((acc, [key, value]) => {
                        if (vllmFields.includes(key)) acc.vLLM[key] = value;
                        else if (llamaCPPFields.includes(key)) acc.llamaCPP[key] = value;
                        else if (mlxFields.includes(key)) acc.mlx[key] = value;
                        else acc.general[key] = value;
                        return acc;
                    }, { general: {}, vLLM: {}, llamaCPP: {}, mlx: {} });
                    
                    setModelConfigData(prevState => ({
                        ...prevState,
                        general: { ...prevState.general, ...general },
                        vLLM: { ...prevState.vLLM, ...vLLM },
                        llamaCPP: { ...prevState.llamaCPP, ...llamaCPP },
                        mlx: { ...prevState.mlx, ...mlx },
                    }));
                } else {
                    console.log("[useModelConfigForm] Default config data found but the config object is empty.");
                }
            } else if (response.status === 200) {
                setError("WARNING: No default model configuration found. You can try to re-download a .json config file if it is a hub model to regenerate a config.");
            } else {
                console.error('[useModelConfigForm] Failed to fetch default model config');
                setError(`Failed to fetch default model configuration. Status: ${response.status}`);
            }
        } catch (err) {
            console.error('[useModelConfigForm] Error fetching default model config:', err);
            setError(`Failed to fetch default configuration: ${err.response?.data?.detail || err.message || 'Unknown error'}`);
        }
    };

    const fetchModelDetailAndUpdateFileTypes = async () => {
        try {
            const response = await axios.get(`${BASE_URL}/models/${m_id}`);
            const model = response.data;

            let ggufPresentForDropdownLocal = false;
            let ggufPresentForSectionLocal = false;
            let safetensorsPresentForSectionLocal = false;

            if (model.m_files && model.m_files.length > 0) {
                model.m_files.forEach(file => {
                    if (file && file.name) {
                        const fileNameLower = file.name.toLowerCase();
                        if (fileNameLower.endsWith('.gguf')) {
                            ggufPresentForSectionLocal = true;
                        }
                        if (fileNameLower.endsWith('.safetensors')) {
                            safetensorsPresentForSectionLocal = true;
                        }
                    } else {
                    }
                });
            } else {
            }
            
            setHasGgufFilesForSection(ggufPresentForSectionLocal);
            setHasSafetensorsFilesForSection(safetensorsPresentForSectionLocal);

            ggufPresentForDropdownLocal = (model.name && model.name.toLowerCase().includes('gguf')) ||
                                   (model.repo_modelId && model.repo_modelId.toLowerCase().includes('gguf')) ||
                                   (model.m_files && model.m_files.some(file => file && file.name && file.name.toLowerCase().endsWith('.gguf')));
            setIsGgufModelForDropdown(ggufPresentForDropdownLocal);

            if (ggufPresentForDropdownLocal) {
                const downloadedGgufFiles = model.m_files.filter(file =>
                    file && file.name && file.name.toLowerCase().endsWith('.gguf') &&
                    file.is_downloading === false &&
                    file.storage_location !== null
                );
                const groupedFiles = downloadedGgufFiles.reduce((acc, file) => {
                    const baseName = file.name.replace(/-?\d+(-of-\d+)?\.gguf$/, '');
                    if (!acc[baseName]) acc[baseName] = [];
                    acc[baseName].push(file);
                    return acc;
                }, {});
                const filteredFiles = Object.values(groupedFiles).map(group => 
                    group.sort((a, b) => (parseInt(a.name.match(/-(\d+)-of-/)?.[1] || '0')) - (parseInt(b.name.match(/-(\d+)-of-/)?.[1] || '0')))[0]
                );
                setModelFiles(filteredFiles);
                
                // For Add mode, auto-select first file if available
                if (!isEditMode && filteredFiles.length > 0) {
                    setSelectedModelFile(filteredFiles[0].id);
                }
                
            }

        } catch (err) {
            console.error('[useModelConfigForm] Error fetching model details for file type analysis:', err);
            setError(`Failed to fetch model details: ${err.response?.data?.detail || err.message || 'Unknown error'}`);
        }
    };

    const validateNumericField = (value, min = 0, max = null) => {
        const num = parseFloat(value);
        return !isNaN(num) && num >= min && (max === null || num <= max);
    };

    const handleInputChange = (event, section) => {
        const { name, value, type, checked } = event.target;
        
        // Validate numeric fields
        const numericFields = {
            // vLLM fields
            max_model_len: { min: 1, max: null },
            pipeline_parallel_size: { min: 1, max: null },
            tensor_parallel_size: { min: 1, max: null },
            max_parallel_loading_workers: { min: 1, max: null },
            block_size: { min: 8, max: null },
            seed: { min: 0, max: null },
            swap_space: { min: 0, max: null },
            gpu_memory_utilization: { min: 0, max: 1 },
            max_num_batched_tokens: { min: 1, max: null },
            max_num_seqs: { min: 1, max: null },
            cpu_offload_gb: { min: 0, max: null },
            max_paddings: { min: 0, max: null },
            max_lora_rank: { min: 1, max: null },
            lora_extra_vocab_size: { min: 0, max: null },
            max_cpu_loras: { min: 1, max: null },
            // llamaCPP fields
            n_ctx: { min: 1, max: null },
            n_batch: { min: 1, max: null },
            n_threads: { min: 1, max: null },
            n_gpu_layers: { min: 0, max: null },
            timeout: { min: 1, max: null },
            n_predict: { min: -1, max: null },
            grp_attn_n: { min: 1, max: null },
            grp_attn_w: { min: 1, max: null },
            // mlx fields
            top_p: { min: 0, max: 1 },
            temp: { min: 0, max: null },
            max_kv_size: { min: 1, max: null },
        };
        
        if (type !== 'checkbox' && value !== '' && numericFields[name]) {
            const { min, max } = numericFields[name];
            if (!validateNumericField(value, min, max)) {
                setError(`Invalid value for ${name}. Must be a number between ${min} and ${max || 'infinity'}`);
                return;
            }
        }
        
        setModelConfigData(prevState => ({
            ...prevState,
            [section]: {
                ...prevState[section],
                [name]: type === 'checkbox' ? checked : value,
            },
        }));
    };

    const handleAddCustomConfig = () => {
        if (newConfigKey && newConfigValue) {
            const allKeys = [
                ...Object.keys(modelConfigData.general),
                ...Object.keys(modelConfigData.vLLM),
                ...Object.keys(modelConfigData.llamaCPP),
                ...Object.keys(modelConfigData.custom)
            ];

            if (allKeys.includes(newConfigKey)) {
                alert('Config already exists');
            } else {
                setModelConfigData(prevState => ({
                    ...prevState,
                    custom: {
                        ...prevState.custom,
                        [newConfigKey]: newConfigValue
                    }
                }));
                setNewConfigKey('');
                setNewConfigValue('');
            }
        }
    };

    const handleRemoveCustomConfig = (key) => {
        setModelConfigData(prevState => {
            const newCustom = { ...prevState.custom };
            delete newCustom[key];
            return {
                ...prevState,
                custom: newCustom
            };
        });
    };

    const validateAndPreparePayload = () => {
        if (isGgufModelForDropdown && !selectedModelFile && ((serverOs === 'darwin' && hasGgufFilesForSection) || serverOs === 'linux')) {
            alert("This model type uses GGUF files. Please select a downloaded .gguf model file.");
            return null;
        }

        const submittedData = {
            ...modelConfigData.general,
            ...modelConfigData.vLLM,
            ...modelConfigData.llamaCPP,
            ...modelConfigData.mlx,
            ...modelConfigData.custom,
        };

        const modelConfigPayload = {
            m_id: m_id,
            default: isEditMode ? (originalConfigData?.default || false) : false,
            name: submittedData.name,
            description: submittedData.description,
            config: Object.entries(submittedData).reduce((acc, [key, value]) => {
                if (key !== 'name' && key !== 'description' && value !== '' && value !== null) {
                    let includeKey = true;
                    const useFallback = serverOs === 'darwin' && !isLoadingServerOs && !serverOsError && !hasGgufFilesForSection && !hasSafetensorsFilesForSection;
                    const showLlama = (serverOs === 'darwin' && (hasGgufFilesForSection || useFallback)) || (serverOs === 'linux');
                    const showMlxLocally = serverOs === 'darwin' && (hasSafetensorsFilesForSection || useFallback);

                    if (key.startsWith('vllm_') && serverOs !== 'linux') includeKey = false;
                    if (key.startsWith('llama_cpp_') && !showLlama) includeKey = false;
                    if (key.startsWith('mlx_') && !showMlxLocally) includeKey = false;
                    
                    if (includeKey) acc[key] = value;
                }
                return acc;
            }, {}),
        };
        
        if (selectedModelFile) modelConfigPayload.m_file_id = selectedModelFile.id || selectedModelFile;

        return modelConfigPayload;
    };

    // Compute derived states
    const useFallbackDisplayOnDarwin = serverOs === 'darwin' && !hasGgufFilesForSection && !hasSafetensorsFilesForSection;
    const canShowVLLM = serverOs === 'linux';
    const showLlamaCPPSectionForUI = (serverOs === 'darwin' && (hasGgufFilesForSection || useFallbackDisplayOnDarwin)) || (serverOs === 'linux');
    const showMLXSectionForUI = serverOs === 'darwin' && (hasSafetensorsFilesForSection || useFallbackDisplayOnDarwin);
    const showGgufDropdown = isGgufModelForDropdown && (showLlamaCPPSectionForUI || serverOs === 'linux');
    
    let isButtonDisabled = false;
    if (showGgufDropdown && modelFiles.length === 0 && selectedModelFile === '') {
        isButtonDisabled = true; 
    }
    if (serverOs === 'darwin' && !showLlamaCPPSectionForUI && !showMLXSectionForUI) {
         isButtonDisabled = true;
    }
    if (serverOs === 'linux' && !canShowVLLM && !showLlamaCPPSectionForUI) {
        isButtonDisabled = true;
    }

    const showMainFormContent = serverOs === 'linux' || 
                              (serverOs === 'darwin' && (hasGgufFilesForSection || hasSafetensorsFilesForSection || useFallbackDisplayOnDarwin)) ||
                              (isGgufModelForDropdown && modelFiles.length > 0);

    return {
        // State
        modelConfigData,
        setModelConfigData,
        showVLLMFields,
        setShowVLLMFields,
        showLlamaCPPFields,
        setShowLlamaCPPFields,
        showMLXFields,
        setShowMLXFields,
        newConfigKey,
        setNewConfigKey,
        newConfigValue,
        setNewConfigValue,
        modelFiles,
        selectedModelFile,
        setSelectedModelFile,
        isGgufModelForDropdown,
        hasGgufFilesForSection,
        hasSafetensorsFilesForSection,
        originalConfigData,
        
        // Server info
        serverOs,
        isLoadingServerOs,
        serverOsError,
        error,
        setError,
        isLoading,
        
        // Derived states
        useFallbackDisplayOnDarwin,
        canShowVLLM,
        showLlamaCPPSectionForUI,
        showMLXSectionForUI,
        showGgufDropdown,
        isButtonDisabled,
        showMainFormContent,
        
        // Functions
        handleInputChange,
        handleAddCustomConfig,
        handleRemoveCustomConfig,
        validateAndPreparePayload,
    };
};