import React, { createContext, useContext, useState } from 'react';
import PropTypes from 'prop-types';
import Joyride, { STATUS } from 'react-joyride';
import { useTheme } from '@mui/material/styles';

// Development mode check
const isDevelopment = process.env.NODE_ENV === 'development';

const TourContext = createContext();

export const useTour = () => {
  const context = useContext(TourContext);
  if (!context) {
    throw new Error('useTour must be used within a TourProvider');
  }
  return context;
};

export const TourProvider = ({ children }) => {
  const theme = useTheme();
  const [tourState, setTourState] = useState({
    run: false,
    stepIndex: 0,
    tourKey: null,
  });

  // Check if user is in novice mode
  const isNoviceMode = localStorage.getItem('advancedDeveloperMode') !== 'true';

  // Track completed tours
  const getCompletedTours = () => {
    return JSON.parse(localStorage.getItem('completedTours') || '{}');
  };

  const markTourComplete = tourKey => {
    const completed = getCompletedTours();
    completed[tourKey] = true;
    localStorage.setItem('completedTours', JSON.stringify(completed));
  };

  const startTour = (tourKey, force = false) => {
    if (!isNoviceMode && !force) return; // Only run tours in novice mode unless forced

    // Don't start a new tour if one is already running
    if (tourState.run) {
      if (isDevelopment) {
        console.log('Tour already running:', tourState.tourKey, 'Skipping:', tourKey);
      }
      return;
    }

    const completed = getCompletedTours();
    if (isDevelopment) {
      console.log('Starting tour:', tourKey, 'Completed tours:', completed, 'Force:', force);
    }

    if (!completed[tourKey] || force) {
      setTourState({
        run: true,
        stepIndex: 0,
        tourKey,
      });
    }
  };

  const resetTours = () => {
    localStorage.removeItem('completedTours');
  };

  const skipAllTours = () => {
    const allTours = ['homepage', 'models', 'appGarden', 'cluster', 'toolshed'];
    allTours.forEach(tour => markTourComplete(tour));
  };

  return (
    <TourContext.Provider
      value={{
        startTour,
        resetTours,
        skipAllTours,
        isNoviceMode,
        isRunning: tourState.run,
      }}
    >
      {children}
      {tourState.run && tourState.tourKey && (
        <TourComponent
          key={tourState.tourKey} // Use tourKey for proper remounting between tours
          tourKey={tourState.tourKey}
          theme={theme}
          onComplete={completedTourKey => {
            if (isDevelopment) {
              console.log('Completing tour:', completedTourKey);
            }
            markTourComplete(completedTourKey);
            setTourState(prev => ({ ...prev, run: false }));
          }}
        />
      )}
    </TourContext.Provider>
  );
};

TourProvider.propTypes = {
  children: PropTypes.node.isRequired,
};

// Tour Component with Material-UI Styling
const TourComponent = ({ tourKey, theme, onComplete }) => {
  const [stepIndex, setStepIndex] = React.useState(0);
  const [run] = React.useState(true);

  // Listen for tab click events during cluster tour
  React.useEffect(() => {
    if (tourKey !== 'cluster') return;

    const handleTabClick = event => {
      // Advance to next step when tabs are clicked
      if (event.detail === 'hardware') {
        setStepIndex(3); // Move to hardware list step
      }
    };

    // Also listen for hardware link clicks
    const handleHardwareLinkClick = event => {
      const target = event.target;
      const isHardwareLink = target.closest('[data-tour="hardware-list"] li a');
      if (isHardwareLink && stepIndex === 3) {
        // User clicked hardware link, advance to final explanation
        setTimeout(() => {
          setStepIndex(4); // Move to final step
        }, 100);
      }
    };

    window.addEventListener('tourTabClicked', handleTabClick);
    document.addEventListener('click', handleHardwareLinkClick, true);

    return () => {
      window.removeEventListener('tourTabClicked', handleTabClick);
      document.removeEventListener('click', handleHardwareLinkClick, true);
    };
  }, [tourKey, stepIndex]);
  const tours = {
    homepage: [
      {
        target: 'body',
        content:
          'Welcome to Kamiwaza. This GenAI orchestration engine allows you to deploy models, apps, and tools; catalog and index your data and retrieve it, and more.',
        placement: 'center',
        disableBeacon: true,
      },
      {
        target: '[data-tour="mode-toggle"]',
        content:
          "You're currently in Novice Mode. This provides a simplified interface with guided experiences. You can switch back to Advanced Mode anytime for full control.",
        placement: 'bottom',
      },
      {
        target: '[data-tour="models-card"]',
        content:
          'The Models section lets you discover, download, and deploy AI models with one-click simplicity. You can also access it from the left navigation menu.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="app-garden-card"]',
        content:
          'The App Garden contains pre-built AI applications ready to deploy with your models. You can also get community apps or create your own for yourself or your enterprise. Access it from the cards or left navigation.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="notebooks-card"]',
        content:
          "Notebooks provide a JupyterLab environment for experimentation and development. We've included some starter notebooks that show using Kamiwaza via our SDK. Access from the cards or left navigation.",
        placement: 'bottom',
      },
      {
        target: '[data-tour="api-docs-card"]',
        content:
          "Kamiwaza is easy to use for users and developers. Everything you're doing in this interface is done over an API, and they're all open to use. We have interactive Swagger documentation to use them with.",
        placement: 'bottom',
      },
      {
        target: '[data-tour="hamburger-menu"]',
        content:
          "Access the full navigation menu here. In Novice Mode, it's hidden by default to keep things simple.",
        placement: 'bottom',
      },
    ],
    models: [
      {
        target: '[data-tour="recommended-models"]',
        content:
          "This is a curated selection of current top models from the Kamiwaza team. You can search for any model in advanced mode.",
        placement: 'top',
        disableBeacon: true,
      },
      {
        target: '[data-tour="model-card"]:first-child',
        content:
          'Each model card shows key information: name, use case, size, quality rating, and hardware requirements. You can see and download any of the millions of models available on Huggingface and other sources from the Models tab.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="use-case-chip"]',
        content:
          'Models are categorized by their primary use case - "General Use", "Reasoning", "Coding", "Audio Understanding & Generation", "Agentic", "Testing", "Vision"',
        placement: 'top',
      },
      {
        target: '[data-tour="context-info"]',
        content:
          "Context is the number of tokens that a model can process at once. CLICK ON this number in novice mode to adjust the context size. If you are having a conversation, your entire conversation counts towards that context limit. Depending on the model and your hardware, context can take quite a lot of additional memory when it's large.",
        placement: 'top',
      },
      {
        target: '[data-tour="quality-rating"]',
        content: 'Quality ratings help you understand model performance across different tasks. Model performance has increased dramatically at every model size for two years, but variance is still high.',
        placement: 'top',
      },
      {
        target: '[data-tour="platform-compatibility"]',
        content:
          'We automatically filter out models that are not available for your hardware, but you can toggle Show Unsupported to remove our filters. If your VRAM is low, there are some tricks; see our docs at https://kamiwaza-ai.github.io/',
        placement: 'top',
      },
      {
        target: '[data-tour="auto-config"]',
        content:
          "If there's more than one variant that is compatible with your hardware, you can change your choice here or you can change the default here. In some cases for quantized models we select both 6-bit (for quality) and 4-bit (for less memory usage).",
        placement: 'top',
      },
      {
        target: '[data-tour="deploy-button"]',
        content:
          'Once you find a model you like, click Download and Deploy to launch it with optimal settings.',
        placement: 'top',
      },
    ],
    appGarden: [
      {
        target: '[data-tour="app-garden-title"]',
        content:
          'Welcome to the App Garden! This is where you can deploy pre-configured applications.',
        placement: 'bottom',
        disableBeacon: true,
      },
      {
        target: '[data-tour="pull-images-button"]',
        content:
          'Kamiwaza may automatically refresh app definitions, but click here to import the initial app catalog.',
        disableOverlayClose: true,
        spotlightClicks: true,
        hideCloseButton: true,
        placement: 'bottom',
      },
      {
        target: '[data-tour="app-gallery"]',
        content: 'Browse through our collection of ready-to-deploy applications.',
        placement: 'top',
      },
      {
        target: '[data-tour="app-card"]:first-child',
        content: 'Each app card shows details like description, tags, and deployment options.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="app-pull-images"]',
        content:
          "If you haven't already retrieved the image for this application and you're going to deploy it, click this to download the application first.",
        spotlightClicks: true,
        placement: 'top',
      },
      {
        target: '[data-tour="app-deploy"]',
        content: 'Click Deploy to launch an application on your cluster.',
        placement: 'left',
      },
      {
        target: '[data-tour="risk-tiers"]',
        content: 'Apps are categorized by risk tiers - Guided apps are verified and safe to run.',
        placement: 'bottom',
      },
    ],
    cluster: [
      {
        target: '[data-tour="cluster-overview"]',
        content: 'The Cluster view shows your hardware resources and deployed models.',
        placement: 'top',
        disableBeacon: true,
      },
      {
        target: '[data-tour="cluster-tabs"]',
        content:
          'Navigate between different views: Clusters for overall status, Locations for geographic distribution, Nodes for individual machines, Running Nodes for active deployments, and Hardware for resource details.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="hardware-tab"]',
        content: 'Click the Hardware tab to explore your system resources.',
        placement: 'bottom',
        spotlightClicks: true,
        disableOverlayClose: true,
        disableBeacon: true,
        hideFooter: true,
        styles: {
          options: {
            zIndex: 10000,
          },
        },
      },
      {
        target: '[data-tour="hardware-list"] li:first-child a',
        content: 'Click on this hardware entry to view detailed specifications.',
        placement: 'bottom',
        spotlightClicks: true,
        disableOverlayClose: true,
        hideFooter: true,
        styles: {
          options: {
            zIndex: 10000,
          },
        },
      },
      {
        target: '[data-tour="hardware-details-card"]',
        content:
          'Great! From hardware details, you can see processor type, GPU specifications, available memory, and system configuration. Kamiwaza automatically manages deployment of models, apps, and tools across all your resources.',
        placement: 'top',
        disableBeacon: true,
      },
    ],
    toolshed: [
      {
        target: '[data-tour="toolshed-intro"]',
        content:
          'Welcome to the ToolShed! This is where you can find specialized tools that enhance your AI models with new capabilities.',
        placement: 'top',
        disableBeacon: true,
      },
      {
        target: '[data-tour="tool-filters"]',
        content:
          'Tools are organized by function - Internet Data, Model Tools, Communication, Data Transformation, and Personal Productivity. Filter tools by category or internet access requirements.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="tool-categories"]',
        content:
          'Each tool gives your models specific abilities. Filter by category to find the tools you need.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="tool-card"]:first-child',
        content:
          'Each tool card shows what capabilities it provides, which models can use it, and any special requirements.',
        placement: 'bottom',
      },
      {
        target: '[data-tour="tool-integration"]',
        content:
          'Tools work seamlessly with your deployed models, providing them with real-world interaction capabilities.',
        placement: 'top',
      },
      {
        target: '[data-tour="tool-deploy"]',
        content:
          "Deploy tools just like apps - they'll automatically integrate with your compatible models and apps.",
        placement: 'left',
      },
    ],
  };

  // Material-UI themed styles
  const tourStyles = {
    options: {
      primaryColor: theme.palette.primary.main || '#00c07f',
      backgroundColor: theme.palette.background.paper || '#1a1a1a',
      textColor: theme.palette.text.primary || '#ffffff',
      overlayColor: 'rgba(0, 0, 0, 0.7)',
      arrowColor: theme.palette.background.paper || '#1a1a1a',
      width: 380,
      zIndex: 10000,
      spotlightPadding: 10,
    },
    tooltip: {
      backgroundColor: theme.palette.background.paper || '#1a1a1a',
      borderRadius: 8,
      border: '1px solid rgba(0, 192, 127, 0.3)',
      color: theme.palette.text.primary || '#ffffff',
      fontSize: 14,
      padding: 20,
      boxShadow: '0 4px 20px rgba(0, 192, 127, 0.2)',
    },
    tooltipContainer: {
      textAlign: 'left',
    },
    tooltipContent: {
      lineHeight: 1.6,
    },
    buttonNext: {
      backgroundColor: theme.palette.primary.main || '#00c07f',
      borderRadius: 4,
      color: theme.palette.primary.contrastText || '#000',
      fontWeight: 600,
      padding: '8px 16px',
      '&:hover': {
        backgroundColor: theme.palette.primary.dark || '#00a060',
      },
    },
    buttonBack: {
      color: theme.palette.primary.main || '#00c07f',
      marginRight: 10,
      fontWeight: 500,
    },
    buttonSkip: {
      color: theme.palette.text.secondary || '#666',
    },
    spotlight: {
      backgroundColor: 'transparent',
      border: '2px solid ' + (theme.palette.primary.main || '#00c07f'),
      borderRadius: 4,
      boxShadow: '0 0 0 9999px rgba(0, 0, 0, 0.7)',
    },
    overlay: {
      backgroundColor: 'transparent',
      mixBlendMode: 'hard-light',
      opacity: 1,
      position: 'fixed',
    },
    progress: {
      backgroundColor: 'rgba(255, 255, 255, 0.1)',
      borderRadius: 2,
      height: 4,
    },
    progressbar: {
      backgroundColor: theme.palette.primary.main || '#00c07f',
      borderRadius: 2,
      height: 4,
    },
  };

  const handleJoyrideCallback = data => {
    const { status, type, action, index, lifecycle, size } = data;

    // Handle step changes for controlled tour
    if (tourKey === 'cluster') {
      // Check if we're trying to go past the last step (finish button)
      if (action === 'next' && index === size - 1 && lifecycle === 'complete') {
        onComplete(tourKey);
        return;
      }

      if (lifecycle === 'complete') {
        if (action === 'next' && index < size - 1) {
          setStepIndex(index + 1);
        } else if (action === 'prev' && index > 0) {
          setStepIndex(index - 1);
        }
      }

      // Also handle skip
      if (status === STATUS.SKIPPED) {
        onComplete(tourKey);
        return;
      }
    }

    // Handle target not found
    if (type === 'error:target_not_found') {
      if (isDevelopment) {
        console.error('Tour target not found:', data);
      }

      // Special handling for cluster tour hardware list
      if (tourKey === 'cluster' && (index === 3 || index === 4)) {
        // Hardware list or details card not ready yet, wait and retry
        setTimeout(() => {
          const target =
            index === 3
              ? document.querySelector('[data-tour="hardware-list"] li:first-child a')
              : document.querySelector('[data-tour="hardware-details-card"]');
          if (target) {
            // Force re-render with same step
            setStepIndex(index);
          }
        }, 500);
        return;
      }

      // Special handling for App Garden pull images button
      if (tourKey === 'appGarden' && data.target === '[data-tour="pull-images-button"]') {
        // Skip to next step if pull images button not found
        setStepIndex(index + 1);
        return;
      }
      // Don't complete the tour on target not found, let it continue
      return;
    }

    // Simple check for completion for non-cluster tours
    if (tourKey !== 'cluster' && (status === STATUS.FINISHED || status === STATUS.SKIPPED)) {
      onComplete(tourKey);
    }
  };

  // Ensure we have valid steps
  const tourSteps = tours[tourKey] || [];

  if (!tourSteps.length) {
    if (isDevelopment) {
      console.warn(`No tour steps found for tour: ${tourKey}`);
    }
    onComplete();
    return null;
  }

  return (
    <Joyride
      key={tourKey} // Use tourKey for proper remounting between different tours
      steps={tourSteps}
      run={run}
      stepIndex={tourKey === 'cluster' ? stepIndex : undefined}
      continuous={true}
      showProgress={true}
      showSkipButton={true}
      styles={tourStyles}
      callback={handleJoyrideCallback}
      disableCloseOnEsc={false}
      disableOverlay={false}
      scrollOffset={20}
      locale={{
        back: 'Back',
        close: 'Close',
        last: 'Finish',
        next: 'Next',
        skip: 'Skip Tour',
      }}
      floaterProps={{
        disableAnimation: false,
        styles: {
          arrow: {
            length: 10,
            spread: 15,
          },
        },
      }}
      disableOverlayClose={false}
      hideCloseButton={false}
      scrollToFirstStep={true}
      spotlightClicks={true}
      disableScrolling={false}
      disableScrollParentFix={true}
      spotlightPadding={10}
    />
  );
};

TourComponent.propTypes = {
  tourKey: PropTypes.string.isRequired,
  theme: PropTypes.shape({
    palette: PropTypes.shape({
      primary: PropTypes.shape({
        main: PropTypes.string,
        dark: PropTypes.string,
        contrastText: PropTypes.string,
      }),
      background: PropTypes.shape({
        paper: PropTypes.string,
      }),
      text: PropTypes.shape({
        primary: PropTypes.string,
        secondary: PropTypes.string,
      }),
    }),
  }).isRequired,
  onComplete: PropTypes.func.isRequired,
};
