import React, { createContext, useContext, useState, useEffect } from 'react';
import PropTypes from 'prop-types';
import axios from 'axios';
import { BASE_URL } from '../const';

const ClusterContext = createContext();

export const useCluster = () => {
  const context = useContext(ClusterContext);
  if (!context) {
    throw new Error('useCluster must be used within a ClusterProvider');
  }
  return context;
};

export const ClusterProvider = ({ children }) => {
  const [clusterCapabilities, setClusterCapabilities] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  const fetchClusterCapabilities = async () => {
    try {
      setLoading(true);
      setError(null);
      const response = await axios.get(`${BASE_URL}/cluster/cluster_capabilities`);
      setClusterCapabilities(response.data);
      console.log('Cluster capabilities loaded:', response.data);
    } catch (error) {
      console.error('Failed to fetch cluster capabilities:', error);
      setError(error);
      // Fallback to browser-based detection if cluster capabilities fail
      setClusterCapabilities({
        system_type: 'unknown',
        available_platforms: ['Fast CPU'],
        has_llamacpp: false,
        supports_gguf: false,
        total_node_vram: 0,
        gpu_count: 0,
        error: 'Failed to fetch from server'
      });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchClusterCapabilities();
  }, []);

  const calculateMaxVram = (capabilities) => {
    if (!capabilities) return 32;
    
    const { system_type, total_node_vram, gpu_count, gpu_types = [], gpus = [] } = capabilities;
    const hasCudaUmaType = Array.isArray(gpu_types) && gpu_types.includes('cuda-uma');
    const hasUmaGpu = Array.isArray(gpus) && gpus.some(gpu => {
      const memoryType = typeof gpu?.memory_type === 'string' ? gpu.memory_type.toLowerCase() : '';
      const profileType = typeof gpu?.profile === 'string' ? gpu.profile.toLowerCase() : '';
      return memoryType === 'uma' || profileType === 'cuda-uma';
    });
    const usesSharedGpuMemory = system_type === 'apple' || hasCudaUmaType || hasUmaGpu;
    
    console.log('VRAM Calculation Debug:', {
      system_type,
      total_node_vram,
      gpu_count,
      capabilities
    });
    
    // Apple Silicon systems get special handling (even though they have GPU)
    if (usesSharedGpuMemory) {
      // macOS: min 8GB, 10%, max 24GB overhead
      const totalRam = total_node_vram || parseFloat(gpus?.[0]?.vram) || 64; // Default to 64GB if not available
      const tenPercent = totalRam * 0.1;
      const overhead = Math.max(8, Math.min(tenPercent, 24));
      const result = Math.floor(totalRam - overhead);
      console.log('Shared-memory GPU calculation:', {
        totalRam,
        tenPercent,
        overhead: `max(8, min(${tenPercent}, 24)) = ${overhead}`,
        result: `${totalRam} - ${overhead} = ${result}`
      });
      return result;
    } else if (gpu_count > 0) {
      // Non-Apple GPU systems: use 92% of VRAM (8% overhead)
      const result = Math.floor(total_node_vram * 0.92);
      console.log('GPU calculation:', total_node_vram, '* 0.92 =', result);
      return result;
    } else {
      // CPU only: system ram - 16GB
      const totalRam = total_node_vram || 32; // Default to 32GB if not available
      const result = Math.max(16, totalRam - 16);
      console.log('CPU calculation:', totalRam, '- 16 =', result);
      return result;
    }
  };

  const maxVram = clusterCapabilities ? calculateMaxVram(clusterCapabilities) : null;
  const memoryWarning = clusterCapabilities?.memory_warning || null;
  const twoNodeInfo = clusterCapabilities?.two_node?.remote_host
    ? {
        ...clusterCapabilities.two_node,
        inventory_source: clusterCapabilities?.gpu_inventory_source || 'local',
      }
    : null;

  const value = {
    clusterCapabilities,
    maxVram,
    memoryWarning,
    loading,
    error,
    refetchCapabilities: fetchClusterCapabilities,
    calculateMaxVram,
    twoNodeInfo,
  };

  return (
    <ClusterContext.Provider value={value}>
      {children}
    </ClusterContext.Provider>
  );
};

ClusterProvider.propTypes = {
  children: PropTypes.node.isRequired,
};
