import React, { createContext, useState, useEffect, useContext } from 'react';
import { BASE_URL } from '../const';

export const ServerInfoContext = createContext(undefined);

export const ServerInfoProvider = ({ children }) => {
  const [serverOs, setServerOs] = useState(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchServerInfo = async () => {
      setIsLoading(true);
      setError(null);
      try {
        const response = await fetch(`${BASE_URL}/cluster/hardware?limit=1`);
        if (!response.ok) {
          throw new Error(`Failed to fetch server info: ${response.status} ${response.statusText}`);
        }
        const data = await response.json();
        if (data && data.length > 0 && data[0].os) {
          setServerOs(data[0].os.toLowerCase());
        } else {
          throw new Error('No hardware information found or OS field missing');
        }
      } catch (e) {
        console.error("Error fetching server info:", e);
        setError(e.message || 'Unknown error fetching server info');
      } finally {
        setIsLoading(false);
      }
    };

    fetchServerInfo();
  }, []);

  return (
    <ServerInfoContext.Provider value={{ serverOs, isLoading, error }}>
      {children}
    </ServerInfoContext.Provider>
  );
};

export const useServerInfo = () => {
  const context = useContext(ServerInfoContext);
  if (context === undefined) {
    throw new Error('useServerInfo must be used within a ServerInfoProvider');
  }
  return context;
}; 