/**
 * Security Context for managing classification banners and consent gate.
 *
 * This context provides:
 * - Security configuration fetched from the backend
 * - Consent acceptance state tracked in sessionStorage
 * - Functions to accept/decline consent
 */
import React, { createContext, useContext, useState, useEffect, useCallback } from 'react';
import PropTypes from 'prop-types';
import { BASE_URL } from '../const';

// Session storage key for tracking consent acceptance
const CONSENT_STORAGE_KEY = 'kamiwaza_consent_accepted';

export const SecurityContext = createContext();

/**
 * Hook to access the security context.
 * @returns {Object} Security context value with config, consent state, and actions.
 */
export const useSecurity = () => {
  const context = useContext(SecurityContext);
  if (context === undefined) {
    throw new Error('useSecurity must be used within a SecurityProvider');
  }
  return context;
};

/**
 * SecurityProvider component that fetches security configuration
 * and manages consent state.
 */
export const SecurityProvider = ({ children }) => {
  // Security configuration from backend
  const [config, setConfig] = useState({
    consent_enabled: false,
    consent_content: null,
    consent_button_label: null,
    banner_enabled: false,
    banner_top_text: null,
    banner_top_color: null,
    banner_bottom_text: null,
    banner_bottom_color: null,
  });

  // Loading state for initial config fetch
  const [loading, setLoading] = useState(true);

  // Error state for config fetch failures
  const [error, setError] = useState(null);

  // Track if config failed to load (for fail-closed behavior)
  const [configLoadFailed, setConfigLoadFailed] = useState(false);

  // Consent acceptance state (tracked in sessionStorage)
  const [consentAccepted, setConsentAccepted] = useState(() => {
    // Initialize from sessionStorage
    if (typeof window !== 'undefined') {
      return sessionStorage.getItem(CONSENT_STORAGE_KEY) === 'true';
    }
    return false;
  });

  /**
   * Fetch security configuration from the backend.
   * This endpoint is public and does not require authentication.
   */
  const fetchConfig = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      setConfigLoadFailed(false);

      const response = await fetch(`${BASE_URL}/security/public/config`, {
        method: 'GET',
        credentials: 'include',
        headers: {
          Accept: 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`Failed to fetch security config: ${response.status}`);
      }

      const data = await response.json();
      setConfig({
        consent_enabled: data.consent_enabled ?? false,
        consent_content: data.consent_content ?? null,
        consent_button_label: data.consent_button_label ?? null,
        banner_enabled: data.banner_enabled ?? false,
        banner_top_text: data.banner_top_text ?? null,
        banner_top_color: data.banner_top_color ?? null,
        banner_bottom_text: data.banner_bottom_text ?? null,
        banner_bottom_color: data.banner_bottom_color ?? null,
      });
    } catch (err) {
      console.error('[Security] CRITICAL: Failed to load security config:', err);
      setError(err.message);
      setConfigLoadFailed(true);

      // FAIL CLOSED: Assume consent is required if we can't verify config.
      // This is critical for compliance - we cannot allow access without
      // confirming security requirements from the backend.
      setConfig({
        consent_enabled: true, // Fail closed for security
        consent_content:
          '<p><strong>System Configuration Unavailable</strong></p>' +
          '<p>Unable to load security configuration from the server. ' +
          'Please ensure the backend is running and try again.</p>' +
          '<p>If this issue persists, contact your system administrator.</p>',
        consent_button_label: 'Retry',
        banner_enabled: false,
        banner_top_text: null,
        banner_top_color: null,
        banner_bottom_text: null,
        banner_bottom_color: null,
      });
    } finally {
      setLoading(false);
    }
  }, []);

  // Fetch config on mount
  useEffect(() => {
    fetchConfig();
  }, [fetchConfig]);

  /**
   * Accept consent and record it on the backend for audit logging.
   * If config failed to load (fail-closed state), this will retry fetching config.
   */
  const acceptConsent = useCallback(async () => {
    // If config failed to load, retry fetching instead of accepting
    if (configLoadFailed) {
      await fetchConfig();
      return;
    }

    try {
      // Record acceptance on backend for audit logging
      await fetch(`${BASE_URL}/security/consent/accept`, {
        method: 'POST',
        credentials: 'include',
        headers: {
          Accept: 'application/json',
        },
      });
    } catch (err) {
      // Log but don't block - consent acceptance should still work locally
      console.warn('[Security] Failed to record consent acceptance:', err);
    }

    // Update local state
    setConsentAccepted(true);
    if (typeof window !== 'undefined') {
      sessionStorage.setItem(CONSENT_STORAGE_KEY, 'true');
    }
  }, [configLoadFailed, fetchConfig]);

  /**
   * Decline consent - user navigates away or closes the browser.
   * This clears the session storage flag.
   */
  const declineConsent = useCallback(() => {
    setConsentAccepted(false);
    if (typeof window !== 'undefined') {
      sessionStorage.removeItem(CONSENT_STORAGE_KEY);
    }
  }, []);

  /**
   * Determine if consent gate should be shown.
   * Shows when consent is enabled AND user hasn't accepted yet.
   */
  const shouldShowConsentGate = config.consent_enabled && !consentAccepted && !loading;

  /**
   * Determine if classification banners should be shown.
   * Shows when enabled and at least top text is configured.
   */
  const shouldShowBanners = config.banner_enabled && config.banner_top_text && !loading;

  const value = {
    // Configuration
    config,
    loading,
    error,
    configLoadFailed,

    // Consent state and actions
    consentAccepted,
    acceptConsent,
    declineConsent,

    // Computed flags
    shouldShowConsentGate,
    shouldShowBanners,

    // Refresh config (useful for testing)
    refreshConfig: fetchConfig,
  };

  return (
    <SecurityContext.Provider value={value}>
      {children}
    </SecurityContext.Provider>
  );
};

SecurityProvider.propTypes = {
  children: PropTypes.node.isRequired,
};
