import React, { createContext, useState, useEffect, useContext } from 'react';
import PropTypes from 'prop-types';
import { BASE_URL } from '../const';

export const RoutingConfigContext = createContext(undefined);

/**
 * Normalizes the routing mode for display purposes.
 * - 'path' stays 'path'
 * - 'port', 'dual', or anything else becomes 'port'
 */
export const normalizeRoutingModeForDisplay = (mode) => {
  if (mode === 'path') {
    return 'path';
  }
  // dual, port, or any other value normalizes to port
  return 'port';
};

export const RoutingConfigProvider = ({ children }) => {
  const [routingMode, setRoutingMode] = useState('port'); // Default to port
  const [rawRoutingMode, setRawRoutingMode] = useState(null); // The actual mode from config
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchRoutingConfig = async () => {
      setIsLoading(true);
      setError(null);
      try {
        const response = await fetch(`${BASE_URL}/config/routing`);
        if (!response.ok) {
          throw new Error(`Failed to fetch routing config: ${response.status} ${response.statusText}`);
        }
        const data = await response.json();
        if (data && data.mode) {
          setRawRoutingMode(data.mode);
          // Store raw mode but provide normalized for most uses
          setRoutingMode(data.mode);
        } else {
          // Default to port mode if not specified
          setRawRoutingMode('port');
          setRoutingMode('port');
        }
      } catch (e) {
        console.error('Error fetching routing config:', e);
        setError(e.message || 'Unknown error fetching routing config');
        // Default to port mode on error
        setRawRoutingMode('port');
        setRoutingMode('port');
      } finally {
        setIsLoading(false);
      }
    };

    fetchRoutingConfig();
  }, []);

  return (
    <RoutingConfigContext.Provider value={{
      routingMode,        // Raw mode from config (path, port, dual)
      rawRoutingMode,     // Alias for clarity
      isLoading,
      error,
      // Helper to check if we're in dual mode (shows both endpoints for model detail)
      isDualMode: routingMode === 'dual',
      // Normalized mode for display (dual -> port)
      normalizedMode: normalizeRoutingModeForDisplay(routingMode),
    }}>
      {children}
    </RoutingConfigContext.Provider>
  );
};

RoutingConfigProvider.propTypes = {
  children: PropTypes.node.isRequired,
};

export const useRoutingConfig = () => {
  const context = useContext(RoutingConfigContext);
  if (context === undefined) {
    throw new Error('useRoutingConfig must be used within a RoutingConfigProvider');
  }
  return context;
};
