// frontend/src/context/AuthContext.js
import React, { createContext, useContext, useState, useEffect, useRef, useCallback } from 'react';
import PropTypes from 'prop-types';
import { BASE_URL } from '../const';
import { setSessionAuthToken } from '../utils/sessionToken';

export const AuthContext = createContext();

/**
 * Hook to access the auth context.
 * @returns {Object} Auth context value with user, login, logout, etc.
 */
export const useAuth = () => {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
};

const REFRESH_MIN_INTERVAL_MS = 60 * 1000; // 1 minute throttle
const REFRESH_IDLE_TIMEOUT_MS = 60 * 60 * 1000; // 1 hour inactivity window
const REFRESH_CHECK_INTERVAL_MS = 30 * 1000; // Polling cadence for refresh checks
const ACTIVITY_EVENTS = ['mousemove', 'keydown', 'click', 'scroll', 'touchstart'];

export const AuthProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);
  const [sessionToken, setSessionToken] = useState(null);

  const refreshIntervalRef = useRef(null);
  const lastActivityRef = useRef(Date.now());
  const lastRefreshRef = useRef(0);
  const isRefreshingRef = useRef(false);

  const decodeToken = (token) => {
    try {
      const tokenParts = token.split('.');
      const payload = JSON.parse(atob(tokenParts[1]));

      let roles = [];
      if (Array.isArray(payload.roles)) {
        roles = payload.roles;
      } else if (payload.realm_access && Array.isArray(payload.realm_access.roles)) {
        roles = payload.realm_access.roles;
      } else if (payload.resource_access) {
        for (const client of Object.values(payload.resource_access)) {
          if (client && Array.isArray(client.roles)) {
            roles = roles.concat(client.roles);
          }
        }
      }

      const username =
        payload.preferred_username ||
        payload.username ||
        payload.name ||
        payload.email ||
        payload.sub ||
        '';

      const uniqueRoles = Array.from(new Set(roles));
      return {
        username,
        email: payload.email,
        roles: uniqueRoles,
        is_superuser: uniqueRoles.includes('admin'),
      };
    } catch (err) {
      console.error('Failed to decode token', err);
      return null;
    }
  };

  const buildUserFromHeaders = (response) => {
    const rolesHeader = response.headers.get('X-User-Roles') || '';
    const roles = rolesHeader
      .split(',')
      .map(role => role.trim())
      .filter(Boolean);

    const usernameHeader = response.headers.get('X-User-Name');
    const userId = response.headers.get('X-User-Id');
    const email = response.headers.get('X-User-Email') || '';

    const username = usernameHeader || userId || '';
    if (!username && roles.length === 0 && !email) {
      return null;
    }

    return {
      username,
      email,
      roles,
      is_superuser: roles.includes('admin'),
    };
  };

  const buildUserFromResponse = (response) => {
    const headerUser = buildUserFromHeaders(response);
    if (headerUser) {
      return headerUser;
    }

    const token = response.headers.get('X-Auth-Token');
    if (token) {
      return decodeToken(token);
    }
    return null;
  };

  const fetchUserFromServer = useCallback(async () => {
    try {
      const resp = await fetch(`${BASE_URL}/auth/validate`, { credentials: 'include' });
      if (!resp.ok) {
        setSessionToken(null);
        setSessionAuthToken(null);
        lastRefreshRef.current = 0;
        return null;
      }
      const serverToken = resp.headers.get('X-Auth-Token');
      if (serverToken) {
        setSessionToken(serverToken);
        setSessionAuthToken(serverToken);
      } else {
        setSessionToken(null);
        setSessionAuthToken(null);
      }

      const userData = buildUserFromResponse(resp);
      if (!userData) {
        setSessionToken(null);
        setSessionAuthToken(null);
        lastRefreshRef.current = 0;
        return null;
      }
      lastActivityRef.current = Date.now();
      return userData;
    } catch (error) {
      console.error('Auth validation failed', error);
      setSessionToken(null);
      setSessionAuthToken(null);
      lastRefreshRef.current = 0;
      return null;
    }
  }, []);

  const stopRefreshLoop = useCallback(() => {
    if (refreshIntervalRef.current) {
      clearInterval(refreshIntervalRef.current);
      refreshIntervalRef.current = null;
    }
  }, []);

  const refreshSession = useCallback(async () => {
    if (!user || isRefreshingRef.current) {
      return false;
    }

    isRefreshingRef.current = true;
    try {
      const response = await fetch(`${BASE_URL}/auth/refresh`, {
        method: 'POST',
        credentials: 'include',
      });

      if (!response.ok) {
        if (response.status === 401) {
          stopRefreshLoop();
          setUser(null);
          setSessionToken(null);
          setSessionAuthToken(null);
          lastRefreshRef.current = 0;
        }
        return false;
      }

      const refreshedUser = await fetchUserFromServer();
      if (refreshedUser) {
        setUser(refreshedUser);
      }
      lastRefreshRef.current = Date.now();
      return true;
    } catch (error) {
      console.warn('Session refresh failed', error);
      return false;
    } finally {
      isRefreshingRef.current = false;
    }
  }, [user, fetchUserFromServer, stopRefreshLoop]);

  const maybeRefreshSession = useCallback(async () => {
    if (!user) {
      stopRefreshLoop();
      return;
    }

    const now = Date.now();
    const idleDuration = now - lastActivityRef.current;

    if (idleDuration > REFRESH_IDLE_TIMEOUT_MS) {
      stopRefreshLoop();
      return;
    }

    if (now - lastRefreshRef.current < REFRESH_MIN_INTERVAL_MS) {
      return;
    }

    await refreshSession();
  }, [user, refreshSession, stopRefreshLoop]);

  const startRefreshLoop = useCallback(() => {
    if (!user || refreshIntervalRef.current) {
      return;
    }
    void maybeRefreshSession();
    refreshIntervalRef.current = setInterval(() => {
      void maybeRefreshSession();
    }, REFRESH_CHECK_INTERVAL_MS);
  }, [user, maybeRefreshSession]);

  useEffect(() => {
    const bootstrap = async () => {
      const serverUser = await fetchUserFromServer();
      setUser(serverUser);
      setLoading(false);
    };
    bootstrap();
  }, [fetchUserFromServer]);

  useEffect(() => {
    if (user) {
      lastActivityRef.current = Date.now();
      if (!lastRefreshRef.current) {
        lastRefreshRef.current = Date.now();
      }
      startRefreshLoop();
    } else {
      stopRefreshLoop();
      lastRefreshRef.current = 0;
    }
  }, [user, startRefreshLoop, stopRefreshLoop]);

  useEffect(() => {
    if (typeof window === 'undefined') {
      return undefined;
    }

    const handleActivity = () => {
      lastActivityRef.current = Date.now();
      if (user && !refreshIntervalRef.current) {
        startRefreshLoop();
      }
    };

    ACTIVITY_EVENTS.forEach(eventName => window.addEventListener(eventName, handleActivity));

    return () => {
      ACTIVITY_EVENTS.forEach(eventName => window.removeEventListener(eventName, handleActivity));
    };
  }, [user, startRefreshLoop]);

  useEffect(() => stopRefreshLoop, [stopRefreshLoop]);

  const verifyToken = async () => {
    const serverUser = await fetchUserFromServer();
    setUser(serverUser);
    return serverUser;
  };

  const login = async (credentials) => {
    try {
      // Drop any cached session token so the upcoming validation prefers new cookies.
      setSessionToken(null);
      setSessionAuthToken(null);

      const body = credentials instanceof URLSearchParams
        ? credentials.toString()
        : (typeof credentials === 'string' ? credentials : new URLSearchParams(credentials).toString());

      const response = await fetch(`${BASE_URL}/auth/token`, {
        method: 'POST',
        credentials: 'include',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
        body,
      });

      if (!response.ok) {
        const text = await response.text();
        throw new Error(text || 'Login failed');
      }

      const serverUser = await fetchUserFromServer();
      if (serverUser) {
        setUser(serverUser);
      }
      return serverUser;
    } catch (error) {
      console.error('Login failed', error);
      throw error;
    }
  };

  const logout = async () => {
    let frontChannelUrl = null;
    let fallbackRedirect = null;
    const defaultRedirect =
      typeof window !== 'undefined' ? `${window.location.origin}/login` : '/login';

    stopRefreshLoop();
    isRefreshingRef.current = false;
    lastRefreshRef.current = 0;

    try {
      const payload = {
        reason: 'user_logout',
        revoke_token: true,
      };
      if (defaultRedirect) {
        payload.post_logout_redirect_uri = defaultRedirect;
      }

      const response = await fetch(`${BASE_URL}/auth/logout`, {
        method: 'POST',
        credentials: 'include',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(payload),
      });

      if (response && response.ok) {
        const contentType = response.headers.get('content-type') || '';
        if (contentType.includes('application/json')) {
          try {
            const data = await response.json();
            if (data?.front_channel_logout_url) {
              frontChannelUrl = data.front_channel_logout_url;
            }
            if (data?.post_logout_redirect_uri) {
              fallbackRedirect = data.post_logout_redirect_uri;
            }
          } catch (parseError) {
            console.warn('Failed to parse logout response', parseError);
          }
        }
      }
    } catch (error) {
      console.warn('Logout request failed', error);
    } finally {
      setUser(null);
      setSessionToken(null);
      setSessionAuthToken(null);
      lastActivityRef.current = Date.now();

      // Clear consent acceptance so the consent gate shows again after logout
      if (typeof window !== 'undefined') {
        sessionStorage.removeItem('kamiwaza_consent_accepted');
      }

      if (typeof window !== 'undefined') {
        const origin = window.location.origin;
        const resolveUrl = (target) => {
          if (!target) return null;
          try {
            const abs = new URL(target, origin);
            if (['localhost', '127.0.0.1'].includes(abs.hostname)) {
              const current = new URL(origin);
              abs.protocol = current.protocol;
              abs.hostname = current.hostname;
              abs.port = current.port;
            }
            return abs.toString();
          } catch (_) {
            if (target.startsWith('/')) {
              return `${origin}${target}`;
            }
            let sanitizedPath = target;
            while (sanitizedPath.startsWith('/')) {
              sanitizedPath = sanitizedPath.slice(1);
            }
            return `${origin}/${sanitizedPath}`;
          }
        };

        const redirectTarget =
          resolveUrl(fallbackRedirect) || defaultRedirect || `${origin}/login`;

        if (frontChannelUrl) {
          const frontChannelTarget = resolveUrl(frontChannelUrl);
          if (frontChannelTarget) {
            try {
              fetch(frontChannelTarget, {
                credentials: 'include',
                mode: 'cors',
                keepalive: true,
              }).catch(() => {});
            } catch (frontChannelError) {
              console.warn('Front-channel logout trigger failed', frontChannelError);
            }
          }
        }

        if (redirectTarget) {
          window.location.href = redirectTarget;
        }
      }
    }
  };

  const createUser = async () => {
    throw new Error('User creation not implemented: /auth/users endpoint not available in integrated auth');
  };

  const getWelcomeMessage = () => {
    return user ? `Welcome, ${user.username}` : '';
  };

  const fetchUserData = async () => {
    const serverUser = await fetchUserFromServer();
    setUser(serverUser);
    return serverUser;
  };

  return (
    <AuthContext.Provider value={{
      user,
      loading,
      login,
      logout,
      createUser,
      getWelcomeMessage,
      fetchUserData,
      verifyToken,
      sessionToken,
      refreshSession
    }}>
      {children}
    </AuthContext.Provider>
  );
};

AuthProvider.propTypes = {
  children: PropTypes.node.isRequired,
};
