import React from 'react';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Tab, 
  Tabs, 
  Paper
} from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import StorageIcon from '@mui/icons-material/Storage';
import AddCircleOutlineIcon from '@mui/icons-material/AddCircleOutline';
import Spinner from '../common/Spinner';
import VectorDBList from './VectorDBList';
import AddVectorDBModal from './AddVectorDBForm';
import ErrorComponent from '../common/ErrorComponent';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const TabsContainer = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: '12px 12px 0 0',
  boxShadow: 'none',
  borderBottom: '1px solid rgba(255, 255, 255, 0.08)',
  marginBottom: theme.spacing(3),
  marginTop: theme.spacing(3),
}));

const StyledTabs = styled(Tabs)(({ theme }) => ({
  '& .MuiTabs-indicator': {
    backgroundColor: theme.palette.primary.main,
    height: 3,
  },
}));

const StyledTab = styled(Tab)(({ theme }) => ({
  textTransform: 'none',
  fontSize: '0.9rem',
  fontWeight: 500,
  padding: theme.spacing(2, 3),
  color: theme.palette.text.secondary,
  '&.Mui-selected': {
    color: theme.palette.primary.main,
    fontWeight: 600,
  },
  '&:hover': {
    color: theme.palette.text.primary,
    opacity: 1,
  },
  '& .MuiSvgIcon-root': {
    marginRight: theme.spacing(1),
    fontSize: '1.2rem',
  },
}));

const TabPanel = styled(Box)(({ theme }) => ({
  padding: theme.spacing(3, 0),
}));

function TabPanelContent(props) {
  const { children, value, index, ...other } = props;

  return (
    <TabPanel
      role="tabpanel"
      hidden={value !== index}
      id={`vectordb-tabpanel-${index}`}
      aria-labelledby={`vectordb-tab-${index}`}
      {...other}
    >
      {value === index && children}
    </TabPanel>
  );
}

const VectorDBHome = () => {
  const [loading, setLoading] = React.useState(true);
  const [error, setError] = React.useState(null);
  const [tabValue, setTabValue] = React.useState(0);
  const [showAddModal, setShowAddModal] = React.useState(false);

  // Only set loading to false when all components have mounted
  React.useEffect(() => {
    setLoading(false);
  }, []);

  const handleTabChange = (event, newValue) => {
    setTabValue(newValue);
  };

  const handleOpenAddModal = () => {
    setShowAddModal(true);
  };

  const handleCloseAddModal = () => {
    setShowAddModal(false);
  };

  return (
    <ContentContainer>
      <SectionTitle variant="h4">
        VectorDB Home
      </SectionTitle>

      {loading && <Spinner />}
      {error && <ErrorComponent message={error.message} />}

      <TabsContainer>
        <StyledTabs 
          value={tabValue} 
          onChange={handleTabChange}
          variant="scrollable"
          scrollButtons="auto"
          aria-label="vectordb management tabs"
        >
          <StyledTab icon={<StorageIcon />} label="Vector Databases" iconPosition="start" />
          <StyledTab icon={<AddCircleOutlineIcon />} label="Add New VectorDB" iconPosition="start" onClick={handleOpenAddModal} />
        </StyledTabs>
      </TabsContainer>

      <TabPanelContent value={tabValue} index={0}>
        <VectorDBList isEmbedded={true} />
      </TabPanelContent>

      {showAddModal && (
        <AddVectorDBModal 
          showModal={showAddModal}
          closeModal={handleCloseAddModal}
          addVectorDB={() => {}} // This will need to be implemented
        />
      )}
    </ContentContainer>
  );
};

export default VectorDBHome;
